/*
 * Copyright (C) 2014 Canonical Ltd.
 *
 * This file is part of unity-chromium-extension
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QByteArray>
#include <QDebug>
#include <QJsonDocument>
#include <QProcess>
#include <QSet>
#include <QSignalSpy>
#include <QTest>

class ConnectionTest: public QObject
{
    Q_OBJECT

public:
    ConnectionTest();

    void postMessage(const QByteArray &message);
    QByteArray readReply();
    QVariantMap jsonToMap(const QByteArray &json);

private Q_SLOTS:
    void init();
    void cleanup();
    void testMessaging();

private:
    QProcess m_process;
};

ConnectionTest::ConnectionTest():
    QObject(0)
{
    m_process.setProgram("./connection-host");
    m_process.setProcessChannelMode(QProcess::ForwardedErrorChannel);
    m_process.setReadChannel(QProcess::StandardOutput);
}

void ConnectionTest::postMessage(const QByteArray &message)
{
    quint32 length = message.length();

    m_process.write((char *)&length, sizeof(length));
    m_process.write(message);
}

QByteArray ConnectionTest::readReply()
{
    quint32 length;
    m_process.waitForReadyRead();
    m_process.read((char *)&length, sizeof(length));
    return m_process.read(length);
}

QVariantMap ConnectionTest::jsonToMap(const QByteArray &json)
{
    QJsonDocument doc = QJsonDocument::fromJson(json);
    return doc.toVariant().toMap();
}

void ConnectionTest::init()
{
    m_process.start();
    m_process.waitForStarted();
}

void ConnectionTest::cleanup()
{
    m_process.kill();
    m_process.waitForFinished();
}

void ConnectionTest::testMessaging()
{
    postMessage("{\"msg\":\"hi\"}");
    QVariantMap reply = jsonToMap(readReply());
    QCOMPARE(reply.count(), 2);
    QCOMPARE(reply.value("count").toInt(), 1);
    QCOMPARE(reply.value("keys").toStringList(), QStringList() << "msg");

    postMessage("{\"list\": [\"one\", \"two\"],"
                "\"name\": \"Tom\","
                "\"number\": 23}");
    reply = jsonToMap(readReply());
    QCOMPARE(reply.count(), 2);
    QCOMPARE(reply.value("count").toInt(), 3);
    QSet<QString> expectedKeys;
    expectedKeys << "list" << "name" << "number";
    QCOMPARE(reply.value("keys").toStringList().toSet(), expectedKeys);

    m_process.terminate();
}

QTEST_MAIN(ConnectionTest);

#include "tst_connection.moc"
