/*****************************************************************************
    TRAVIS - Trajectory Analyzer and Visualizer
    http://www.travis-analyzer.de/

    Copyright (c) 2009-2014 Martin Brehm
                  2012-2014 Martin Thomas

    This file written by Martin Brehm.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/

#include "voroanalysis.h"
#include "globalvar.h"
#include "maintools.h"


CVoroPoint::CVoroPoint()
{
	m_iaVoroCells.SetName("CVoroPoint::m_iaVoroCells");
}


CVoroPoint::~CVoroPoint()
{
}


/*CVoroFace::CVoroFace()
{
}


CVoroFace::~CVoroFace()
{
}*/


CDelaunayPoint::CDelaunayPoint()
{
}


CDelaunayPoint::~CDelaunayPoint()
{
}


CDelaunayTetrahedron::CDelaunayTetrahedron()
{
}


CDelaunayTetrahedron::~CDelaunayTetrahedron()
{
}


CVoroCell::CVoroCell()
{
	m_iaVoroPoints.SetName("CVoroCell::m_iaVoroPoints");
	m_iaNeighborCells.SetName("CVoroCell::m_iaNeighborCells");
}


CVoroCell::~CVoroCell()
{
}


CVoroAnalysis::CVoroAnalysis()
{
	m_oaVoroCells.SetName("CVoroAnalysis::m_oaVoroCells");
	m_oaDelaunayTetrahedra.SetName("CVoroAnalysis::m_oaDelaunayTetrahedra");
	m_oaDelaunayPoints.SetName("CVoroAnalysis::m_oaDelaunayPoints");
	m_oaVoroPoints.SetName("CVoroAnalysis::m_oaVoroPoints");
}


CVoroAnalysis::~CVoroAnalysis()
{
}


void CVoroAnalysis::Build(CTimeStep *ts, bool sanity, bool verbose)
{
	voronoicell_neighbor c;
	container_periodic_poly *con;
	CVoroCell *vc;
	int ijk, q, z, z2, z3, faces, fc, id, ti, i, cc;
	double *pp;
	vector<int> nb;
	vector<int> fv;
	CVoroPoint *vp;
	CDelaunayTetrahedron *dt;
	CDelaunayPoint *dp;
	double px, py, pz, pk[4];
	CxVector3 vec;
	double tfa[8], tf;

#define EPS 0.000001

	try { con = new container_periodic_poly(g_fBoxX/1000.0,0,g_fBoxY/1000.0,0,0,g_fBoxZ/1000.0,g_pVoroWrapper->m_iBlocksX,g_pVoroWrapper->m_iBlocksY,g_pVoroWrapper->m_iBlocksZ,g_iVoroMemory); } catch(...) { con = NULL; }
	if (con == NULL) NewException((double)sizeof(container_periodic_poly),__FILE__,__LINE__,__PRETTY_FUNCTION__);

	m_oaDelaunayPoints.SetMaxSize(g_iGesAtomCount);
	m_oaVoroCells.SetSize(g_iGesAtomCount);

	for (z=0;z<g_iGesAtomCount;z++)
	{
		try { dp = new CDelaunayPoint(); } catch(...) { dp = NULL; }
		if (dp == NULL) NewException((double)sizeof(CDelaunayPoint),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		dp->m_vPos = ts->m_vaCoords[z];
		m_oaDelaunayPoints.Add(dp);
		con->put(z,ts->m_vaCoords[z][0]/1000.0,ts->m_vaCoords[z][1]/1000.0,ts->m_vaCoords[z][2]/1000.0,0.0005);
	}

	c_loop_all_periodic vl(*con);

	if (sanity)
	{
		mprintf("    Voronoi decomposition:\n");
		mprintf(WHITE,"      [");
	}

	cc = 0;
	if (vl.start()) 
	{
		do 
		{
			if (sanity)
				if (fmod(cc,g_iGesAtomCount/60.0) < 1.0)
					mprintf(WHITE,"#");

			if (con->compute_cell(c,vl))
			{
				ijk=vl.ijk;
				q=vl.q;
				pp=con->p[ijk]+con->ps*q;

				id = con->id[ijk][q];

				try { vc = new CVoroCell(); } catch(...) { vc = NULL; }
				if (vc == NULL) NewException((double)sizeof(CVoroCell),__FILE__,__LINE__,__PRETTY_FUNCTION__);
				
				m_oaVoroCells[id] = vc;
				vc->m_vPos[0] = ts->m_vaCoords[id][0]/1000.0;
				vc->m_vPos[1] = ts->m_vaCoords[id][1]/1000.0;
				vc->m_vPos[2] = ts->m_vaCoords[id][2]/1000.0;

				c.neighbors(nb);
				faces = c.number_of_faces();
				c.face_vertices(fv);

				ti = 0;

				if (verbose)
				{
					mprintf("### Zelle %d\n",id+1);
					mprintf("  %d Nachbarn: ",nb.size());
				}

				vc->m_iaNeighborCells.SetSize(nb.size());
				for (z=0;z<(int)nb.size();z++)
				{
					if (verbose)
						mprintf("%d, ",nb[z]+1);
					vc->m_iaNeighborCells[z] = nb[z];
				}

				if (verbose)
				{
					mprintf("\n");
					mprintf("   %d Flaechen, %d Vertices.\n",faces,fv.size());
				}

				i = 0;
				for (z=0;z<faces;z++)
				{
					fc = fv[ti];

					if (verbose)
						mprintf("    - Flaeche %d: %d Punkte - ",z+1,fc);

					for (z2=0;z2<fc;z2++)
					{
						if (verbose)
							mprintf("%d, ",fv[ti+z2+1]+1);
						if (i < fv[ti+z2+1])
							i = fv[ti+z2+1];
					}
					if (verbose)
						mprintf("\n");

					ti += fv[ti]+1;
				}

				if (verbose)
				{
					mprintf("    %d Voronoi-Punkte.\n",i+1);
					mprintf("    Base Point ist %.6f | %.6f | %.6f .\n",*pp,pp[1],pp[2]);
				}

				for (z=0;z<=i;z++)
				{
					px = *pp+c.pts[z*3]*0.5;
					py = pp[1]+c.pts[z*3+1]*0.5;
					pz = pp[2]+c.pts[z*3+2]*0.5;

					while (px < 0) px += g_fBoxX/1000.0;
					while (px >= g_fBoxX/1000.0) px -= g_fBoxX/1000.0;
					while (py < 0) py += g_fBoxY/1000.0;
					while (py >= g_fBoxY/1000.0) py -= g_fBoxY/1000.0;
					while (pz < 0) pz += g_fBoxZ/1000.0;
					while (pz >= g_fBoxZ/1000.0) pz -= g_fBoxZ/1000.0;

/*					while (px < -0.5) px += 1.0;
					while (px >= 0.5) px -= 1.0;
					while (py < -0.5) py += 1.0;
					while (py >= 0.5) py -= 1.0;
					while (pz < -0.5) pz += 1.0;
					while (pz >= 0.5) pz -= 1.0;*/

					if (verbose)
						mprintf("      %2d: %.6f | %.6f | %.6f\n",z+1,px,py,pz);

					for (z2=0;z2<m_oaVoroPoints.GetSize();z2++)
					{
						vp = (CVoroPoint*)m_oaVoroPoints[z2];
						if (fabs(vp->m_vPos[0]-px) > EPS)
							continue;
						if (fabs(vp->m_vPos[1]-py) > EPS)
							continue;
						if (fabs(vp->m_vPos[2]-pz) > EPS)
							continue;
						if (verbose)
							mprintf("      Punkt bereits vorhanden: %d.\n",z2+1);
						vc->m_iaVoroPoints.Add(z2);
						goto _done;
					}
					vc->m_iaVoroPoints.Add(m_oaVoroPoints.GetSize());

					try { vp = new CVoroPoint(); } catch(...) { vp = NULL; }
					if (vp == NULL) NewException((double)sizeof(CVoroPoint),__FILE__,__LINE__,__PRETTY_FUNCTION__);
					
					vp->m_vPos[0] = px;
					vp->m_vPos[1] = py;
					vp->m_vPos[2] = pz;
					m_oaVoroPoints.Add(vp);
					if (verbose)
						mprintf(" *    Fuege Punkt als %d hinzu.\n",m_oaVoroPoints.GetSize());
_done:;
				}
			}
			cc++;
		} while (vl.inc());

		if (sanity)
			m_iMaxVoroPoints = m_oaVoroPoints.GetSize() * 2;

		if (verbose)
			mprintf("Insgesamt %d verschiedene Voronoi-Punkte.\n\n",m_oaVoroPoints.GetSize());

		if (sanity)
		{
			mprintf(WHITE,"]\n");
			mprintf("\n    Found %d different voronoi points.\n\n",m_oaVoroPoints.GetSize());
		}
	}

	if (sanity)
	{
		mprintf("    Delaunay tetrahedron generation:\n");
		mprintf(WHITE,"      [");
	}

	for (z=0;z<m_oaVoroPoints.GetSize();z++)
	{
		if (sanity)
			if (fmod(z,m_oaVoroPoints.GetSize()/60.0) < 1.0)
				mprintf(WHITE,"#");

		vp = (CVoroPoint*)m_oaVoroPoints[z];

		if (verbose)
			mprintf("### Voronoi-Punkt %d: %.6f | %.6f | %.6f\n",z+1,vp->m_vPos[0],vp->m_vPos[1],vp->m_vPos[2]);

		try { dt = new CDelaunayTetrahedron(); } catch(...) { dt = NULL; }
		if (dt == NULL) NewException((double)sizeof(CDelaunayTetrahedron),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		ti = 0;

		for (z2=0;z2<m_oaVoroCells.GetSize();z2++)
		{
			vc = (CVoroCell*)m_oaVoroCells[z2];
			for (z3=0;z3<vc->m_iaVoroPoints.GetSize();z3++)
			{
				if (vc->m_iaVoroPoints[z3] == z)
				{
					if (verbose)
					{
						vec = vp->m_vPos - vc->m_vPos;

						while (vec[0] < -g_fBoxX/2000.0) vec[0] += g_fBoxX/1000.0;
						while (vec[0] >= g_fBoxX/2000.0) vec[0] -= g_fBoxX/1000.0;
						while (vec[1] < -g_fBoxY/2000.0) vec[1] += g_fBoxY/1000.0;
						while (vec[1] >= g_fBoxY/2000.0) vec[1] -= g_fBoxY/1000.0;
						while (vec[2] < -g_fBoxZ/2000.0) vec[2] += g_fBoxZ/1000.0;
						while (vec[2] >= g_fBoxZ/2000.0) vec[2] -= g_fBoxZ/1000.0;

/*						while (vec[0] < -0.5) vec[0] += 1.0;
						while (vec[0] >= 0.5) vec[0] -= 1.0;
						while (vec[1] < -0.5) vec[1] += 1.0;
						while (vec[1] >= 0.5) vec[1] -= 1.0;
						while (vec[2] < -0.5) vec[2] += 1.0;
						while (vec[2] >= 0.5) vec[2] -= 1.0;*/

						px = vec.GetLength();

						mprintf("    - In Zelle %d als Punkt %d, Abstand %.8f.  ",z2+1,z3+1,px);
						vc->m_vPos.Dump();
						mprintf("\n");
					}

					if (ti < 4)
						dt->m_iPointID[ti] = z2;

					ti++;

				}
			}
		}

		if (ti != 4)
		{
			eprintf("\nError: %d instead of 4 points in tetrahedron %d.\n",ti,z+1);
			for (z2=0;z2<m_oaVoroCells.GetSize();z2++)
			{
				vc = (CVoroCell*)m_oaVoroCells[z2];
				for (z3=0;z3<vc->m_iaVoroPoints.GetSize();z3++)
				{
					if (vc->m_iaVoroPoints[z3] == z)
					{
						vec = vp->m_vPos - vc->m_vPos;

						while (vec[0] < -g_fBoxX/2000.0) vec[0] += g_fBoxX/1000.0;
						while (vec[0] >= g_fBoxX/2000.0) vec[0] -= g_fBoxX/1000.0;
						while (vec[1] < -g_fBoxY/2000.0) vec[1] += g_fBoxY/1000.0;
						while (vec[1] >= g_fBoxY/2000.0) vec[1] -= g_fBoxY/1000.0;
						while (vec[2] < -g_fBoxZ/2000.0) vec[2] += g_fBoxZ/1000.0;
						while (vec[2] >= g_fBoxZ/2000.0) vec[2] -= g_fBoxZ/1000.0;

						px = vec.GetLength();

						mprintf("    - In cell %d as point %d, distance %.8f.  ",z2+1,z3+1,px);
						vc->m_vPos.Dump();
						mprintf("\n");
					}
				}
			}
			delete dt;
			continue;
		}

		m_oaDelaunayTetrahedra.Add(dt);

		if (verbose || sanity)
		{
			pk[0] = 1E10;
			pk[1] = 1E10;
			pk[2] = 1E10;
			pk[3] = 1E10;

			for (z2=0;z2<m_oaVoroCells.GetSize();z2++)
			{
				vc = (CVoroCell*)m_oaVoroCells[z2];

				vec = vp->m_vPos - vc->m_vPos;

				while (vec[0] < -g_fBoxX/2000.0) vec[0] += g_fBoxX/1000.0;
				while (vec[0] >= g_fBoxX/2000.0) vec[0] -= g_fBoxX/1000.0;
				while (vec[1] < -g_fBoxY/2000.0) vec[1] += g_fBoxY/1000.0;
				while (vec[1] >= g_fBoxY/2000.0) vec[1] -= g_fBoxY/1000.0;
				while (vec[2] < -g_fBoxZ/2000.0) vec[2] += g_fBoxZ/1000.0;
				while (vec[2] >= g_fBoxZ/2000.0) vec[2] -= g_fBoxZ/1000.0;

/*				while (vec[0] < -0.5) vec[0] += 1.0;
				while (vec[0] >= 0.5) vec[0] -= 1.0;
				while (vec[1] < -0.5) vec[1] += 1.0;
				while (vec[1] >= 0.5) vec[1] -= 1.0;
				while (vec[2] < -0.5) vec[2] += 1.0;
				while (vec[2] >= 0.5) vec[2] -= 1.0;*/

				px = vec.GetLength();

//				px = FoldVector1(vp->m_vPos - vc->m_vPos).GetLength();

				if (px < pk[3])
					pk[3] = px;
				if (pk[3] < pk[2])
				{
					px = pk[2];
					pk[2] = pk[3];
					pk[3] = px;
				}
				if (pk[2] < pk[1])
				{
					px = pk[1];
					pk[1] = pk[2];
					pk[2] = px;
				}
				if (pk[1] < pk[0])
				{
					px = pk[0];
					pk[0] = pk[1];
					pk[1] = px;
				}
			}

			if (verbose)
				mprintf("  Kleinste 4 Abstaende:\n      %.8f\n      %.8f\n      %.8f\n      %.8f\n",pk[0],pk[1],pk[2],pk[3]);

			if (sanity || verbose)
			{
				for (z2=0;z2<4;z2++)
					tfa[z2] = pk[z2];

				for (z2=0;z2<4;z2++)
				{
					vc = (CVoroCell*)m_oaVoroCells[dt->m_iPointID[z2]];
					for (z3=0;z3<vc->m_iaVoroPoints.GetSize();z3++)
					{
						if (vc->m_iaVoroPoints[z3] == z)
						{
							vec = vp->m_vPos - vc->m_vPos;

							while (vec[0] < -g_fBoxX/2000.0) vec[0] += g_fBoxX/1000.0;
							while (vec[0] >= g_fBoxX/2000.0) vec[0] -= g_fBoxX/1000.0;
							while (vec[1] < -g_fBoxY/2000.0) vec[1] += g_fBoxY/1000.0;
							while (vec[1] >= g_fBoxY/2000.0) vec[1] -= g_fBoxY/1000.0;
							while (vec[2] < -g_fBoxZ/2000.0) vec[2] += g_fBoxZ/1000.0;
							while (vec[2] >= g_fBoxZ/2000.0) vec[2] -= g_fBoxZ/1000.0;

/*							while (vec[0] < -0.5) vec[0] += 1.0;
							while (vec[0] >= 0.5) vec[0] -= 1.0;
							while (vec[1] < -0.5) vec[1] += 1.0;
							while (vec[1] >= 0.5) vec[1] -= 1.0;
							while (vec[2] < -0.5) vec[2] += 1.0;
							while (vec[2] >= 0.5) vec[2] -= 1.0;*/

							tfa[z2+4] = vec.GetLength();
						}
					}
				}

				tf = MaxDiff_DoubleArray(tfa,8);
				if (tf > EPS)
				{
					eprintf("\nSanity check failed: Largest Deviation is %.8f > EPS = %.8f.\n%.8f\n%.8f\n%.8f\n%.8f\n%.8f\n%.8f\n%.8f\n%.8f\n\n",tf,EPS,tfa[0],tfa[1],tfa[2],tfa[3],tfa[4],tfa[5],tfa[6],tfa[7]);
				}
			}
		}

	}

	if (sanity)
	{
		mprintf(WHITE,"]\n");
		mprintf("\n    Test passed.\n\n");
	}

	for (z=0;z<m_oaDelaunayTetrahedra.GetSize();z++)
	{
		dt = (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z];

		dt->m_vCenter[0] = ((CVoroPoint*)m_oaVoroPoints[z])->m_vPos[0] * 1000.0;
		dt->m_vCenter[1] = ((CVoroPoint*)m_oaVoroPoints[z])->m_vPos[1] * 1000.0;
		dt->m_vCenter[2] = ((CVoroPoint*)m_oaVoroPoints[z])->m_vPos[2] * 1000.0;

//		vec = dt->m_vCenter - ((CDelaunayPoint*)m_oaDelaunayPoints[dt->m_iPointID[0]])->m_vPos;

		dt->m_fRadius = FoldVector(dt->m_vCenter - ((CDelaunayPoint*)m_oaDelaunayPoints[dt->m_iPointID[0]])->m_vPos).GetLength();
	}

	delete con;
}


void CVoroAnalysis::Parse()
{
	CTimeStep *t;
	CMolecule *m;
	CSingleMolecule *sm;
	int z;

	mprintf(WHITE,">>> Void Analysis >>>\n\n");

	mprintf("    Initializing...\n");
	g_pVoroWrapper->Init();
	mprintf("\n");

	mprintf("*** Voro: Box density is %f particles / Angstrom^3.\n",g_pVoroWrapper->m_fBoxDens);
	mprintf("*** Voro: Using %d x %d x %d blocks.\n",g_pVoroWrapper->m_iBlocksX,g_pVoroWrapper->m_iBlocksY,g_pVoroWrapper->m_iBlocksZ);

	try { t = new CTimeStep(); } catch(...) { t = NULL; }
	if (t == NULL) NewException((double)sizeof(CTimeStep),__FILE__,__LINE__,__PRETTY_FUNCTION__);
	
	t->CopyFrom(&g_TimeStep);

	m = (CMolecule*)g_oaMolecules[0];
	sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[0]];

	t->CenterPos(t->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[sm->m_baAtomIndex.GetSize()-1])->GetAt(1)]);
	t->CenterPos(CxVector3(-g_fBoxX/2.0,-g_fBoxY/2.0,-g_fBoxZ/2.0));
	t->FoldAtomsPositive();
	g_pVoroWrapper->Dump("voro.txt",t);
	mprintf("\n");
	mprintf("    Voro++: Using cell memory for %d particles.\n\n",g_iVoroMemory);

	mprintf("    Performing sanity check...\n\n");
	Build(t,true,false);

//	g_pVoroWrapper->WritePOV(t,"voro.pov",0,0);
	delete t;

//	AskYesNo("    This is hard-coded for Ulrikes Bmim-Br ^^ [accept] ",false);

	m_bSphereHole = AskYesNo("    Do you want to create a spherical hole distribution function (y/n)? [yes] ",true);

	if (m_bSphereHole)
	{
		try { m_pSphereHoleDF = new CDF(); } catch(...) { m_pSphereHoleDF = NULL; }
		if (m_pSphereHoleDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pSphereHoleDF->m_fMinVal = 0;
		m_pSphereHoleDF->m_fMaxVal = AskFloat("    Enter the maximal hole diameter (in pm): [1500.0] ",1500.0f);
		m_pSphereHoleDF->m_iResolution = AskUnsignedInteger("    Enter the binning resolution: [%d] ",int(m_pSphereHoleDF->m_fMaxVal),int(m_pSphereHoleDF->m_fMaxVal));
		m_pSphereHoleDF->Create();
		m_pSphereHoleDF->SetLabelX("Hole diameter");
		m_pSphereHoleDF->SetLabelY("Occurrence");
		mprintf("\n");
	}

	m_bSphereHoleRDF = AskYesNo("    Do you want to create a spherical hole RDF (y/n)? [yes] ",true);

	if (m_bSphereHoleRDF)
	{
		try { m_pSphereHoleRDF = new CDF(); } catch(...) { m_pSphereHoleRDF = NULL; }
		if (m_pSphereHoleRDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pSphereHoleRDF->m_fMinVal = 0;
		m_pSphereHoleRDF->m_fMaxVal = AskUnsignedInteger("    Enter the RDF range (in pm): [%d] ",HalfBox(),HalfBox());
		m_pSphereHoleRDF->m_iResolution = AskUnsignedInteger("    Enter the RDF resolution: [300] ",300);
		m_pSphereHoleRDF->Create();
		mprintf("\n");
	}

	m_bSphereHole2DF = AskYesNo("    Do you want to create a spherical hole 2D DF (y/n)? [no] ",false);

	if (m_bSphereHole2DF)
	{
		try { m_pSphereHole2DF = new C2DF(); } catch(...) { m_pSphereHole2DF = NULL; }
		if (m_pSphereHole2DF == NULL) NewException((double)sizeof(C2DF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pSphereHole2DF->m_fMinVal[0] = 0;
		m_pSphereHole2DF->m_fMaxVal[0] = AskUnsignedInteger("    Enter the RDF channel range (in pm): [%d] ",HalfBox(),HalfBox());
		m_pSphereHole2DF->m_iRes[0] = AskUnsignedInteger("    Enter the RDF channel resolution: [100] ",100);
		m_pSphereHole2DF->m_fMinVal[1] = 0;
		m_pSphereHole2DF->m_fMaxVal[1] = AskUnsignedInteger("    Enter the hole radius range (in pm): [300] ",300);
		m_pSphereHole2DF->m_iRes[1] = AskUnsignedInteger("    Enter the hole radius channel resolution: [50] ",50);
		m_pSphereHole2DF->SetLabelX("Void distance [pm]");
		m_pSphereHole2DF->SetLabelY("Minimal void radius [pm]");
		m_pSphereHole2DF->SetLabelZ("Occurence");
		m_pSphereHole2DF->Create();

		try { m_pSphereHole2DFCounter = new unsigned long[m_pSphereHole2DF->m_iRes[1]]; } catch(...) { m_pSphereHole2DFCounter = NULL; }
		if (m_pSphereHole2DFCounter == NULL) NewException((double)m_pSphereHole2DF->m_iRes[1]*sizeof(unsigned long),__FILE__,__LINE__,__PRETTY_FUNCTION__);

		for (z=0;z<m_pSphereHole2DF->m_iRes[1];z++)
			m_pSphereHole2DFCounter[z] = 0;

		mprintf("\n");
	}

	m_bAllAtomRDF = AskYesNo("    Do you want to create an all-atom RDF (y/n)? [no] ",false);

	if (m_bAllAtomRDF)
	{
		try { m_pAllAtomRDF = new CDF(); } catch(...) { m_pAllAtomRDF = NULL; }
		if (m_pAllAtomRDF == NULL) NewException((double)sizeof(CDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pAllAtomRDF->m_fMinVal = 0;
		m_pAllAtomRDF->m_fMaxVal = AskUnsignedInteger("    Enter the RDF range (in pm): [%d] ",HalfBox(),HalfBox());
		m_pAllAtomRDF->m_iResolution = AskUnsignedInteger("    Enter the RDF resolution: [300] ",300);
		m_pAllAtomRDF->Create();
		mprintf("\n");
	}


	m_bHoleDump = AskYesNo("    Do you want to write out a XYZ hole trajectory (y/n)? [no] ",false);

	if (m_bHoleDump)
	{
		m_fDumpMinHole = AskFloat("    Enter the minimal hole diameter to output (in pm): [400] ",400) / 2.0;
		m_fHoleDump = OpenFileWrite("holedump.xyz",true);
	}

	m_bSphereHoleSDF = AskYesNo("    Create a spherical hole SDF (y/n)? [no] ",false);

	if (m_bSphereHoleSDF)
	{
		g_bVoidSDF = true;

		try { m_pSphereHoleSDF = new CSDF(); } catch(...) { m_pSphereHoleSDF = NULL; }
		if (m_pSphereHoleSDF == NULL) NewException((double)sizeof(CSDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pSphereHoleSDF->Parse(true);

		m_pSphereHoleSDF->m_pSDF->m_fMinVal[0] = -m_pSphereHoleSDF->m_fRadius;
		m_pSphereHoleSDF->m_pSDF->m_fMaxVal[0] = m_pSphereHoleSDF->m_fRadius;
		m_pSphereHoleSDF->m_pSDF->m_fMinVal[1] = -m_pSphereHoleSDF->m_fRadius;
		m_pSphereHoleSDF->m_pSDF->m_fMaxVal[1] = m_pSphereHoleSDF->m_fRadius;
		m_pSphereHoleSDF->m_pSDF->m_fMinVal[2] = -m_pSphereHoleSDF->m_fRadius;
		m_pSphereHoleSDF->m_pSDF->m_fMaxVal[2] = m_pSphereHoleSDF->m_fRadius;
		m_pSphereHoleSDF->m_pSDF->m_iRes[0] = m_pSphereHoleSDF->m_iResolution;
		m_pSphereHoleSDF->m_pSDF->m_iRes[1] = m_pSphereHoleSDF->m_iResolution;
		m_pSphereHoleSDF->m_pSDF->m_iRes[2] = m_pSphereHoleSDF->m_iResolution;
		m_pSphereHoleSDF->m_pSDF->m_iHistogramRes = m_pSphereHoleSDF->m_iHistogramRes;
		m_pSphereHoleSDF->m_pSDF->Create();
	}

	m_bEmptySpaceSDF = AskYesNo("    Create an empty space SDF (y/n)? [no] ",false);

	if (m_bEmptySpaceSDF)
	{
		g_bVoidSDF = true;

		m_bNewEmptyMode = AskYesNo("    Use new method (y/n)? [no] ",false);

		if (m_bNewEmptyMode)
		{
			m_iTempRes = AskUnsignedInteger("    Enter resoultion of the temporary SDF: [300] ",300);

			try { m_pTempSDF = new C3DF(); } catch(...) { m_pTempSDF = NULL; }
			if (m_pTempSDF == NULL) NewException((double)sizeof(C3DF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			m_pTempSDF->m_fMinVal[0] = 0;
			m_pTempSDF->m_fMaxVal[0] = g_fBoxX;
			m_pTempSDF->m_fMinVal[1] = 0;
			m_pTempSDF->m_fMaxVal[1] = g_fBoxY;
			m_pTempSDF->m_fMinVal[2] = 0;
			m_pTempSDF->m_fMaxVal[2] = g_fBoxZ;
			m_pTempSDF->m_iRes[0] = m_iTempRes;
			m_pTempSDF->m_iRes[1] = m_iTempRes;
			m_pTempSDF->m_iRes[2] = m_iTempRes;
			m_pTempSDF->m_iHistogramRes = m_pEmptySpaceSDF->m_iHistogramRes;
		} else
		{
			m_bEmptySpaceSDF_RefMol = AskYesNo("    Include the atoms of the reference molecule (y/n)? [yes] ",true);
			m_iEmptySpaceBorder = AskUnsignedInteger("    Enter the size of the border in bins: [10] ",10);

			try { m_pTempSDF = new C3DF(); } catch(...) { m_pTempSDF = NULL; }
			if (m_pTempSDF == NULL) NewException((double)sizeof(C3DF),__FILE__,__LINE__,__PRETTY_FUNCTION__);

			m_pTempSDF->m_fMinVal[0] = -m_pEmptySpaceSDF->m_fRadius;
			m_pTempSDF->m_fMaxVal[0] = m_pEmptySpaceSDF->m_fRadius;
			m_pTempSDF->m_fMinVal[1] = -m_pEmptySpaceSDF->m_fRadius;
			m_pTempSDF->m_fMaxVal[1] = m_pEmptySpaceSDF->m_fRadius;
			m_pTempSDF->m_fMinVal[2] = -m_pEmptySpaceSDF->m_fRadius;
			m_pTempSDF->m_fMaxVal[2] = m_pEmptySpaceSDF->m_fRadius;
			m_pTempSDF->m_iRes[0] = m_pEmptySpaceSDF->m_iResolution;
			m_pTempSDF->m_iRes[1] = m_pEmptySpaceSDF->m_iResolution;
			m_pTempSDF->m_iRes[2] = m_pEmptySpaceSDF->m_iResolution;
			m_pTempSDF->m_iHistogramRes = m_pEmptySpaceSDF->m_iHistogramRes;
		}
		m_pTempSDF->Create();

		try { m_pEmptySpaceSDF = new CSDF(); } catch(...) { m_pEmptySpaceSDF = NULL; }
		if (m_pEmptySpaceSDF == NULL) NewException((double)sizeof(CSDF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
		
		m_pEmptySpaceSDF->Parse(true);

		m_pEmptySpaceSDF->m_pSDF->m_fMinVal[0] = -m_pEmptySpaceSDF->m_fRadius;
		m_pEmptySpaceSDF->m_pSDF->m_fMaxVal[0] = m_pEmptySpaceSDF->m_fRadius;
		m_pEmptySpaceSDF->m_pSDF->m_fMinVal[1] = -m_pEmptySpaceSDF->m_fRadius;
		m_pEmptySpaceSDF->m_pSDF->m_fMaxVal[1] = m_pEmptySpaceSDF->m_fRadius;
		m_pEmptySpaceSDF->m_pSDF->m_fMinVal[2] = -m_pEmptySpaceSDF->m_fRadius;
		m_pEmptySpaceSDF->m_pSDF->m_fMaxVal[2] = m_pEmptySpaceSDF->m_fRadius;
		m_pEmptySpaceSDF->m_pSDF->m_iRes[0] = m_pEmptySpaceSDF->m_iResolution;
		m_pEmptySpaceSDF->m_pSDF->m_iRes[1] = m_pEmptySpaceSDF->m_iResolution;
		m_pEmptySpaceSDF->m_pSDF->m_iRes[2] = m_pEmptySpaceSDF->m_iResolution;
		m_pEmptySpaceSDF->m_pSDF->m_iHistogramRes = m_pEmptySpaceSDF->m_iHistogramRes;
		m_pEmptySpaceSDF->m_pSDF->Create();
	}

	mprintf(WHITE,"\n<<< End of Void Analysis <<<\n\n");
}


void CVoroAnalysis::Step(CTimeStep *ts)
{
	int z, z2, z3, z4, ti;
	CxVec3Array tva;
	CDelaunayTetrahedron *dt, *dt2;
	CMolecule *m;
	CSingleMolecule *sm;
	double tf, tf2, tfx;


	if (m_bSphereHole || m_bSphereHoleRDF || m_bSphereHole2DF || m_bHoleDump || m_bSphereHoleRDF)
	{
		Clean();

		Build(ts,false,false);
	}

	if (m_bSphereHole)
	{
		for (z=0;z<m_oaDelaunayTetrahedra.GetSize();z++)
		{
			dt = (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z];
			m_pSphereHoleDF->AddToBin(dt->m_fRadius*2.0);

			if (dt->m_fRadius > 250.0)
				tva.Add(dt->m_vCenter);
		}
	}

	if (m_bSphereHoleRDF)
	{
		for (z=0;z<m_oaDelaunayTetrahedra.GetSize();z++)
		{
			dt = (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z];

			for (z2=z+1;z2<m_oaDelaunayTetrahedra.GetSize();z2++)
			{
				dt2 = (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z2];

				tf = FoldVector(dt->m_vCenter - dt2->m_vCenter).GetLength();

				m_pSphereHoleRDF->AddToBin(tf);
			}
		}
	}

	if (m_bSphereHole2DF)
	{
		tfx = 1.0 / m_pSphereHole2DF->m_iRes[1] * (m_pSphereHole2DF->m_fMaxVal[1]-m_pSphereHole2DF->m_fMinVal[1]);

		for (z2=0;z2<m_oaDelaunayTetrahedra.GetSize();z2++)
		{
			dt = (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z2];

			for (z3=z2+1;z3<m_oaDelaunayTetrahedra.GetSize();z3++)
			{
				dt2 = (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z3];

				tf2 = FoldVector(dt->m_vCenter - dt2->m_vCenter).GetLength();

				tf = m_pSphereHole2DF->m_fMinVal[1];

				for (z=0;z<m_pSphereHole2DF->m_iRes[1];z++)
				{
					tf += tfx;

					if (dt->m_fRadius < tf)
						continue;

					if (dt2->m_fRadius < tf)
						continue;

					m_pSphereHole2DFCounter[z]++;

					m_pSphereHole2DF->AddToBin(tf2,tf);
				}
			}
		}
	}

	if (m_bAllAtomRDF)
	{
		for (z=0;z<g_iGesAtomCount;z++)
		{
			for (z2=z+1;z2<g_iGesAtomCount;z2++)
			{
				tf = FoldVector(ts->m_vaCoords[z] - ts->m_vaCoords[z2]).GetLength();
				m_pAllAtomRDF->AddToBin(tf);
			}
		}
	}

	if (m_bHoleDump)
	{
		mfprintf(m_fHoleDump,"  %d\n",g_iGesAtomCount+m_iMaxVoroPoints);
		mfprintf(m_fHoleDump,"\n");

		for (z=0;z<g_oaMolecules.GetSize();z++)
		{
			m = (CMolecule*)g_oaMolecules[z];
			if (m->m_bPseudo)
				continue;
			for (z2=0;z2<m->m_laSingleMolIndex.GetSize();z2++)
			{
				sm = (CSingleMolecule*)g_oaSingleMolecules[m->m_laSingleMolIndex[z2]];
				for (z3=0;z3<m->m_baAtomIndex.GetSize();z3++)
				{
					if ((!g_bSaveVirtAtoms) && (m->m_baAtomIndex[z3] == g_iVirtAtomType))
						continue;
					for (z4=0;z4<((CxIntArray*)sm->m_oaAtomOffset[z3])->GetSize();z4++)
						mfprintf(m_fHoleDump,"  %s  %8.5f  %8.5f  %8.5f\n",((CAtom*)g_oaAtoms[m->m_baAtomIndex[z3]])->m_sName,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][0]/100.0f,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][1]/100.0f,ts->m_vaCoords[((CxIntArray*)sm->m_oaAtomOffset[z3])->GetAt(z4)][2]/100.0f);
				}
			}
		}

		ti = 0;
		for (z=0;z<m_oaDelaunayTetrahedra.GetSize();z++)
		{
			dt = (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z];
			if (dt->m_fRadius > m_fDumpMinHole)
			{
				ti++;
				mfprintf(m_fHoleDump,"  He  %8.5f  %8.5f  %8.5f\n",dt->m_vCenter[0]/100.0,dt->m_vCenter[1]/100.0,dt->m_vCenter[2]/100.0);
			}
		}
		for (z=ti;z<m_iMaxVoroPoints;z++)
			mfprintf(m_fHoleDump,"  He  %8.5f  %8.5f  %8.5f\n",ts->m_vaCoords[0][0]/100.0f,ts->m_vaCoords[0][1]/100.0f,ts->m_vaCoords[0][2]/100.0f);
	}
}


void CVoroAnalysis::Finish()
{
	int z, z2;
	C3DF *tempSDF;
	char buf[256];

	if (m_bSphereHole)
	{
		m_pSphereHoleDF->NormBinIntegral(100000.0);
		m_pSphereHoleDF->Write("","sphere_hole_df.csv","",false);
	}

	if (m_bSphereHoleRDF)
	{
		m_pSphereHoleRDF->CorrectRadialDist();
		m_pSphereHoleRDF->MultiplyBin(g_fBoxX*g_fBoxY*g_fBoxZ / (4.0/3.0*Pi) / (m_pSphereHoleRDF->m_fBinEntries+m_pSphereHoleRDF->m_fSkipEntries));
		m_pSphereHoleRDF->Write("","sphere_hole_rdf.csv","",false);
	}

	if (m_bSphereHole2DF)
	{
		m_pSphereHole2DF->CorrectRadialDist(0);
		for (z=0;z<m_pSphereHole2DF->m_iRes[1];z++)
		{
			if (m_pSphereHole2DFCounter[z] != 0)
			{
				for (z2=0;z2<m_pSphereHole2DF->m_iRes[0];z2++)
					m_pSphereHole2DF->m_pBin[z2+m_pSphereHole2DF->m_iRes[0]*z] *= 10000.0 / m_pSphereHole2DFCounter[z];
			}
		}
		m_pSphereHole2DF->Log();
//		m_pSphereHoleRDF->MultiplyBin(g_fBoxX*g_fBoxY*g_fBoxZ / (4.0/3.0*Pi) / (m_pSphereHoleRDF->m_fBinEntries+m_pSphereHoleRDF->m_fSkipEntries));
		m_pSphereHole2DF->WriteMathematicaNb("","sphere_hole_2df.nb","",false);
		m_pSphereHole2DF->WriteGnuplotInput("","sphere_hole_2df","",false);
	}

	if (m_bAllAtomRDF)
	{
		m_pAllAtomRDF->CorrectRadialDist();
		m_pAllAtomRDF->MultiplyBin(g_fBoxX*g_fBoxY*g_fBoxZ / (4.0/3.0*Pi) / g_iSteps / g_iGesAtomCount / g_iGesAtomCount * 2.0 * g_iStride);
		m_pAllAtomRDF->Write("","all_atom_rdf.csv","",false);
	}

	if (m_bHoleDump)
	{
		fclose(m_fHoleDump);
	}

	if (m_bEmptySpaceSDF)
	{
		mprintf(WHITE,"*** Empty Space SDF ***\n");
		m_pEmptySpaceSDF->m_pSDF->MultiplyBin(pow(m_pEmptySpaceSDF->m_iResolution/m_pEmptySpaceSDF->m_fRadius*1000.0,3) / (double)g_iSteps / ((CMolecule*)g_oaMolecules[g_iFixMol])->m_laSingleMolIndex.GetSize() * (g_bDoubleBox?g_iDoubleBoxFactor:1));
		for (z2=0;z2<=g_iSDFSmoothGrade;z2++)
		{
			try { tempSDF = new C3DF(); } catch(...) { tempSDF = NULL; }
			if (tempSDF == NULL) NewException((double)sizeof(C3DF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			tempSDF->CopyFrom(m_pEmptySpaceSDF->m_pSDF);
			if (z2 != 0)
			{
				tempSDF->Smooth(z2);
				sprintf(buf,".s%d.plt",z2);
			} else sprintf(buf,".plt");
			mprintf("    Saving SDF as \"sdf_emptyspace%s\"...\n",buf);
			tempSDF->WritePLT("sdf_","emptyspace",buf,true);

			if (z2 != 0)
				sprintf(buf,".s%d.cube",z2);
					else sprintf(buf,".cube");
			mprintf("    Saving SDF as \"sdf_emptyspace%s\"...\n",buf);
			tempSDF->WriteCube("sdf_","emptyspace",buf,true);
		}
	}

	if (m_bSphereHoleSDF)
	{
		mprintf(WHITE,"*** Spherical Hole SDF ***\n");
		m_pSphereHoleSDF->m_pSDF->MultiplyBin(pow(m_pSphereHoleSDF->m_iResolution/m_pSphereHoleSDF->m_fRadius*1000.0,3) / (double)g_iSteps / ((CMolecule*)g_oaMolecules[g_iFixMol])->m_laSingleMolIndex.GetSize() * (g_bDoubleBox?g_iDoubleBoxFactor:1));
		for (z2=0;z2<=g_iSDFSmoothGrade;z2++)
		{
			try { tempSDF = new C3DF(); } catch(...) { tempSDF = NULL; }
			if (tempSDF == NULL) NewException((double)sizeof(C3DF),__FILE__,__LINE__,__PRETTY_FUNCTION__);
			
			tempSDF->CopyFrom(m_pSphereHoleSDF->m_pSDF);
			if (z2 != 0)
			{
				tempSDF->Smooth(z2);
				sprintf(buf,".s%d.plt",z2);
			} else sprintf(buf,".plt");
			mprintf("    Saving SDF as \"sdf_spherehole%s\"...\n",buf);
			tempSDF->WritePLT("sdf_","spherehole",buf,true);

			if (z2 != 0)
				sprintf(buf,".s%d.cube",z2);
					else sprintf(buf,".cube");
			mprintf("    Saving SDF as \"sdf_spherehole%s\"...\n",buf);
			tempSDF->WriteCube("sdf_","spherehole",buf,true);
		}
	}
}


void CVoroAnalysis::Clean()
{
	int z;

	for (z=0;z<m_oaDelaunayPoints.GetSize();z++)
		delete (CDelaunayPoint*)m_oaDelaunayPoints[z];
	m_oaDelaunayPoints.RemoveAll_KeepSize();

	for (z=0;z<m_oaDelaunayTetrahedra.GetSize();z++)
		delete (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z];
	m_oaDelaunayTetrahedra.RemoveAll_KeepSize();

	for (z=0;z<m_oaVoroCells.GetSize();z++)
		delete (CVoroCell*)m_oaVoroCells[z];
	m_oaVoroCells.RemoveAll_KeepSize();

	for (z=0;z<m_oaVoroPoints.GetSize();z++)
		delete (CVoroPoint*)m_oaVoroPoints[z];
	m_oaVoroPoints.RemoveAll_KeepSize();
}


void CVoroAnalysis::BinEmptySDF()
{
	int x, y, z, ti, ti2, ti3;

	for (z=m_iEmptySpaceBorder;z<m_pTempSDF->m_iRes[2]-m_iEmptySpaceBorder;z++)
	{
		ti = z*m_pTempSDF->m_iResXY;
		for (y=m_iEmptySpaceBorder;y<m_pTempSDF->m_iRes[1]-m_iEmptySpaceBorder;y++)
		{
			ti2 = ti + y * m_pTempSDF->m_iRes[0];
			for (x=m_iEmptySpaceBorder;x<m_pTempSDF->m_iRes[0]-m_iEmptySpaceBorder;x++)
			{
				ti3 = ti2 + x;
				if (m_pTempSDF->m_pBin[ti3] == 0)
					m_pEmptySpaceSDF->m_pSDF->m_pBin[ti3]++;
			}
		}
	}
}


void CVoroAnalysis::BinEmptySDF_New(CxMatrix3 *m, CxVector3 *c)
{
	int x, y, z, ti, ti2, ti3;
	CxVector3 v, v2;

	for (z=0;z<m_pEmptySpaceSDF->m_pSDF->m_iRes[2];z++)
	{
		ti = z*m_pEmptySpaceSDF->m_pSDF->m_iResXY;
		v[2] = ((double)z)/m_pEmptySpaceSDF->m_pSDF->m_iRes[2]*(m_pEmptySpaceSDF->m_pSDF->m_fMaxVal[2]-m_pEmptySpaceSDF->m_pSDF->m_fMinVal[2])+m_pEmptySpaceSDF->m_pSDF->m_fMinVal[2] + (*c)[2];
		for (y=0;y<m_pEmptySpaceSDF->m_pSDF->m_iRes[1];y++)
		{
			ti2 = ti + y * m_pEmptySpaceSDF->m_pSDF->m_iRes[0];
			v[1] = ((double)y)/m_pEmptySpaceSDF->m_pSDF->m_iRes[1]*(m_pEmptySpaceSDF->m_pSDF->m_fMaxVal[1]-m_pEmptySpaceSDF->m_pSDF->m_fMinVal[1])+m_pEmptySpaceSDF->m_pSDF->m_fMinVal[1] + (*c)[1];
			for (x=0;x<m_pEmptySpaceSDF->m_pSDF->m_iRes[0];x++)
			{
				ti3 = ti2 + x;
				v[0] = ((double)x)/m_pEmptySpaceSDF->m_pSDF->m_iRes[0]*(m_pEmptySpaceSDF->m_pSDF->m_fMaxVal[0]-m_pEmptySpaceSDF->m_pSDF->m_fMinVal[0])+m_pEmptySpaceSDF->m_pSDF->m_fMinVal[0] + (*c)[0];

				v2 = *m * v;

	//			mprintf(" %f %f %f --> %f %f %f\n",v[0],v[1],v[2],v2[0],v2[1],v2[2]);

				while (v2[0] >= g_fBoxX)
					v2[0] -= g_fBoxX;
				while (v2[0] < 0)
					v2[0] += g_fBoxX;
				while (v2[1] >= g_fBoxY)
					v2[1] -= g_fBoxY;
				while (v2[1] < 0)
					v2[1] += g_fBoxY;
				while (v2[2] >= g_fBoxZ)
					v2[2] -= g_fBoxZ;
				while (v2[2] < 0)
					v2[2] += g_fBoxZ;

	//			mprintf(" %f %f %f\n",v2[0],v2[1],v2[2]);

				if (m_pTempSDF->IsZero(v2[0],v2[1],v2[2]))
					m_pEmptySpaceSDF->m_pSDF->m_pBin[ti3]++;
			}
		}
	}
//	m_pEmptySpaceSDF->m_pSDF->WritePLT("","bla2.plt","",true);
//	abort();
}


void CVoroAnalysis::BinSphereHoleSDF(CxMatrix3 *m, CxVector3 *c)
{
	int z;
	CDelaunayTetrahedron *dt;
	CxVector3 vec;

	m_pTempVA->RemoveAll_KeepSize();

	for (z=0;z<m_oaDelaunayTetrahedra.GetSize();z++)
	{
		dt = (CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z];
		vec = dt->m_vCenter - *c;

		while (vec[0] < -g_fBoxX/2.0) vec[0] += g_fBoxX;
		while (vec[0] >= g_fBoxX/2.0) vec[0] -= g_fBoxX;
		while (vec[1] < -g_fBoxY/2.0) vec[1] += g_fBoxY;
		while (vec[1] >= g_fBoxY/2.0) vec[1] -= g_fBoxY;
		while (vec[2] < -g_fBoxZ/2.0) vec[2] += g_fBoxZ;
		while (vec[2] >= g_fBoxZ/2.0) vec[2] -= g_fBoxZ;

		m_pTempVA->Add(vec);
	}

	for (z=0;z<m_oaDelaunayTetrahedra.GetSize();z++)
	{
		vec = *m * (*m_pTempVA)[z];

		if (m_pSphereHoleSDF->m_bVdWSpheres)
			m_pSphereHoleSDF->m_pSDF->AddToBin_Sphere(vec,((CDelaunayTetrahedron*)m_oaDelaunayTetrahedra[z])->m_fRadius);
				else m_pSphereHoleSDF->m_pSDF->AddToBin(vec);
	}

}
