use core;
use core:lang;

// Generate the SrcName for the StrBuf type.
SrcName strBufType() {
	SrcName t;
	t.add("core");
	t.add("StrBuf");
	t;
}

// Implementation of the string concatenation operator (#).
class ConcatOperator extends OpInfo {
	init(SStr op, Int prio) {
		init(op, prio, false) {}
	}

	Expr meaning(Block block, Expr lhs, Expr rhs) {
		ConcatExpr r(pos, block);
		appendTo(r, lhs);
		appendTo(r, rhs);
		r;
	}

}

// Helper to append parameters.
void appendTo(ConcatExpr expr, Expr e) on Compiler {
	if (e as Operator) {
		OpInfo info = e.op;
		if (info as ConcatOperator) {
			appendTo(expr, e.lhs);
			appendTo(expr, e.rhs);
		} else {
			expr.append(e);
		}
	} else {
		expr.append(e);
	}
}


// The concatenation expression itself. It generates code of
// the structure: { StrBuf b; { buf.add(...); buf.add(...); } buf; }
class ConcatExpr extends Expr {
	// Inner block, so that we can easily add new expressions at the right position.
	ExprBlock outer;
	ExprBlock inner;
	Var strBufVar;

	// Saved state, if any.
	Var? savedState;

	// Create the root node.
	init(SrcPos pos, Block parent) {
		ExprBlock outer(pos, parent);
		ExprBlock inner(pos, outer);
		init(pos) {
			outer = outer;
			inner = inner;
			strBufVar(outer, strBufType, SStr("b"), Actuals());
		}

		// Generate the outer parts.
		outer.add(strBufVar);
		outer.add(inner);
		outer.add(namedExpr(outer, SStr("toS"), LocalVarAccess(pos, strBufVar.var), Actuals()));
	}

	// Save the state of the string buffer. Only done the first time called.
	void saveState() {
		if (savedState)
			return;

		Var v(inner, SStr("fmt"), namedExpr(inner, SStr("format"), LocalVarAccess(SrcPos(), strBufVar.var)));
		savedState = v;
		inner.add(v);
	}

	// Restore the state of the string buffer.
	void restoreState() {
		unless (savedState) {
			return;
		}

		LocalVarAccess buf(SrcPos(), strBufVar.var);
		LocalVarAccess fmt(SrcPos(), savedState.var);
		inner.add(namedExpr(inner, SStr("format"), buf, Actuals(fmt)));
	}

	// Append a node to the end of the concatenation.
	void append(Expr e) {
		// Does the StrBuf class have a member 'add(T)'?
		if (appendStrBuf(e)) {
		} else if (appendToS(e)) {
		} else {
			StrBuf msg;
			msg << "Can not convert " << e.result << " to string by calling 'toS'.";
			throw SyntaxError(pos, msg.toS);
		}
	}

	// Append a node, trying to use a << operator directly.
	private Bool appendStrBuf(Expr e) {
		// Try to find an << operator in our current scope.
		Scope s = outer.scope;
		if (appendStrBuf(e, s))
			return true;

		// Try findin it in the parent scope of 'e'.
		if (in = e.result.type.type) {
			if (appendStrBuf(e, s.child(in)))
				return true;
		}

		// Try to find in StrBuf explicitly.
		if (in = rootScope.find(strBufType)) {
			if (appendStrBuf(e, s.child(in)))
				return true;
		}

		false;
	}

	// Append a node, trying to call add on the StrBuf using a << operator found using 'scope'.
	private Bool appendStrBuf(Expr e, Scope scope) {
		Value v = rootScope.value(strBufType);
		var part = SimplePart("<<", Value:[v, e.result.type]);
		if (member = scope.find(Name(part)) as Function) {
			Actuals params;
			params.add(LocalVarAccess(pos, strBufVar.var));
			params.add(e);
			inner.add(FnCall(pos, scope, member, params));
			true;
		} else {
			false;
		}
	}

	// Append a node, trying to call toS on the object first.
	private Bool appendToS(Expr e) {
		Value r = e.result.type;
		if (t = r.type) {
			Scope findIn = outer.scope.child(t);
			var part = SimplePart("toS", Value:[r]);
			Named? member = findIn.find(Name(part));
			if (member as Function) {
				FnCall call(pos, outer.scope, member, Actuals(e));
				appendStrBuf(call);
			} else {
				false;
			}
		} else {
			false;
		}
	}

	// Relay code generation and type info to the blocks.
	ExprResult result() {
		outer.result();
	}

	void code(CodeGen g, CodeResult r) {
		outer.code(g, r);
	}

	// Nice output.
	Str toS() {
		outer.toS;
	}

}


// Fixup the strings from multiline str{} literals.
Expr rawMultilineStrConstant(SrcPos pos, Str str) {
	rawStrConstant(pos, str.removeIndentation.trimBlankLines);
}

/**
 * Interpolated strings. Uses the # syntax.
 */

/**
 * An interpolated string element.
 */
class InterpolatedItem on Compiler {
	// Modifiers to append before 'insert'.
	Expr[] modifiers;

	// Expression to insert.
	Expr insert;

	// Context.
	Block context;

	// Create.
	init(Block ctx, Expr e) {
		init() { insert = e; context = ctx; }
	}

	// Add a modifier.
	void modify(SrcPos pos, Str name, Actuals params) {
		SrcName n(pos);
		n.add("core");
		n.add(name);

		modifiers << namedExpr(context, n, params);
	}

	// Wrap 'expr'.
	void wrap(SrcPos pos, Str name) {
		SrcName n(pos);
		n.add("core");
		n.add(name);

		insert = namedExpr(context, n, Actuals(insert));
	}
}

// Create string interpolation expressions.
Expr interpolateStr(Block parent, Expr first, InterpolatedItem item, Str second) {
	var r = if (first as ConcatExpr) {
		first;
	} else {
		ConcatExpr r(first.pos, parent);
		r.append(first);
		r;
	};

	// Append 'item' with any modifiers. If we have any modifiers, save the state first, so that we
	// can restore it later!
	if (item.modifiers.any) {
		r.saveState();

		for (m in item.modifiers)
			r.append(m);

		r.append(item.insert);
		r.restoreState();
	} else {
		r.append(item.insert);
	}

	r.append(strConstant(first.pos, second));
	r;
}

// Set width of the output.
void setWidth(SrcPos pos, InterpolatedItem item, Str nr) {
	item.modify(pos, "width", Actuals(intConstant(pos, nr)));
}

// Set to left align.
void setLeft(SrcPos pos, InterpolatedItem item) {
	item.modify(pos, "left", Actuals());
}

// Set to right align.
void setRight(SrcPos pos, InterpolatedItem item) {
	item.modify(pos, "right", Actuals());
}

// Set fill character.
void setFill(SrcPos pos, InterpolatedItem item, Str fill) {
	var cp = NumLiteral(pos, fill.begin.v.codepoint.long);
	SrcName charName(pos);
	charName.add("core");
	charName.add("Char");
	var ch = namedExpr(item.context, charName, Actuals(cp));
	item.modify(pos, "fill", Actuals(ch));
}

// Set to hex.
void setHex(SrcPos pos, InterpolatedItem item) {
	item.wrap(pos, "hex");
}

// Set significant digits.
void setSignificant(SrcPos pos, InterpolatedItem item, Str digits) {
	item.modify(pos, "significant", Actuals(intConstant(pos, digits)));
}

void setDecimals(SrcPos pos, InterpolatedItem item, Str digits) {
	item.modify(pos, "fixed", Actuals(intConstant(pos, digits)));
}

void setScientific(SrcPos pos, InterpolatedItem item, Str digits) {
	item.modify(pos, "scientific", Actuals(intConstant(pos, digits)));
}

