# Sorune Libraries
# Copyright (C) 2004-2005 Darren Smith
# All Rights Reserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# e-mail: sorune2004@yahoo.com
#

my $message_window = undef;
sub enableGuiMessaging($)
{
    $message_window = shift;
}

sub message($$)
{
    my ($type,$message) = @_;

    if (defined $message_window) {
        if ($type eq "ERR") {
            $message_window->insert('end',"Error: $message");
        } elsif ($type eq "WARN") {
            $message_window->insert('end',"Warning: $message");
        } elsif ($type eq "INFO") {
            $message_window->insert('end',$message);
        }
        $message_window->see('end');
    } else {
        if ($type eq "ERR") {
            print STDERR "Error: $message";
        } elsif ($type eq "WARN") {
            print STDERR "Warning: $message";
        } elsif ($type eq "INFO") {
            print $message;
        }
    }
}

sub delByName($$$$$)
{
    my ($dbRef,$playlistDbRef,$type,$name,$neurosHome) = @_;
    foreach my $file (keys %$dbRef) {
        if (defined $dbRef->{$file}{$type} and
            $dbRef->{$file}{$type} eq $name) {
            removeFromPlaylist($dbRef,$playlistDbRef,undef,$file,$neurosHome);
            (my $localFile = $file) =~ s/^[CD]:/$neurosHome/i;
            if (-r $localFile) {
                $dbRef->{$file}{'delete'} = 1;
            } else {
                delete $dbRef->{$file};
            }
        }
    }
}

sub delByKey($$$$$)
{
    my ($dbRef,$playlistDbRef,$file,$neurosHome,$neurosDrive) = @_;
    if (defined $dbRef->{$file}) {
        removeFromPlaylist($dbRef,$playlistDbRef,undef,$file,$neurosHome);
        (my $localFile = $file) =~ s/^$neurosDrive/$neurosHome/;
        if (-r $localFile) {
            $dbRef->{$file}{'delete'} = 1;
        } else {
            delete $dbRef->{$file};
        }
    }
}

sub fixDuplicateAlbums($)
{
    my ($musicDbRef) = @_;
    my %albums = ();

    foreach my $key (keys %$musicDbRef) {
        push @{$albums{$musicDbRef->{$key}{'album'}}}, $key;
    }

    foreach my $name (keys %albums) {
        my ($dup,$dirname,$olddir,@artists) = (0,undef,undef,());
        foreach my $key (sort @{$albums{$name}}) {
            $dirname = dirname($key);
            if (!defined $olddir) {
                $olddir = $dirname;
            } elsif ($dirname ne $olddir) {
                if (defined $musicDbRef->{$key}{'artist'}) {
                    push @artists, $musicDbRef->{$key}{'artist'};
                }
                $dup = 1;
                $olddir = $dirname
            }
        }

        if ($dup) {
            foreach my $key (@{$albums{$name}}) {
                if (defined $musicDbRef->{$key}{'artist'}) {
                    my $count = 0;
                    foreach my $artist (@artists) {
                        if ($musicDbRef->{$key}{'artist'} eq $artist) {
                            $count++;
                        }
                    }
                    if ($name !~ /\]$/) {
                        my $artist = $musicDbRef->{$key}{'artist'};
                        if ($count > 1 and defined 
                            $musicDbRef->{$key}{'date'}) {
                            my $date = $musicDbRef->{$key}{'date'};
                            $musicDbRef->{$key}{'album'} = 
                                "$name [$artist, $date]";
                        } else {
                            $musicDbRef->{$key}{'album'} = "$name [$artist]";
                        }
                    }
                }
            }
        }
    }
}

my ($addFile_dir,$addFile_oldDir,$addFile_tracknum) = ("","",0);
my $addFile_year = (localtime(time))[5] + 1900;
sub addFile($$$$$$$$)
{
    my ($file,$musicHome,$neurosHome,$musicDbRef,$playlistDbRef,
        $cfgRef,$neurosDrive,$force) = @_;

    $file = ascii($file);
    if (!-r $file) {
        message('ERR',"$file is not readable, skipping.\n");
        return -1;
    }
    $file =~ s/\\/\//g;
    $musicHome =~ s/\\/\//g;
    my $neurosPath = getNeurosFilename($musicHome,$file,$neurosDrive);

    # Do not parse if it already exists
    if (defined $musicDbRef->{$neurosPath} and !$force) {
        # Verify same size
        if ($musicDbRef->{$neurosPath}{'size'} == (stat($file))[7]) {
            if (defined $musicDbRef->{$neurosPath}{'delete'}) {
                delete $musicDbRef->{$neurosPath}{'delete'};
            }
            (my $neurosFile = $neurosPath) =~ s/^[CD]:/$neurosHome/i;
            if (!-r $neurosFile) {
                $musicDbRef->{$neurosPath}{'add'} = 1;
            } elsif (defined $musicDbRef->{$neurosPath}{'add'}) {
                delete $musicDbRef->{$neurosPath}{'add'};
            }
            return 0;
        } else {
            delete $musicDbRef->{$neurosPath};
        }
    }

    if ($file =~ /\.ogg$/i) {
        $musicDbRef->{$neurosPath} = parseOgg($file,$cfgRef);
    } elsif ($file =~ /\.mp3$/i) {
        $musicDbRef->{$neurosPath} = parseMp3($file,$cfgRef);
    } elsif ($file =~ /\.wma$/i) {
        $musicDbRef->{$neurosPath} = parseWma($file,$cfgRef);
    } elsif ($file =~ /\.wav$/i) {
        $musicDbRef->{$neurosPath} = parseWav($file,$cfgRef);
    } elsif ($file =~ /\.m3u$/i) {
        parseM3u($file,$musicDbRef,$playlistDbRef,$cfgRef,
            $musicHome,$neurosHome,$neurosDrive);
        relate($musicDbRef,$playlistDbRef);
        return 0;
    }

    # If unable to determine length, delete from db
    if (!defined $musicDbRef->{$neurosPath}{'length'} or 
        $musicDbRef->{$neurosPath}{'length'} eq "0") {
        message('ERR',"$file is unsupported, skipping.\n");
        if (defined $musicDbRef->{$neurosPath}) {
            delete $musicDbRef->{$neurosPath};
        }
        return -1;
    }

    # Set size and localFile fields
    $musicDbRef->{$neurosPath}{'size'} = (stat($file))[7];
    $musicDbRef->{$neurosPath}{'localFile'} = $file;

    # Check if file already exists on Neuros
    (my $neurosFile = $neurosPath) =~ s/^[CD]:/$neurosHome/i;
    if ($file ne $neurosFile) {
        if (-r $neurosFile) {
            if (defined $musicDbRef->{$neurosPath}{'add'}) {
                delete $musicDbRef->{$neurosPath}{'add'};
            }
            if (defined $musicDbRef->{$neurosPath}{'delete'}) {
                delete $musicDbRef->{$neurosPath}{'delete'};
            }
        } else {
            $musicDbRef->{$neurosPath}{'add'} = 1;
        }
    }

    # Check for Various artist and for moving "The" to end of artist name
    if (defined $musicDbRef->{$neurosPath}{'artist'}) {
        if ($musicDbRef->{$neurosPath}{'artist'} eq 'Various') {
            if ($musicDbRef->{$neurosPath}{'title'} =~ /^(.*)\s*\((.*)\)$/) {
                $musicDbRef->{$neurosPath}{'artist2'} = $2;
                $musicDbRef->{$neurosPath}{'title'} = $1;
            } else {
                $musicDbRef->{$neurosPath}{'artist2'} = "Unknown";
            }
        } elsif ($musicDbRef->{$neurosPath}{'artist'} =~ /^The /i) {
            if ($cfgRef->{'nam'}{'artistthe'} == 1) {
                $musicDbRef->{$neurosPath}{'artist'} =~ s/^(The) (.*)/$2, $1/i;
            } elsif ($cfgRef->{'nam'}{'artistthe'} == 2) {
                $musicDbRef->{$neurosPath}{'artist'} =~ s/^The //i;
            }
        }
    }

    # Correct tracknumber x/y tags (only use characters up to first non-digit).
    if (defined $musicDbRef->{$neurosPath}{'tracknumber'}) {
        $musicDbRef->{$neurosPath}{'tracknumber'} =~ s/[^\d].*//;
    }

    # Check for missing fields
    my $missing = 0;
    foreach my $field ('album','artist','date','genre','title',
        'tracknumber') {
        if (!(defined $musicDbRef->{$neurosPath}{$field} and
            $musicDbRef->{$neurosPath}{$field} ne "")) {
            $missing = 1;
            last;
        }
    }

    if ($missing) {
        # Fill in missing tags by filename
        if (defined $cfgRef->{'general'}{'untaggedformat'}) {
            tagByFilename($file,$musicDbRef,$musicHome,$neurosPath,
                $cfgRef->{'general'}{'untaggedformat'});
        }

        # Fill in missing tags with unknown
        foreach my $field (('album','artist','genre')) {
            if (!defined $musicDbRef->{$neurosPath}{$field} or
                $musicDbRef->{$neurosPath}{$field} eq "") {
                $musicDbRef->{$neurosPath}{$field} = "Unknown";
            }
        }

        # Fill in missing title with filename
        if (!defined $musicDbRef->{$neurosPath}{'title'} or
            $musicDbRef->{$neurosPath}{'title'} eq "") {
            (my $tag = $neurosPath) =~ s/^.*\/(.*)\..*$/$1/;
            fixTag(\$tag);
            $musicDbRef->{$neurosPath}{'title'} = $tag;
        }

        # Fill in default date
        if (!defined $musicDbRef->{$neurosPath}{'date'} or
            $musicDbRef->{$neurosPath}{'date'} eq "0" or
            $musicDbRef->{$neurosPath}{'date'} eq "") {
            $musicDbRef->{$neurosPath}{'date'} = $addFile_year;
        }

        # Fill in the track number (make it up if it doesn't exist)
        if (!defined $musicDbRef->{$neurosPath}{'tracknumber'} or
            $musicDbRef->{$neurosPath}{'tracknumber'} eq "0" or
            $musicDbRef->{$neurosPath}{'tracknumber'} eq "") {
            # Try to get the track number from the beginning of the filename
            my $filename = basename($file);
            if ($filename =~ /^\d+/) {
                (my $tracknum = $filename) =~ s/^(\d+).*/$1/;
                $musicDbRef->{$neurosPath}{'tracknumber'} = $tracknum;
            } else {
                $addFile_dir = dirname($file);
                if (!defined $addFile_oldDir or
                    $addFile_dir ne $addFile_oldDir) {
                    $addFile_oldDir = $addFile_dir;
                    $addFile_tracknum = 1;
                }
                $musicDbRef->{$neurosPath}{'tracknumber'} = $addFile_tracknum++;
            }
        }
    }

    # Normalize tags
    if ($cfgRef->{'general'}{'normalize'} == 1) {
        normalizeTags($musicDbRef,$neurosPath);
    }

    # Compare tags
    foreach my $field ('album','artist','genre','title') {
        tagCompare($musicDbRef,$neurosPath,$field,$field,
            $cfgRef->{'general'}{"${field}diff"});
    }
    tagCompare($musicDbRef,$neurosPath,'artist2','artist',
        $cfgRef->{'general'}{'artistdiff'});

    return 0;
}

sub removeFromPlaylist($$$$$)
{
    my ($musicDbRef,$playlistDbRef,$playlist,$file,$neurosHome) = @_;

    if (defined $musicDbRef->{$file}{'playlist'}) {
        foreach my $list (@{$musicDbRef->{$file}{'playlist'}}) {
            next if (defined $playlist and $playlist ne $list);
            my @newList = ();
            foreach my $pFile (@{$playlistDbRef->{$list}}) {
                if ($file ne $pFile) {
                    push @newList, $pFile;
                }
            }
            # Must have at least one file in list
            if (scalar @newList) {
                @{$playlistDbRef->{$list}} = @newList;
            } else {
                delete $playlistDbRef->{$list};
                if (-r "$neurosHome/music/$list.m3u") {
                    unlink "$neurosHome/music/$list.m3u";
                }
            }
        }
    }
}

sub delPlaylist($$$$$)
{
    my ($musicDbRef,$playlistDbRef,$neurosHome,$list,$cli) = @_;

    if (!defined $playlistDbRef->{$list}) { return; }

    # Remove playlist association from all files
    foreach my $pFile (@{$playlistDbRef->{$list}}) {
        my @newlist = ();
        # Remove only this playlist
        foreach my $playlist 
            (@{$musicDbRef->{$pFile}{'playlist'}}) {
            if ($playlist ne $list) {
                push @newlist, $playlist;
            }
        }
        # If this is the only playlist for pFile, delete playlist field
        if (scalar @newlist) {
            @{$musicDbRef->{$pFile}{'playlist'}} = @newlist;
        } else {
            delete $musicDbRef->{$pFile}{'playlist'};
        }
    }

    # Query for deletion of associated playlist files
    if ($cli and queryUser
        ("Deleted playlist $list. Delete associated files?")) {
        foreach my $file (@{$playlistDbRef->{$list}}) {
            removeFromPlaylist($musicDbRef,$playlistDbRef,undef,$file,
                $neurosHome);
            $musicDbRef->{$file}{'delete'} = 1;
        }
    }

    # Remove playlist from database and from disk
    delete $playlistDbRef->{$list};
    if (-r "$neurosHome/music/$list.m3u") {
        unlink "$neurosHome/music/$list.m3u";
    }
}

sub randomizePlaylist($$)
{
    my ($playlistDbRef,$list) = @_;

    if (defined $playlistDbRef->{$list}) {
        my @oldlist = @{$playlistDbRef->{$list}};
        my @newlist = ();
        while (my $count = scalar @oldlist) {
            my $num = int(rand $count);
            push @newlist, $oldlist[$num];
            splice @oldlist, $num, 1;
        }
        @{$playlistDbRef->{$list}} = @newlist;
    }
}

sub createPlaylists($$$)
{
    my ($neurosHome,$musicDbRef,$playlistDbRef) = @_;
    my $fh = new FileHandle;

    foreach my $playlist (keys %{$playlistDbRef}) {
        my $outFile = "$neurosHome/music/$playlist.m3u";
        if ($fh->open(">$outFile")) {
            print $fh "#EXTM3U\n";
            if (defined $playlistDbRef->{$playlist}) {
                foreach my $pFile (@{$playlistDbRef->{$playlist}}) {
                    next if (defined $musicDbRef->{$pFile}{'add'} and
                        $musicDbRef->{$pFile}{'add'} eq '1');
                    if (defined $musicDbRef->{$pFile}{'length'} and
                        defined $musicDbRef->{$pFile}{'artist'} and
                        defined $musicDbRef->{$pFile}{'title'}) {
                        print $fh "#EXTINF:",
                            $musicDbRef->{$pFile}{'length'}, ",",
                            $musicDbRef->{$pFile}{'artist'}, " / ",
                            $musicDbRef->{$pFile}{'title'}, "\n";
                    }
                    (my $relativeFile = $pFile) =~ 
                        s/^[CD]:\/music\///i;
                    print $fh "$relativeFile\n";
                }
            }
            $fh->close;
        } else {
            message('ERR',"Could not export playlist $outFile: $!\n");
        }
    }
}

sub list($$$$)
{
    my ($musicDbRef,$recordDbRef,$neurosHome,$type) = @_;
    my @list = ();
    my $found;

    foreach my $file (keys %$musicDbRef) {
        (my $localFile =$file) =~ s/^[CD]:/$neurosHome/i;
        if ($type eq "add") {
            if (defined $musicDbRef->{$file}{'add'} and
                $musicDbRef->{$file}{'add'} eq "1") {
                push @list, $file;
            }
        } elsif ($type eq "delete") {
            if (defined $musicDbRef->{$file}{'delete'} and
                $musicDbRef->{$file}{'delete'} eq "1") {
                push @list, $file;
            }
        } else {
            $found = 0;
            foreach my $item (@list) {
                if ($musicDbRef->{$file}{$type} eq $item) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @list, $musicDbRef->{$file}{$type};
            }
        }
    }

    if ($type eq "delete") {
        foreach my $file (keys %$recordDbRef) {
        (my $localFile =$file) =~ s/^[CD]:/$neurosHome/i;
            if (defined $recordDbRef->{$file}{'delete'} and
                $recordDbRef->{$file}{'delete'} eq "1") {
                push @list, $file;
            }
        }
    }

    return @list;
}

sub parseOgg($$)
{
    my ($file,$cfgRef) = @_;
    my %retHash = ();

    my $ogg = Ogg::Vorbis::Header::PurePerl->new($file);
    if (!defined $ogg) { return \%retHash; }

    if ($cfgRef->{'general'}{'usetags'} ne '0') {
        foreach my $key ("album","artist","date","genre","title",
            "tracknumber") {
            if (defined $ogg->comment($key)) {
                foreach my $tag ($ogg->comment($key)) {
                    fixTag(\$tag);
                    $retHash{$key} = $tag;
                }
            }
        }
    }
    
    if (defined $ogg->info->{'length'}) {
        $retHash{'length'} = int($ogg->info->{'length'});
    }
    return \%retHash;
}

sub parseMp3($$)
{
   my ($file,$cfgRef) = @_;
    my %retHash = ();

    # parse MP3 tags
    if ($cfgRef->{'general'}{'usetags'} ne '0') {
        my $mp3 = MP3::Info::get_mp3tag($file);
        if (defined $mp3) {
            if (defined $mp3->{'ALBUM'}) {
                my $tag = $mp3->{'ALBUM'};
                fixTag(\$tag);
                $retHash{'album'} = $tag;
            }
            if (defined $mp3->{'ARTIST'}) {
                my $tag = $mp3->{'ARTIST'};
                fixTag(\$tag);
                $retHash{'artist'} = $tag;
            }
            if (defined $mp3->{'GENRE'}) {
                my $tag = $mp3->{'GENRE'};
                fixTag(\$tag);
                $retHash{'genre'} = $tag;
            } elsif (defined $mp3->{'COMMENT'}) {
                my $tag = $mp3->{'COMMENT'};
                fixTag(\$tag);
                $retHash{'genre'} = $tag;
            }
            if (defined $mp3->{'TITLE'}) {
                my $tag = $mp3->{'TITLE'};
                fixTag(\$tag);
                $retHash{'title'} = $tag;
            }
            if (defined $mp3->{'TRACKNUM'}) {
                my $tag = $mp3->{'TRACKNUM'};
                fixTag(\$tag);
                $retHash{'tracknumber'} = $tag;
            }
            if (defined $mp3->{'YEAR'}) {
                my $tag = $mp3->{'YEAR'};
                fixTag(\$tag);
                $retHash{'date'} = $tag;
            }
        }
    }
   
    # get MP3 length 
    my $info = MP3::Info::get_mp3info($file);
    if (defined $info and defined $info->{SECS} and int($info->{SECS}) != 0) {
        $retHash{'length'} = int($info->{SECS});
    } else {
        # Try to get length by reading all mpeg frames (slow!!!)
        my $length = 0;
        if (open FILE, $file) {
            my $frame;
            while ($frame = MPEG::Audio::Frame->read(\*FILE)) {
                $length += $frame->seconds();
            }
            close FILE;
        }
        if ($length > 0) { $retHash{'length'} = int($length+1); }
    }
    return \%retHash;
}

sub parseM3u($$$$$$$)
{
    my ($file,$musicDbRef,$playlistDbRef,$cfgRef,$musicHome,$neurosHome,
        $neurosDrive) = @_;
    my $fh = new FileHandle;

    if ($fh->open($file)) {
        my $list = basename($file);
        $list =~ s/\.m3u$//i;
        fixTag(\$list);

        # if playlist already exists, delete it and re-add (it's fast enough)
        if (defined $playlistDbRef->{$list}) {
            delete $playlistDbRef->{$list};
        }

        while (<$fh>) {
            next if (/^#/ or /^\s+$/);
            s/[\r\n]//g;
            my $pFile = File::Spec->rel2abs($_,dirname($file));
            $pFile =~ s/\\/\//g;
            if ($^O eq "MSWin32" and $pFile =~ /^\//) {
                (my $drive = $file) =~ s/^([A-Za-z]):.*/\u$1:/;
                $pFile = "$drive$pFile";
            }
            if (addFile($pFile,$musicHome,$neurosHome,$musicDbRef,
                $playlistDbRef,$cfgRef,$neurosDrive,0) == 0) {
                push @{$playlistDbRef->{$list}},
                    getNeurosFilename($musicHome,$pFile,$neurosDrive);
            } else {
                message('WARN',"$pFile could not be added from $file, skipping.\n");
            }
        }

        if (!defined $playlistDbRef->{$list}) {
            message('ERR', "No files to add from $file, skipping.\n");
            delete $playlistDbRef->{$list};
        }

        $fh->close;
    } else {
        message("ERR","Could not open $file: $!\n");
    }
}

sub parseWma($$)
{
    my ($file,$cfgRef) = @_;
    my %retHash = ();

    my $wma = Audio::WMA->new($file);
    if (!defined $wma) { return \%retHash; }

    if ($cfgRef->{'general'}{'usetags'} ne '0') {
        my $tags = $wma->tags;
        if (defined $tags) {
            if (defined $tags->{'ALBUMTITLE'} and 
                $tags->{'ALBUMTITLE'} ne "0") {
                my $tag = $tags->{'ALBUMTITLE'};
                fixTag(\$tag);
                $retHash{'album'} = $tag;
            }
            if (defined $tags->{'AUTHOR'} and $tags->{'AUTHOR'} ne "0") {
                my $tag = $tags->{'AUTHOR'};
                fixTag(\$tag);
                $retHash{'artist'} = $tag;
            }
            if (defined $tags->{'GENRE'} and $tags->{'GENRE'} ne "0") {
                my $tag = $tags->{'GENRE'};
                fixTag(\$tag);
                $retHash{'genre'} = $tag;
            }
            if (defined $tags->{'TITLE'} and $tags->{'TITLE'} ne "0") {
                my $tag = $tags->{'TITLE'};
                fixTag(\$tag);
                $retHash{'title'} = $tag;
            }
            if (defined $tags->{'TRACKNUMBER'} and
                $tags->{'TRACKNUMBER'} =~ /\d+/) {
                my $tag = $tags->{'TRACKNUMBER'};
                fixTag(\$tag);
                $retHash{'tracknumber'} = $tag;
            }
            if (defined $tags->{'YEAR'} and $tags->{'YEAR'} =~ /\d+/) {
                my $tag = $tags->{'YEAR'};
                fixTag(\$tag);
                $retHash{'date'} = $tag;
            }
        }
    }

    my $info = $wma->info;
    if (defined $info) {
        if (defined $info->{'playtime_seconds'}) {
            $retHash{'length'} = int($info->{'playtime_seconds'});
        }
        if (defined $info->{'drm'} and $info->{'drm'} == 1) {
            message('WARN',"$file is DRM'd, skipping.\n");
        }
    }
    return \%retHash;
}

sub wav_error { }
sub parseWav($$)
{
    my ($file,$cfgRef) = @_;
    my %retHash = ();

    my $wav = new Audio::Wav;
    $wav->set_error_handler(\&wav_error);
    my $read = $wav->read($file);
    if (!defined $read) { return \%retHash; }

    if ($cfgRef->{'general'}{'usetags'} ne '0') {
        my $info = $read->get_info();
        if (defined $info->{'IPRD'}) {
            my $tag = $info->{'IPRD'};
            fixTag(\$tag);
            $retHash{'album'} = $tag;
        }
        if (defined $info->{'IART'}) {
            my $tag = $info->{'IART'};
            fixTag(\$tag);
            $retHash{'artist'} = $tag;
        }
        if (defined $info->{'IGNR'}) {
            my $tag = $info->{'IGNR'};
            fixTag(\$tag);
            $retHash{'genre'} = $tag;
        }
        if (defined $info->{'INAM'}) {
            my $tag = $info->{'INAM'};
            fixTag(\$tag);
            $retHash{'title'} = $tag;
        }
        if (defined $info->{'itrk'}) {
            my $tag = $info->{'itrk'};
            fixTag(\$tag);
            $retHash{'tracknumber'} = $tag;
        }
        if (defined $info->{'ICRD'}) {
            my $tag = $info->{'ICRD'};
            fixTag(\$tag);
            $retHash{'date'} = $tag;
        }
    }

    $retHash{'length'} = int($read->length_seconds());
    return \%retHash;
}

sub fixTag($)
{
    $_ = ${$_[0]};

    # remove utf8 accents
    s/Á/A/og;
    s/Æ/AE/og;
    s/Ç/C/og;
    s/Ð/D/og;
    s/È/E/og;
    s/É/E/og;
    s/Ê/E/og;
    s/Ë/E/og;
    s/Ì/I/og;
    s/Í/I/og;
    s/Î/I/og;
    s/Ï/I/og;
    s/Ñ/N/og;
    s/Ò/O/og;
    s/Ó/O/og;
    s/Ô/O/og;
    s/Õ/O/og;
    s/Ö/O/og;
    s/Ù/U/og;
    s/Ú/U/og;
    s/Û/U/og;
    s/Ü/U/og;
    s/Ý/Y/og;
    s/à/a/og;
    s/á/a/og;
    s/â/a/og;
    s/ã/a/og;
    s/æ/ae/og;
    s/ä/a/og;
    s/å/a/og;
    s/æ/a/og;
    s/ç/c/og;
    s/è/e/og;
    s/é/e/og;
    s/ê/e/og;
    s/ë/e/og;
    s/ì/i/og;
    s/í/i/og;
    s/î/i/og;
    s/ï/i/og;
    s/ĭ/i/og;
    s/ī/i/og;
    s/ñ/n/og;
    s/ò/o/og;
    s/ó/o/og;
    s/ô/o/og;
    s/õ/o/og;
    s/ö/o/og;
    s/ø/o/og;
    s/ţ/t/og;
    s/ù/u/og;
    s/ú/u/og;
    s/û/u/og;
    s/ü/u/og;
    s/ű/u/og;
    s/ý/y/og;
    s/ÿ/y/og;

    # remove latin1 accents
    s//A/og;
    s//A/og;
    s//A/og;
    s//A/og;
    s//Ae/og;
    s//A/og;
    s//AE/og;
    s//C/og;
    s//E/og;
    s//E/og;
    s//E/og;
    s//E/og;
    s//I/og;
    s//I/og;
    s//I/og;
    s//I/og;
    s//ETH/og;
    s//N/og;
    s//O/og;
    s//O/og;
    s//O/og;
    s//O/og;
    s//Oe/og;
    s//O/og;
    s//U/og;
    s//U/og;
    s//U/og;
    s//Ue/og;
    s//Y/og;
    s//THORN/og;
    s//s/og;
    s//a/og;
    s//a/og;
    s//a/og;
    s//a/og;
    s//ae/og;
    s//a/og;
    s//ae/og;
    s//c/og;
    s//e/og;
    s//e/og;
    s//e/og;
    s//e/og;
    s//i/og;
    s//i/og;
    s//i/og;
    s//i/og;
    s//eth/og;
    s//n/og;
    s//o/og;
    s//o/og;
    s//o/og;
    s//o/og;
    s//oe/og;
    s//o/og;
    s//u/og;
    s//u/og;
    s//u/og;
    s//ue/og;
    s//y/og;
    s//thorn/og;
    s//y/og;

    ${$_[0]} = $_;
}

# Relate the audio files to the playlists
sub relate($$)
{
    my ($musicDbRef,$playlistDbRef) = @_;

    foreach my $playlist (keys %$playlistDbRef) {
        foreach my $file (@{$playlistDbRef->{$playlist}}) {
            if (defined $musicDbRef->{$file}) {
                my $found = 0;
                foreach my $list (@{$musicDbRef->{$file}{'playlist'}}) {
                    if ($list eq $playlist) { $found = 1; }
                }
                if (!$found) {
                    push @{$musicDbRef->{$file}{'playlist'}}, $playlist;
                }
            }
        }
    }
}

sub readSoruneDb($$$$)
{
    my ($file,$musicDbRef,$playlistDbRef,$recordDbRef) = @_;
    my ($filename,$key,$value);
    my @lines;

    # remove db entries
    %$musicDbRef = ();
    %$playlistDbRef = ();
    %$recordDbRef = ();

    if ($file =~ /\.gz$/) {
        my $gz = Compress::Zlib::gzopen($file,"rb");
        if ($gz) {
            my $buffer;
            while ($gz->gzread($buffer)) { push @lines, $buffer; }
            $gz->gzclose();
            $buffer = join "", @lines;
            @lines = split /\n/,$buffer;
        } else {
            message('ERR',"Could not open $file: $!\n");
        }
    } else {
        my $fh = new FileHandle;
        if ($fh->open($file)) {
            while(<$fh>) {
                chomp;
                push @lines, $_;
            }
            $fh->close;
        } else {
            message('ERR',"Could not open $file: $!\n");
        }
    }

    while ($_ = shift @lines) {
        if (/\|/) {
            my @parts = split /\|/;
            my $key = shift @parts;
            if ($filename =~ /\.playlist$/i) {
                push @{$playlistDbRef->{$key}}, @parts;
            } elsif ($filename =~ /\.recording$/i) {
                (my $file = $filename) =~ s/\.recording$//;
                $recordDbRef->{$file}{$key} = $parts[0];
            } else {
                if ($key ne "playlist") {
                    $musicDbRef->{$filename}{$key} = $parts[0];
                } else {
                    push @{$musicDbRef->{$filename}{$key}}, @parts;
                }
            }
        } else {
            $filename = $_;
        }
    }
    if (scalar keys %$musicDbRef) {
        message('INFO',"Sorune database loaded.\n"); 
    }
}

sub writeSoruneDb($$$$$)
{
    my ($file,$musicDbRef,$playlistDbRef,$recordDbRef,$cfgRef) = @_;
    my $fh = new FileHandle;
    my $buffer = "";
    my $backups = 1;
    my $dirname = dirname($file);

    # create directory
    eval { mkpath $dirname, 0, 0700; };
    if ($@) {
        message('ERR',"Could not create sorune directory.\n");
        return;
    }

    if (defined $cfgRef->{'general'}{'backups'} and
        $cfgRef->{'general'}{'backups'} =~ /^\d+$/) {
        $backups = $cfgRef->{'general'}{'backups'} - 2;
    }
    for (my $i = $backups; $i >= 0; $i--) {
        if (-e "$file.$i") {
            my $to = $i + 1;
            rename "$file.$i", "$file.$to";
        }
    }
    if (-e $file) {
        if ($backups != 0) { rename $file, "$file.0"; }
        unlink $file;
    }

    foreach my $key (sort keys %$musicDbRef) {
        $buffer .= "$key\n";
        foreach my $ikey (sort keys %{$musicDbRef->{$key}}) {
            if ($ikey ne "playlist") {
                $buffer .= "$ikey|$musicDbRef->{$key}{$ikey}\n";
            } else {
                $buffer .= "$ikey|";
                $buffer .= join "|", sort @{$musicDbRef->{$key}{$ikey}};
                $buffer .= "\n";
            }
        }
    }
    
    foreach my $key (sort keys %$playlistDbRef) {
        $buffer .= "$key.playlist\n";
        $buffer .= "$key|";
        $buffer .= join "|",@{$playlistDbRef->{$key}};
        $buffer .= "\n";
    }

    foreach my $key (sort keys %$recordDbRef) {
        $buffer .= "$key.recording\n";
        foreach my $ikey (sort keys %{$recordDbRef->{$key}}) {
            $buffer .= "$ikey|$recordDbRef->{$key}{$ikey}\n";
        }
    }

    if ($file =~ /\.gz$/) { $buffer = Compress::Zlib::memGzip($buffer); }

    if ($fh->open(">$file")) {
        binmode $fh if ($file =~ /\.gz$/);
        print $fh $buffer;
        $fh->close;
    } else {
        message('ERR',"Could not open $file: $!\n");
    }
}

sub validateSoruneDb
{
    my ($dbRef,$playlistDbRef,$recordDbRef,$cfgRef,$musicHome,$neurosHome,
        $neurosDrive) = @_;
    my @fields = ('album','artist','date','genre','length','size',
        'title','tracknumber');
    my ($dir,$oldDir,$tracknum,$changes) = ("","",1,0);

    if (!-d $neurosHome) {
        return -1;
    }

    foreach my $key (keys %$dbRef) {
        foreach my $field (@fields) {
            if (!defined $dbRef->{$key}{$field} or
                $dbRef->{$key}{$field} eq "" or
                $dbRef->{$key}{$field} eq "0") {
                (my $file = $key) =~ s/^[CD]:/$neurosHome/i;
                if ($field eq "album" or $field eq "artist" or
                    $field eq "genre" or $field eq "title") {
                    $dbRef->{$key}{$field} = "Unknown";
                    $changes++;
                } elsif ($field eq "date") {
                    $dbRef->{$key}{"date"} = $addFile_year;
                    $changes++;
                } elsif ($field eq "size") {
                    if (-r $file) {
                        $dbRef->{$key}{'size'} = (stat($file))[7];
                        $changes++;
                    }
                } elsif ($field eq "tracknumber") {
                    $dir = dirname($file);
                    if ($dir ne $oldDir) {
                        $oldDir = $dir;
                        $tracknum = 1;
                    }
                    $dbRef->{$key}{'tracknumber'} = $tracknum++;
                    $changes++;
                } elsif ($field eq "length") {
                    if ($dbRef == $recordDbRef) {
                        addFile($file,$musicHome,$neurosHome,$dbRef,
                            $playlistDbRef,$cfgRef,$neurosDrive,1);
                    } else {
                        my $neurosRecordHome = "$neurosHome/WOID_RECORD";
                        parseRecording($file,$recordDbRef,$neurosHome,
                            $neurosRecordHome,$cfgRef,$neurosDrive,1);
                    }
                    $changes++;
                }
            }
        }
        if ($dbRef->{$key}{'artist'} eq "Various") {
            if (!defined $dbRef->{$key}{'artist2'} or
                $dbRef->{$key}{'artist2'} eq "Unknown") {
                if ($dbRef->{$key}{'title'} =~ /^(.*)\s*\((.*)\)$/) {
                    $dbRef->{$key}{'artist2'} = $2;
                    $dbRef->{$key}{'title'} = $1;
                    $changes++;
                } else {
                    $dbRef->{$key}{'artist2'} = "Unknown";
                }
            }
        }
    }

    if ($dbRef != $recordDbRef) {
        $changes += validateSoruneDb($recordDbRef,$playlistDbRef,$recordDbRef,
            $cfgRef,$musicHome,$neurosHome,$neurosDrive);
    }

    return $changes;
}

sub setDefine($$$$)
{
    my ($cfgRef,$section,$field,$default) = @_;

    if (defined $cfgRef->{$section}{$field}) {
        $cfgRef->{$section}{$field} =~ s/\\/\//g;
        return "$field = " . $cfgRef->{$section}{$field};
    } else {
        if (defined $default) {
            $default =~ s/\\/\//g;
            $cfgRef->{$section}{$field} = $default;
            return "$field = $default";
        } else {
            return "#$field = ";
        }
    }
}

sub writeSoruneCfg($)
{
    my $cfgRef = shift;
    my ($file,$musichome,$recordingshome,$untaggedformat,$backups,$mediaplayer,
        $font,$geometry,$treewidth,$lastlocation,$usetags,$md5sum,$abjust,
        $normalize,$artistthe,$albumdiff,$artistdiff,$genrediff,$titlediff,
        $duplicates,$version);
    my ($audio,$songs,$playlists,$artists,$albums,$genres,$years,$recordings,
        $artistalbum,$genreartist,$genrealbum,$aasort);
    my ($mainbg,$mainfg,$filebg,$filefg,$filebgsel,$filefgsel,$audiobg,
        $audiofg,$audiobgsel,$audiofgsel,$audiobgdel,$audiobgadd,$statusbarbg,
        $statusbarfg,$progressbg,$progressfg,$menubg,$menufg,$buttonbg,
        $buttonfg,$scrollbg,$scrollfg,$toolbarbg);
    my $fh = new FileHandle;

    if (defined $cfgRef->{'file'}) {
        $file = $cfgRef->{'file'};
    } else {
        return -1;
    }

    # remove trailing slashes from musichome and recordhome
    if (defined $cfgRef->{'general'}{'musichome'}) {
        $cfgRef->{'general'}{'musichome'} =~ s/[\\\/]$//; 
    }
    if (defined $cfgRef->{'general'}{'recordingshome'}) {
        $cfgRef->{'general'}{'recordingshome'} =~ s/[\\\/]$//; 
    }

    $musichome = setDefine($cfgRef,'general','musichome',undef);
    $recordingshome = setDefine($cfgRef,'general','recordingshome',undef);
    $untaggedformat = setDefine($cfgRef,'general','untaggedformat',undef);
    $backups = setDefine($cfgRef,'general','backups','3');
    $mediaplayer = setDefine($cfgRef,'general','mediaplayer',undef);
    $font = setDefine($cfgRef,'general','font','helvetica 10');
    $geometry = setDefine($cfgRef,'general','geometry','640x480+0+0');
    $treewidth = setDefine($cfgRef,'general','treewidth','200');
    $lastlocation = setDefine($cfgRef,'general','lastlocation',undef);
    $usetags = setDefine($cfgRef,'general','usetags','1');
    $normalize = setDefine($cfgRef,'general','normalize','0');
    $duplicates = setDefine($cfgRef,'general','duplicates','0');
    $md5sum = setDefine($cfgRef,'general','md5sum','1');
    $abjust = setDefine($cfgRef,'general','abjust','center');
    $development = setDefine($cfgRef,'general','development','0');
    $albumdiff = setDefine($cfgRef,'general','albumdiff','0');
    $artistdiff = setDefine($cfgRef,'general','artistdiff','0');
    $genrediff = setDefine($cfgRef,'general','genrediff','0');
    $titlediff = setDefine($cfgRef,'general','titlediff','0');
    $version = setDefine($cfgRef,'general','version','0');

    $audio = setDefine($cfgRef,'nam','audio','Neuros Audio');
    $songs = setDefine($cfgRef,'nam','songs','Songs');
    $playlists = setDefine($cfgRef,'nam','playlists','Playlists');
    $artists = setDefine($cfgRef,'nam','artists','Artists');
    $albums = setDefine($cfgRef,'nam','albums','Albums');
    $genres = setDefine($cfgRef,'nam','genres','Genres');
    $years = setDefine($cfgRef,'nam','years','Years');
    $recordings = setDefine($cfgRef,'nam','recordings','Recordings');
    $artistalbum = setDefine($cfgRef,'nam','artistalbum','1');
    $genreartist = setDefine($cfgRef,'nam','genreartist','0');
    $genrealbum = setDefine($cfgRef,'nam','genrealbum','0');
    $aasort = setDefine($cfgRef,'nam','aasort','0');
    $artistthe = setDefine($cfgRef,'nam','artistthe','0');
    

    # Verify valid menuorder
    if (defined $cfgRef->{'nam'}{'menuorder'}) {
        my @list = split /\s*,\s*/, $cfgRef->{'nam'}{'menuorder'};
        my $invalid = 0;
        if (@list == 7) {
            foreach my $item (@list) {
                next if ($item eq 'Songs' or $item eq 'Playlists' or
                    $item eq 'Artists' or $item eq 'Albums' or
                    $item eq 'Genres' or $item eq 'Years' or
                    $item eq 'Recordings');
                $invalid = 1;
                last;
            }
        } else { $invalid = 1; }
        if ($invalid) {
            $cfgRef->{'nam'}{'menuorder'} = 
                'Songs,Playlists,Artists,Albums,Genres,Years,Recordings';
        }
    }
    $menuorder = setDefine($cfgRef,'nam','menuorder',
        'Songs,Playlists,Artists,Albums,Genres,Years,Recordings');

    $mainbg = setDefine($cfgRef,'colors','mainbg','grey93');
    $mainfg = setDefine($cfgRef,'colors','mainfg','black');
    $filebg = setDefine($cfgRef,'colors','filebg','darkred');
    $filefg = setDefine($cfgRef,'colors','filefg','white');
    $filebgsel = setDefine($cfgRef,'colors','filebgsel','grey55');
    $filefgsel = setDefine($cfgRef,'colors','filefgsel','black');
    $audiobg = setDefine($cfgRef,'colors','audiobg','black');
    $audiofg = setDefine($cfgRef,'colors','audiofg','white');
    $audiobgsel = setDefine($cfgRef,'colors','audiobgsel','grey55');
    $audiofgsel = setDefine($cfgRef,'colors','audiofgsel','black');
    $audiobgdel = setDefine($cfgRef,'colors','audiobgdel','darkred');
    $audiobgadd = setDefine($cfgRef,'colors','audiobgadd','darkgreen');
    $statusbarbg = setDefine($cfgRef,'colors','statusbarbg','grey70');
    $statusbarfg = setDefine($cfgRef,'colors','statusbarfg','black');
    $progressbg = setDefine($cfgRef,'colors','progressbg','grey45');
    $progressfg = setDefine($cfgRef,'colors','progressfg','darkblue');
    $menubg = setDefine($cfgRef,'colors','menubg','grey85');
    $menufg = setDefine($cfgRef,'colors','menufg','black');
    $buttonbg = setDefine($cfgRef,'colors','buttonbg','grey85');
    $buttonfg = setDefine($cfgRef,'colors','buttonfg','black');
    $scrollbg = setDefine($cfgRef,'colors','scrollbg','grey85');
    $scrollfg = setDefine($cfgRef,'colors','scrollfg','grey85');
    $toolbarbg = setDefine($cfgRef,'colors','toolbarbg','grey85');

    if ($fh->open(">$file")) {
        print $fh <<EOF;
# Sorune Configuration File
# This file is auto-generated. Changes made to this file may not persist.
# Make sure Sorune is closed before editing this file.

[general]
# The full path to your local music directory. This path is removed from
# the directory structure on the Neuros.
$musichome

# The full path to your local recordings directory. Remove the # sign from
# the beginning of the "recordingshome" line below to enable the moving of
# recordings from the Neuros to this directory.
$recordingshome

# Files without id3 or equivalent tags can be tagged by directory
# structure. This format will be appended to the "musichome" directory
# specified above. Remove the # sign from the beginning of the "Untagged
# Format" line below to enable this feature.
# Available items include:
#   %ALBUM%, %ARTIST%, %GENRE%, %TRACKNUMBER%, %TITLE% and %YEAR%
$untaggedformat

# Number of Sorune backup databases to keep. These are very small if
# the Perl module Compress::Zlib is installed and can be very helpful in
# debugging as well as reverting to a previous state.
$backups

# The Sorune Browser can be configured to launch any multimedia player.
# Note: Sorune will pass an m3u playlist to the player.
$mediaplayer

# The font to use in the GUI.
$font

# Window positioning
$geometry
$treewidth

# The last known location for the Neuros
$lastlocation

# Use ID3 or equivalent tags from the audio files
$usetags

# Normalize tags (initial uppercase all words)
$normalize

# Fix duplicate album entries by appending [artist] or if necessary
# [artist, date]
$duplicates

# Verify files transferred properly to Neuros by md5sum check
$md5sum

# Audio browser justification (center or left)
$abjust

# Enable development options (highly unstable and useless options for
# non-developers) In other words, don't enable this option unless you know
# exactly what it does. Also, be prepared to format your Neuros drive.
$development

# Tag comparison support allows for similar tags to be grouped together.
# See README for more details.
$albumdiff
$artistdiff
$genrediff
$titlediff

# Sorune version
$version

# The Neuros Audio Menu text can be modified by changing these settings.
[nam]
$audio
$songs
$playlists
$artists
$albums
$genres
$years
$recordings
$artistalbum
$genreartist
$genrealbum
$menuorder

# Album under Artist sorting (0 = Alphabetical, 1 = Chronological,
# 2 = Reverse Chronological)
$aasort

# How to handle initial "The" in artist names (0 = leave as is,
# 1 = place at end, 2 = remove)
$artistthe

# GUI color scheme
# Colors can be specified by name (red, blue, darkred, darkblue, etc.) or
# by RGB value (#FF0000, #00FF00, #0000FF, etc.). 
[colors]
$mainbg
$mainfg
$filebg
$filefg
$filebgsel
$filefgsel
$audiobg
$audiofg
$audiobgsel
$audiofgsel
$audiobgdel
$audiobgadd
$statusbarbg
$statusbarfg
$progressbg
$progressfg
$menubg
$menufg
$buttonbg
$buttonfg
$scrollbg
$scrollfg
$toolbarbg
EOF
        $fh->close;
        return 0;
    } else {
        message('WARN',"Could not write sorune configuration!\n");
        return -1;
    }
}

sub readSoruneCfg($$)
{
    my ($file,$cfgRef) = @_;
    my $fh = new FileHandle;
    my ($section,$key,$value) = ("general");
    my ($home,$mediaPlayer) = (getHome(),"");
    my $localrcfile = dirname(File::Spec->rel2abs($0)) . "/sorunerc.cfg";
    %$cfgRef = ();

    # find config file
    if (!defined $file) {
        if (-r $localrcfile) {
            $file = $localrcfile;
        } else {
            $file = "$home/.sorunerc";
        }
        if ($^O eq "MSWin32") {
            $mediaPlayer = "C:/Program Files/Windows Media Player/wmplayer.exe";
        } else {
            $mediaPlayer = "/usr/bin/xmms";
        }
        if (!-r $mediaPlayer) { $mediaPlayer = undef; }
    }

    if (!-e $file) {
        if (-r "$home/music") {
            $cfgRef->{'general'}{'musichome'} = "$home/music";
        }
        $cfgRef->{'general'}{'mediaplayer'} = $mediaPlayer;
        $cfgRef->{'file'} = $file;
        if (writeSoruneCfg($cfgRef) == -1) { return -1; }
    } else {
        $cfgRef->{'file'} = $file;
    }

    if ($fh->open($file)) {
        while(<$fh>) {
            next if (/^#/ or /^\s*$/);
            chomp;
            $_ =~ s/\s*$//;
            if (/^\s*\[(.*)\]\s*$/) {
                $section = $1;
            } else {
                ($key,$value) = split /\s*=\s*/;
                if (defined $value and $value ne "") {
                    $cfgRef->{$section}{lc($key)} = $value;
                }
            }
        }
        $fh->close;
        # make sure all defaults are created by writing the file out
        if (writeSoruneCfg($cfgRef) == -1) { return -1; }
        return 0;
    } else {
        message('WARN',"Could not read sorune configuration!\n");
        return -1;
    }
}

sub validNeurosHome($)
{
    my $path = shift;
    my @files = ("$path/sn.txt","$path/version.txt");
    my @versions = ();
    my $fh = new FileHandle;

    foreach my $file (@files) {
        if ($fh->open($file)) {
            my $version = <$fh>;
            chomp $version;
            $version =~ s/[\s\0]//g;
            push @versions, $version;
            $fh->close;
        } else {
            return ();
        }
    }

    return @versions;
}

sub backupDb($$)
{
    my ($from,$to) = @_;
    my $status = 0;

    if (-d $from) {
        my @findFiles = ();
        find({no_chdir => 1, wanted => sub {
            push @findFiles, $File::Find::name;
        }}, $from);
        
        foreach my $file (sort @findFiles) {
            (my $toFile = $file) =~ s/$from/$to\//;
            if (-d $file) {
                eval { mkpath $toFile, 0, 0700; };
                if ($@) {
                    message('ERR',"Could not create backup directory!\n");
                    return 1;
                }
            } else {
                my $stat = copy($file,$toFile);
                if ($stat != 0) {
                    message('ERR',"Error backing up database file $file!\n");
                    $status = 1;
                }
            }
        }
    }

    if ($status) {
        message('ERR',"Neuros database backup failed.\n");
        return 1;
    } else {
        message('INFO',"Neuros database backup complete.\n");
        return 0;
    }
}

sub restoreDb($$)
{
    my ($from,$to) = @_;
    my $status = 0;

    if (-d $from) {
        my @findFiles = ();
        find({no_chdir => 1, wanted => sub {
            push @findFiles, $File::Find::name;
        }}, $from);
        
        foreach my $file (@findFiles) {
            (my $toFile = $file) =~ s/$from/$to\//;
            if (-d $file) {
                eval { mkpath $toFile, 0, 0700; };
                if ($@) {
                    message('ERR',"Could not create restore directory!\n");
                    return 1;
                }
            } else {
                my $stat = copy($file,$toFile);
                if ($stat != 0) {
                    message('ERR',"Error restoring database file $file!\n");
                    $status = 1;
                }
            }
        }

        # set the status word in audio.mdb to zero to force a menu refresh
        my $db = "$to/audio/audio.mdb";
        if (-r $db) {
            my $size = (stat($db))[7];
            my $fh = new FileHandle;
            if ($fh->open($db)) {
                binmode $fh;
                my $buf;
                read $fh, $buf, $size;
                $fh->close;
                unlink $db;
                wordOverwrite(\$buf,4,0);
                if ($fh->open(">$db")) {
                    binmode $fh;
                    print $fh $buf;
                    $fh->close;
                }
            }
        }
    }

    if ($status) {
        message('ERR',"Neuros database restore failed.\n");
        return 1;
    } else {
        message('INFO',"Neuros database restore complete.\n");
        return 0;
    }
}

sub importDb($$$$$$$$)
{
    my ($file,$musicDbRef,$playlistDbRef,$recordDbRef,$cfgRef,$neurosHome,
        $musicHome,$neurosDrive) = @_;
    my $fh = new FileHandle;
    my $status = 0;

    my %musicDbTmp = ();
    my %recordDbTmp = ();
    my %playlistDbTmp = ();

    if ($fh->open($file)) {
        my $line = 0;
        my $dbRef = undef;
        while (<$fh>) {
            $line++;
            next if (/^#/ or /^\s*$/ or /^NEUROS FILENAME\t/);
            chomp($_);
            my @parts = split /\t/, $_;
            if (scalar @parts != 15) {
                message("ERR","Invalid database, line $line\n");
                $status = 1;
                last;
            }
            
            if ($parts[0] =~ /WOID_RECORD/) {
                $dbRef = \%recordDbTmp;
            } else {
                $dbRef = \%musicDbTmp;
            }

            $dbRef->{$parts[0]}{'localFile'} = $parts[1];
            $dbRef->{$parts[0]}{'album'} = $parts[2];
            $dbRef->{$parts[0]}{'artist'} = $parts[3];
            if ($parts[4] ne "Unknown") {
                $dbRef->{$parts[0]}{'artist2'} = $parts[4];
            }
            $dbRef->{$parts[0]}{'date'} = $parts[5];
            $dbRef->{$parts[0]}{'genre'} = $parts[6];
            $dbRef->{$parts[0]}{'title'} = $parts[7];
            $dbRef->{$parts[0]}{'tracknumber'} = $parts[8];
            $dbRef->{$parts[0]}{'length'} = getSeconds($parts[9]);
            $dbRef->{$parts[0]}{'size'} = $parts[10];
            if ($parts[12] eq '1') {
                $dbRef->{$parts[0]}{'add'} = 1;
            }
            if ($parts[13] eq '1') {
                $dbRef->{$parts[0]}{'delete'} = 1;
            }

            if ($parts[14] ne "None") {
                my @playlists = split /,/, $parts[14];
                @{$dbRef->{$parts[0]}{'playlist'}} = @playlists;
                foreach (@playlists) {
                    push @{$playlistDbTmp{$_}}, $parts[0];
                }
            }
        }
        $fh->close;
        if ($status == 0) {
            $status = validateSoruneDb(\%musicDbTmp,\%playlistDbTmp,
                \%recordDbTmp,$cfgRef,$musicHome,$neurosHome,$neurosDrive);
            if ($status == 0) {
                %$musicDbRef = %musicDbTmp;
                %$recordDbRef = %recordDbTmp;
                %$playlistDbRef = %playlistDbTmp;
            } else {
                message("ERR","Validation of database failed!\n");
                $status = 1;
            }
        }
    } else {
        message("ERR","Could not open file: $!\n");
        $status = 1;
    }
    return $status;
}

sub exportDb($$$)
{
    my ($file,$musicDbRef,$recordDbRef) = @_;
    my $fh = new FileHandle;

    if ($fh->open(">$file")) {
        my @fields = ("album","artist","artist2","date","genre","title",
            "tracknumber","length","size","format","add","delete","playlist");
        my $time;
        print $fh "# Sorune database\n";
        print $fh "# Do not edit the filename fields if you intend to import\n";
        print $fh "# this database back into Sorune later.\n";
        print $fh "NEUROS FILENAME\tLOCAL FILENAME\t";
        foreach my $field (@fields) {
            print $fh uc($field);
            if ($field ne "playlist") { print $fh "\t"; }
        }
        print $fh "\n";
        foreach my $key (sort keys %$musicDbRef) {
            print $fh "$key\t" . $musicDbRef->{$key}{'localFile'} . "\t";
            foreach my $field (@fields) {
                if (defined $musicDbRef->{$key}{$field}) {
                    if ($field eq "playlist") {
                        print $fh join ",", @{$musicDbRef->{$key}{$field}};
                        print $fh "\n";
                    } elsif ($field eq "length") {
                        $time = getTime($musicDbRef->{$key}{'length'});
                        print $fh "$time\t";
                    } else {
                        print $fh "$musicDbRef->{$key}{$field}\t";
                    }
                } else {
                    if ($field eq "playlist") {
                        print $fh "None\n";
                    } elsif ($field eq "format") {
                        (my $format = $key) =~ s/.*\.(.*)$/$1/;
                        print $fh uc($format) . "\t";
                    } elsif ($field eq "add" or $field eq "delete") {
                        print $fh "0\t";
                    } else {
                        print $fh "Unknown\t";
                    }
                }
            }
        }
        pop @fields;
        foreach my $key (sort keys %$recordDbRef) {
            print $fh "$key\t\t";
            foreach my $field (@fields) {
                if (defined $recordDbRef->{$key}{$field}) {
                    if ($field eq "length") {
                        $time = getTime($recordDbRef->{$key}{'length'});
                        print $fh "$time\t";
                    } else {
                        print $fh "$recordDbRef->{$key}{$field}\t";
                    }
                } else {
                    if ($field eq "playlist") {
                        print $fh "None\n";
                    } elsif ($field eq "format") {
                        (my $format = $key) =~ s/.*\.(.*)$/$1/;
                        print $fh uc($format) . "\t";
                    } elsif ($field eq "add" or $field eq "delete") {
                        print $fh "0\t";
                    } else {
                        print $fh "Unknown\t";
                    }
                }
            }
        }
        $fh->close;
        return 0;
    } else {
        message('ERR', "Could not open file: $!\n");
        return 1;
    }
}

sub syncFiles($$$$$)
{
    my ($musicDbRef,$playlistDbRef,$recordDbRef,$cfgRef,$neurosHome) = @_;
    my @addFiles = sort {
        sprintf("%s-%s-%03d",
            $musicDbRef->{$a}{'artist'},
            $musicDbRef->{$a}{'album'},
            $musicDbRef->{$a}{'tracknumber'},
        )
        cmp
        sprintf("%s-%s-%03d",
            $musicDbRef->{$b}{'artist'},
            $musicDbRef->{$b}{'album'},
            $musicDbRef->{$b}{'tracknumber'},
        )
    } (list($musicDbRef,$recordDbRef,$neurosHome,'add'));
    my @delFiles = sort(list($musicDbRef,$recordDbRef,$neurosHome,'delete'));
    my $addFileCount = scalar @addFiles;
    my $delFileCount = scalar @delFiles;
    my ($xferBytesTotal,$xferBytes,$xferElapsed,$xferStart) = (0,0,0,time);

    # Calculate bytes to xfer
    for (my $i = 0; $i < $addFileCount; $i++) {
        $xferBytesTotal += $musicDbRef->{$addFiles[$i]}{'size'};
    }

    # Sync the scheduled additions
    if ($addFileCount) {
        message('INFO',"Transferring files:\n");
        for (my $i = 0; $i < $addFileCount; $i++) {
            my $filename = basename($addFiles[$i]);
            if ($xferBytes > 0 and $xferElapsed > 0) {
                my $strLen = length($filename);
                my $text = sprintf("\r(%s) %s%s", 
                    getTime(($xferBytesTotal - $xferBytes) /
                    ($xferBytes / $xferElapsed)), $filename,
                    " " x (68 - $strLen));
                message('INFO',$text);
            } else {
                my $text = sprintf("\r(estimating) %s",$filename);
                message('INFO',$text);
            }
            syncFile('add',$addFiles[$i],$musicDbRef,$playlistDbRef,
                $recordDbRef,$cfgRef,$neurosHome,undef);
            $xferBytes += $musicDbRef->{$addFiles[$i]}{'size'};
            $xferElapsed = time - $xferStart;
        }
        my $text = sprintf("\r%s\r"," " x 79);
        message('INFO',$text);
    }

    # Sync the scheduled deletions
    if ($delFileCount) {
        for (my $i = 0; $i < $delFileCount; $i++) {
            if ($i % 10 == 0) {
                my $text = sprintf("Deleting files %d%%\r",
                    $i*100/$delFileCount);
                message('INFO',$text);
            }
            syncFile('delete',$delFiles[$i],$musicDbRef,$playlistDbRef,
                $recordDbRef,$cfgRef,$neurosHome,undef);
        }
        message('INFO',"Deleting files 100%\n");
    }

    return $xferBytesTotal;
}

sub syncFile($$$$$$$$)
{
    my ($action,$file,$musicDbRef,$playlistDbRef,$recordDbRef,$cfgRef,
        $neurosHome,$top) = @_;
    (my $neurosFile = $file) =~ s/^[CD]:/$neurosHome/i;

    if ($action eq 'add') {
        my $localFile = $musicDbRef->{$file}{'localFile'};
        if (-r $localFile) {
            if (!-r $neurosFile) {
                my $dirname = dirname($neurosFile);
                eval { mkpath $dirname, 0, 0700; };
                if ($@) {
                    message('ERR',"Could not create directory $dirname!\n");
                    return;
                }
                my @stats = stat($localFile);
                my $stat = copy($localFile,$neurosFile,$top);
                if ($stat == 0) {
                    utime($stats[8],$stats[9],$neurosFile);
                    if (defined $cfgRef->{'general'}{'md5sum'} and
                        $cfgRef->{'general'}{'md5sum'} eq "1") {
                        my ($id, $od);
                        if (open(IFILE,$localFile)) {
                            binmode IFILE;
                            $id = Digest::MD5->new->addfile(
                                *IFILE)->hexdigest;
                            close IFILE;
                        }
                        if (open(OFILE,$neurosFile)) {
                            binmode OFILE;
                            $od = Digest::MD5->new->addfile(
                                *OFILE)->hexdigest;
                            close OFILE;
                        }
                        if ($id ne $od) {
                            message("ERR",
                                "Verification failed for $neurosFile, deleting.\n");
                            unlink $neurosFile;
                            $stat = 1;
                        }
                    }
                    if ($stat == 0) {
                        if (defined $musicDbRef->{$file}{'add'}) {
                            delete $musicDbRef->{$file}{'add'};
                        }
                        if (defined $musicDbRef->{$file}{'delete'}) {
                            delete $musicDbRef->{$file}{'delete'};
                        }
                    }
                }
            }
        }
    } elsif ($action eq 'delete') {
        if (-e $neurosFile) {
            unlink $neurosFile;
            my $dirname = dirname($neurosFile);
            while (rmdir($dirname)) { $dirname =~ s/^(.*)\/[^\/]*$/$1/; }
        }
        removeFromPlaylist($musicDbRef,$playlistDbRef,undef,$file,$neurosHome);
        delete $musicDbRef->{$file};
        delete $recordDbRef->{$file};
    }
}

sub syncRecordings($$$$$$)
{
    my ($recordDbRef,$cfgRef,$neurosHome,$neurosDrive,$top,$haltRef) = @_;
    my $neurosRecordHome = "$neurosHome/WOID_RECORD";
    my @findFiles = ();

    if (-r $neurosRecordHome) {
        if (defined $top) {
            find({no_chdir => 1, wanted => sub {
                if (/\.ogg$/i or /\.mp3$/i or /\.wma$/i or /\.wav$/i or
                    /\.m3u$/i) {
                    push @findFiles, $File::Find::name;
                }
            }, postprocess => sub { $top->update; } }, $neurosRecordHome);
        } else {
            find({no_chdir => 1, wanted => sub {
                if (/\.ogg$/i or /\.mp3$/i or /\.wma$/i or /\.wav$/i or
                    /\.m3u$/i) {
                    push @findFiles, $File::Find::name;
                }
            }}, $neurosRecordHome);
        }

        if (defined $cfgRef->{'general'}{'recordingshome'}) {
            # transfer recordings to recordingshome directory
            my $recordHome = $cfgRef->{'general'}{'recordingshome'};
            eval { mkpath $recordHome, 0, 0700; };
            if (!$@) {
                foreach my $file (@findFiles) {
                    message('INFO',"Moving $file to $recordHome\n");
                    my $stat = copy($file,$recordHome,$top);
                    if (defined $haltRef and $$haltRef) { last; }
                    if ($stat == 0) { unlink $file; }
                    (my $neurosFile = $file) =~ s/$neurosHome/$neurosDrive/;
                    if (defined $recordDbRef->{$neurosFile}) {
                        delete $recordDbRef->{$neurosFile};
                    }
                }
            }
        } else {
            # incorporate recordings into database
            foreach my $file (@findFiles) {
                parseRecording($file,$recordDbRef,$neurosHome,$neurosRecordHome,
                    $cfgRef,$neurosDrive,0);
                if (defined $haltRef and $$haltRef) { last; }
            }

            # remove recordings that no longer exist
            foreach my $file (keys %$recordDbRef) {
                $found = 0;
                foreach my $foundFile (@findFiles) {
                    $foundFile =~ s/$neurosHome/$neurosDrive/;
                    if ($file eq $foundFile) {
                        $found = 1;
                        last;
                    }
                }
                if (!$found) { delete $recordDb{$file}; }
            }
        }
    }

    return scalar @findFiles;
}

sub parseRecording($$$$$$$)
{
    my ($file,$recordDbRef,$neurosHome,$neurosRecordHome,$cfgRef,
        $neurosDrive,$force) = @_;
    my $found = 0;

    if (-r $neurosRecordHome) {
        (my $neurosFile = $file) =~ s/$neurosHome/$neurosDrive/i;
        if (!defined $recordDbRef->{$neurosFile} or $force) {
            # parse mp3/wav recordings
            if ($file =~ /\.mp3$/) {
                $recordDbRef->{$neurosFile} = parseMp3($file,$cfgRef);
            } elsif ($file =~ /\.wav$/) {
                $recordDbRef->{$neurosFile} = parseWav($file,$cfgRef);
            }

            if (!defined $recordDbRef->{$neurosFile}{'length'} or 
                $recordDbRef->{$neurosFile}{'length'} eq "0") {
                message('ERR',"$file is unsupported, skipping.\n");
                if (defined $recordDbRef->{$neurosFile}) {
                    delete $recordDbRef->{$neurosFile};
                }
                return;
            }

            # shrink the recording titles a little
            my $title = "Unknown";
            if ($file =~ /^$neurosRecordHome\/Mic /) {
                ($title = $file) =~ s/^$neurosRecordHome\/Mic (.*)\..*/$1/;
                $recordDbRef->{$neurosFile}{'type'} = "Microphone";
            } elsif ($file =~ /^$neurosRecordHome\/Line /) {
                ($title = $file) =~ s/^$neurosRecordHome\/Line (.*)\..*/$1/;
                $recordDbRef->{$neurosFile}{'type'} = "Line";
            } elsif ($file =~ /^$neurosRecordHome\/FM\d* /) {
                ($title = $file) =~ 
                    s/^$neurosRecordHome\/FM(\d*)(\d) (.*)\..*/$1\.$2 $3/;
                $recordDbRef->{$neurosFile}{'type'} = "FM Radio";
            }

            $recordDbRef->{$neurosFile}{'title'} = $title;
            $recordDbRef->{$neurosFile}{'size'} = (stat($file))[7];
            $recordDbRef->{$neurosFile}{'artist'} = "Unknown";
            $recordDbRef->{$neurosFile}{'album'} = "Unknown";
            $recordDbRef->{$neurosFile}{'genre'} = "Unknown";
            $recordDbRef->{$neurosFile}{'date'} = $addFile_year;
            $recordDbRef->{$neurosFile}{'tracknumber'} = "1";
            $found = 1;
        }
    }
    return $found;
}

sub resetNeuros($$$$$$)
{
    my ($musicDbRef,$playlistDbRef,$recordDbRef,$cfgRef,
        $neurosHome,$fwVersion) = @_;
    my @findFiles = ();
    finddepth({wanted => sub { push @findFiles, $File::Find::name; },
        no_chdir => 1}, $neurosHome);
    my $fileCount = scalar @findFiles;
    for (my $i = 0; $i < $fileCount; $i++) {
        if ($i % 10 == 0) {
            printf("Removing files %d%%\r",$i*100/$fileCount);
        }
        if (-f $findFiles[$i]) {
            if ($findFiles[$i] ne "$neurosHome/bkpk.sn" and
                $findFiles[$i] ne "$neurosHome/sn.txt" and
                $findFiles[$i] ne "$neurosHome/version.txt") {
                unlink $findFiles[$i];
            }
        } else {
            rmdir $findFiles[$i];
        }
    }
    if (scalar @findFiles) {
        message('INFO',"Removing files 100%\n");
    }
    %$musicDbRef = ();
    %$recordDbRef = ();
    %$playlistDbRef = ();
    createNAM($cfgRef,$musicDbRef,$playlistDbRef,$recordDbRef,
        "$neurosHome/woid_db/audio",$fwVersion,undef);
}

sub queryUser($)
{
    my $msg = shift;
    message('INFO', "$msg Continue? [Y/N] ");
    my $response = <STDIN>;
    chomp($response);
    if (lc($response) eq "y" or lc($response) eq "yes") {
        return 1;
    }
    return 0;
}

sub tagByFilename($$$$$)
{
    my ($file,$musicDbRef,$home,$dbKey,$tagFormat) = @_;

    $tagFormat = lc($tagFormat);
    my $fileFormat = $tagFormat;
    $fileFormat =~ s/\%album\%/\(\.*\)/;
    $fileFormat =~ s/\%artist\%/\(\.*\)/;
    $fileFormat =~ s/\%genre\%/\(\.*\)/;
    $fileFormat =~ s/\%title\%/\(\.*\)/;
    $fileFormat =~ s/\%tracknumber\%/\(\[\\d\]*\)/;
    $fileFormat =~ s/\%year\%/\(\.*\)/;
    (my $relFile = $file) =~ s/$home\/(.*)\.[^\.]*$/$1/i;

    # if file matches format, extract info
    if ($relFile =~ /^$fileFormat$/) {
        my @fileParts = ($1,$2,$3,$4,$5,$6);

        my $albumIndex = index $tagFormat, "\%album\%";
        my $artistIndex = index $tagFormat, "\%artist\%";
        my $genreIndex = index $tagFormat, "\%genre\%";
        my $titleIndex = index $tagFormat, "\%title\%";
        my $trackIndex = index $tagFormat, "\%tracknumber\%";
        my $dateIndex = index $tagFormat, "\%year\%";

        my @tagOrder = ();
        my $tagFormatLength = length($tagFormat);
        for (my $i = 0; $i < $tagFormatLength; $i++) {
            if ($albumIndex == $i) {
                push @tagOrder, 'album';
            } elsif ($artistIndex == $i) {
                push @tagOrder, 'artist';
            } elsif ($genreIndex == $i) {
                push @tagOrder, 'genre';
            } elsif ($titleIndex == $i) {
                push @tagOrder, 'title';
            } elsif ($trackIndex == $i) {
                push @tagOrder, 'tracknumber';
            } elsif ($dateIndex  == $i) {
                push @tagOrder, 'date';
            }
        }

        if ($albumIndex == -1) { push @tagOrder, 'album'; } 
        if ($artistIndex == -1) { push @tagOrder, 'artist'; }
        if ($genreIndex == -1) { push @tagOrder, 'genre'; }
        if ($titleIndex == -1) { push @tagOrder, 'title'; }
        if ($trackIndex == -1) { push @tagOrder, 'tracknumber'; }
        if ($dateIndex  == -1) { push @tagOrder, 'date'; }

        foreach my $filePart (@fileParts) {
            if (defined $filePart) {
                if ($filePart eq "") { $filePart = "Unknown"; }
                my $key = shift @tagOrder;
                if (!defined $musicDbRef->{$dbKey}{$key} or
                    $musicDbRef->{$dbKey}{$key} eq "") {
                    $musicDbRef->{$dbKey}{$key} = $filePart;
                }
            }
        }
    }
}

sub getWindowsDrives()
{
    my $drives;
    my @drives = ();

    Win32API::File::GetLogicalDriveStrings(4*26+1,$drives);
    foreach (split /\0/, uc($drives)) {
        # ignore floppy drives
        if (/A:/ or /B:/) { next; }

        # ignore any non-drive devices (e.g. USB printers)
        my ($osVolName,$ouSerialNum,$ouMaxNameLen,$osFsFlags,$osFsType);
        Win32API::File::GetVolumeInformation($_,$osVolName,32,
            $ouSerialNum,$ouMaxNameLen,$osFsFlags,$osFsType,32);
        if ($ouMaxNameLen == 0) { next; }

        if (Win32API::File::GetDriveType($_) == DRIVE_FIXED() or
            Win32API::File::GetDriveType($_) == DRIVE_REMOTE() or
            Win32API::File::GetDriveType($_) == DRIVE_REMOVABLE()) {
            $_ =~ s/\\//;
            push @drives, $_;
        }
    }

    return @drives;
}

sub locateNeuros($)
{
    my ($lastLocation) = @_;
    my @versionInfo = ();

    # check last known location
    if (defined $lastLocation) {
        @versionInfo = validNeurosHome($lastLocation);
        if (scalar @versionInfo) {
            unshift @versionInfo, $lastLocation;
            return @versionInfo;
        }
    }

    # scan for Neuros
    if ($^O eq "MSWin32") {
        foreach my $drive (getWindowsDrives()) {
            @versionInfo = validNeurosHome($drive);
            if (scalar @versionInfo) {
                unshift @versionInfo, $drive;
                last;
            }
        }
    } else {
        my $fh = new FileHandle;
        if ($fh->open("mount|")) {
            while (<$fh>) {
                # only scan vfat/msdos filesystems that aren't floppies
                # FreeBSD uses "msdos" to describe a fat32 filesystem -
                # dave@jetcafe.org
                if ((!/vfat/o and !/msdos/o) or /\/dev\/fd/o) { next; }
                my @parts = split / /;
                @versionInfo = validNeurosHome($parts[2]);
                if (scalar @versionInfo) {
                    unshift @versionInfo, $parts[2];
                    last;
                }
            }
            $fh->close;
        }
    }
    return @versionInfo;
}

sub getMusicSize($$)
{
    my ($musicDbRef,$recordDbRef) = @_;
    my $size = 0;
    foreach my $key (keys %$musicDbRef) {
        if (!defined $musicDbRef->{$key}{'add'} and
            defined $musicDbRef->{$key}{'size'}) {
            $size += $musicDbRef->{$key}{'size'};
        }
    }
    foreach my $key (keys %$recordDbRef) {
        if (!defined $recordDbRef->{$key}{'add'} and
            defined $recordDbRef->{$key}{'size'}) {
            $size += $recordDbRef->{$key}{'size'};
        }
    }
    return $size;
}

sub getHome()
{
    if (!defined $ENV{'HOME'}) {
        if (defined $ENV{'USERPROFILE'}) {
            $ENV{'HOME'} = $ENV{'USERPROFILE'};
        } else {
            $ENV{'HOME'} = File::Spec->rel2abs(dirname($0));
        }
    }
    return $ENV{'HOME'};
}

sub getSize($)
{
    my $size = shift;
    my @units = ("B","KB","MB","GB");
    my $i;
    for ($i = 0; $size > 1000; $i++) { $size /= 1000; }
    return sprintf("%0.2f %s",$size,$units[$i]);
}

sub getSeconds($)
{
    my $time = shift;

    if (!defined $time) { return 0; }
    my @parts = split /:/, $time;

    if (scalar(@parts) == 1) {
        return $parts[0];
    } elsif (scalar(@parts) == 2) {
        return (($parts[0] * 60) + $parts[1]);
    } elsif (scalar(@parts) == 3) {
        return (($parts[0] * 3600) + ($parts[1] * 60) + $parts[2]);
    }
}

sub getTime($)
{
    my $secs = shift;
    my ($mins,$hrs) = (0,0);

    if (!defined $secs) { return "00:00"; }

    if ($secs >= 3600) {
        $hrs = int($secs/3600);
        $secs %= 3600;
    }
    if ($secs >= 60) {
        $mins = int($secs/60);
        $secs %= 60;
    }
    if ($hrs > 0) {
        return sprintf("%d:%02d:%02d",$hrs,$mins,$secs);
    } else {
        return sprintf("%02d:%02d",$mins,$secs);
    }
}

sub getNeurosFilename($$$)
{
    my ($home,$file,$neurosDrive) = ($_[0],decodeFilename($_[1]),$_[2]);

    $home =~ s/\\/\//g;
    $file =~ s/\\/\//g;

    if ($^O ne "MSWin32") {
        # NOTE: This renaming could cause duplicate problems if all characters
        # of a given directory/filename are invalid. This seems very unlikely.
        $file =~ tr/?<>:*^|"/_/;

        # Neuros is unhappy with chars with a unicode scalar value of 0x0100
        # or higher. Having control characters in the filename probably
        # isn't a good idea, either.
        $file = join('', map {
            ($_ < 0x20 || $_ > 0xFF) ? sprintf("{U+%04X}", $_) : chr($_);
        } unpack("U*", $file));

        # vfat gets unhappy when things end with periods. Causes duplicated
        # file names, at least under Linux. Space is documented to be a bad
        # idea, too.
        # rsync just removes these trailing chars, so we'll do that too
        $file =~ s/[. ]$//;
        $file =~ s/[. ]\//\//;

        # OK, we're done with the fun; go back to external encoding.
        $file = encodeFilename($file);
    }

    # Remove the music home directory from the neuros location
    if (defined $home and -d $home and $file =~ /^$home/i) {
        $file =~ s/^$home/$neurosDrive\/music/i;
        return $file;
    }

    if ($^O eq "MSWin32") {
        $file =~ s/^[A-Z]:/$neurosDrive\/music/i;
    } else {
        $file = "$neurosDrive\/music$file";
    }
    return $file;
}

sub copy($$$)
{
    my ($ifile,$ofile,$top) = @_;

    if (-d $ofile) {
        my $ibase = basename($ifile);
        $ofile =~ s/\/$//;
        $ofile .= "/$ibase";
    }
    if ($ifile eq $ofile) { return 0; }
    if (-e $ofile) { unlink $ofile; }

    my $odir = dirname($ofile);
    if (!-d $odir) {
        eval { mkpath $odir, 0, 0700; };
        if ($@) {
            message('ERR', "Could not create directory $odir: $!\n");
            return 1;
        }
    }

    my $chunksize = 1024 * 64;
    my $ifh = new FileHandle;
    my $filesize = (stat($ifile))[7];
    if ($ifh->open($ifile)) {
        my $ofh = new FileHandle;
        binmode $ifh;

        while ($filesize > 0) {
            if ($ofh->open(">$ofile")) {
                my $buf;
                binmode $ofh;

                while ($filesize) {
                    if ($filesize < $chunksize) { $chunksize = $filesize; }
                    my $rlen = sysread($ifh, $buf, $chunksize);
                    if (!defined $rlen) {
                        message('ERR', "Error reading $ifile!\n");
                        $ifh->close;
                        $ofh->close;
                        unlink $ofile;
                        return 1;
                    }
                    my $offset = 0;
                    while ($rlen) {
                        my $wlen = syswrite($ofh, $buf, $chunksize, $offset);
                        if (!defined $wlen) {
                            message('ERR', "Error writing $ifile!\n");
                            $ifh->close;
                            $ofh->close;
                            unlink $ofile;
                            return 1;
                        }
                        $offset += $wlen;
                        $rlen -= $wlen;
                    }
                    $filesize -= $chunksize;
                    if (defined $top) { eval { $top->update; }; }
                }
                $ofh->close;
            } else {
                message('ERR', "Could not open $ofile: $!\n");
                return 1;
            }
        }
        $ifh->close;
    } else {
        message('ERR', "Could not open $ifile: $!\n");
        return 1;
    }

    return 0;
}

# Ugly way to fix a bug in Perl 5.8.4+ having to do with extended ascii chars
# in filenames being converted to utf8 in perl/tk.
sub ascii($)
{
    my $ascii = shift;
    if (!-d $ascii and !-f $ascii and Encode::is_utf8($ascii)) {
        if (my $utf8 = Encode::encode_utf8($ascii)) {
            Encode::from_to($utf8,'utf8','iso-8859-1');
            # conversion made no difference, return original string
            if (!-d $utf8 and !-f $utf8) { return $ascii; }
            # conversion worked, return converted string
            return $utf8;
        }
    }
    return $ascii
}

sub searchPath($)
{
    my $file = shift;
    foreach (split /:/,$ENV{'PATH'}) {
        if (-x "$_/$file") {
            return "$_/$file";
        }
    }
    return undef;
}

sub findDuplicates($$$$)
{
    my ($musicDbRef,$neurosHome,$neurosDrive,$fieldRef) = @_;
    my @files = sort keys %$musicDbRef;
    my $fileCount = scalar @files;
    my @dups = ();
    my $found;

    for (my $i = 0; $i < $fileCount; $i++) {
        for (my $j = $i+1; $j < $fileCount; $j++) {
            $found = 1;
            foreach my $field (@$fieldRef) {
                if ($musicDbRef->{$files[$i]}{$field} ne 
                    $musicDbRef->{$files[$j]}{$field}) {
                    $found = 0;
                    last;
                }
            }
            if ($found) {
                my ($foundi,$foundj) = (0,0);
                foreach my $file (@dups) {
                    if ($file eq $files[$i]) { $foundi = 1; }
                    if ($file eq $files[$j]) { $foundj = 1; }
                }
                if (!$foundi) { push @dups, $files[$i]; }
                if (!$foundj) { push @dups, $files[$j]; }
            }
        }
    }

    return @dups;
}

sub decodeFilename($)
{
    if ($^O ne 'MSWin32') {
        my $encoded_name = shift;
        my $encoding = langinfo(CODESET);
        my $decoded = _eval {
              return decode($encoding, $encoded_name, Encode::FB_CROAK);
        };
        if (!defined($decoded)) {
              message('WARN',
                  "File name $encoded_name failed to decode. Wrong locale?");
              return decode($encoding, $encoded_name, Encode::FB_DEFAULT);
        } else {
              return $decoded;
        }
    }
    return $_[0];
}

sub encodeFilename($)
{
    if ($^O ne 'MSWin32') {
        my $decoded_name = shift;
        my $encoding = langinfo(CODESET);
        return encode($encoding, $decoded_name, Encode::FB_CROAK);
    }
    return $_[0];
}

sub normalizeTags($$)
{
    my ($ref,$key) = @_;
    $ref->{$key}{'album'} =~ s/([^\s]+)/\u$1/g;
    $ref->{$key}{'album'} =~ s/_/ /g;
    $ref->{$key}{'artist'} =~ s/([^\s]+)/\u$1/g;
    $ref->{$key}{'artist'} =~ s/_/ /g;
    if (defined $ref->{$key}{'artist2'}) {
        $ref->{$key}{'artist2'} =~ s/([^\s]+)/\u$1/g;
        $ref->{$key}{'artist2'} =~ s/_/ /g;
    }
    $ref->{$key}{'genre'} =~ s/([^\s]+)/\u$1/g;
    $ref->{$key}{'genre'} =~ s/_/ /g;
    $ref->{$key}{'title'} =~ s/([^\s]+)/\u$1/g;
    $ref->{$key}{'title'} =~ s/_/ /g;
}

sub getFreeSpace($)
{
    my $path = shift;
    my $size = 0;

    if (!defined $path or !-r $path) { return 0; }

    if ($^O eq "MSWin32") {
        $path =~ s/:.*/:/;
        if ($path !~ /^[A-Z]:$/i) {
            message('ERR',"Could not determine free space for $path\n");
            return 0;
        }
        $size = (Win32::DriveInfo::DriveSpace($path))[6];
    } else {
        if (open(H,"df -k $path|")) {
            # getting last line instead of line w/$path to allow for fake setup
            my $lastLine = "";
            while (<H>) {
                chomp;
                ($lastLine = $_) =~ s/^\s+//;
            }
            my @parts = split /\s+/, $lastLine;
            if ($parts[0] =~ /\//) {
                $size = $parts[3] * 1024;
            } else {
                $size = $parts[2] * 1024;
            }
            close H;
        }
    }

    return $size;
}

sub getXferSize($$)
{
    my ($musicDbRef,$recordDbRef) = @_;
    my $size = 0;

    foreach my $key (keys %$musicDbRef) {
        if (defined $musicDbRef->{$key}{'add'} and
            $musicDbRef->{$key}{'add'} == 1) {
            $size += $musicDbRef->{$key}{'size'};
        }
        if (defined $musicDbRef->{$key}{'delete'} and
            $musicDbRef->{$key}{'delete'} == 1) {
            $size -= $musicDbRef->{$key}{'size'};
        }
    }

    foreach my $key (keys %$recordDbRef) {
        if (defined $recordDbRef->{$key}{'add'} and
            $recordDbRef->{$key}{'add'} == 1) {
            $size += $recordDbRef->{$key}{'size'};
        }
        if (defined $recordDbRef->{$key}{'delete'} and
            $recordDbRef->{$key}{'delete'} == 1) {
            $size -= $recordDbRef->{$key}{'size'};
        }
    }

    return $size;
}

# Return all the keys that match a particular dbKey to a uniqueKey
# Ex: If dbKey = genre and uniqueKey = Rock, return all Rock keys
sub getFileData($$$)
{
    my ($musicDbRef,$dbKey,$uniqueKey) = @_;
    my @keys = ();

    foreach my $key (keys %$musicDbRef) {
        if (defined $musicDbRef->{$key}{$dbKey} and
            $musicDbRef->{$key}{$dbKey} eq $uniqueKey) {
            push @keys, $key;
        }
    }

    return @keys;
}

# Return all the unique dbKey2 keys that match a particular dbKey to a uniqueKey
# Ex: If dbKey = artist and uniqueKey = Alanis and dbKey2 = album,
# return unique Alanis album keys
sub getUniqueFileData($$$$)
{
    my ($musicDbRef,$dbKey,$uniqueKey,$dbKey2) = @_;
    my @uniqueKeys = ();

    foreach my $key (keys %$musicDbRef) {
        my $found = 0;
        if (defined $musicDbRef->{$key}{$dbKey} and
            $musicDbRef->{$key}{$dbKey} eq $uniqueKey) {
            foreach my $uniqueKey2 (@uniqueKeys) {
                if ($musicDbRef->{$key}{$dbKey2} eq
                    $musicDbRef->{$uniqueKey2}{$dbKey2}) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) { 
                push @uniqueKeys, $key;
            }
        }
    }

    return @uniqueKeys;
}

# Return unique names for given dbKey
# Ex: If dbKey = genre, return a unique genre list
sub getUniqueNames($$)
{
    my ($musicDbRef,$dbKey) = @_;
    if ($dbKey eq "") { return 0; }

    my @uniqueNames = ();
    my $found;
    foreach my $key (keys %$musicDbRef) {
        $found = 0;
        foreach my $uniqueName (@uniqueNames) {
            if (defined $musicDbRef->{$key}{$dbKey} and
                $musicDbRef->{$key}{$dbKey} eq $uniqueName) {
                $found = 1;
                last;
            }
        }
        if (!$found and defined $musicDbRef->{$key}{$dbKey}) {
            push @uniqueNames, $musicDbRef->{$key}{$dbKey};
        }
    }

    return @uniqueNames;
}

my %tagCompareInfo = ();
sub tagCompare($$$$$)
{
    my ($musicDbRef,$neurosPath,$key,$dbKey,$diff) = @_;

    if ($diff <= 0) { return; }

    my @names = ();
    my $found = 0;
    if (defined $tagCompareInfo{$dbKey}) {
        @names = @{$tagCompareInfo{$dbKey}};
    } else {
        @names = getUniqueNames($musicDbRef,$dbKey);
        @{$tagCompareInfo{$dbKey}} = @names;
    }

    foreach my $name (@names) {
        if (defined $musicDbRef->{$neurosPath}{$key} and
            stridiff($name,$musicDbRef->{$neurosPath}{$key}) <= $diff) {
            $musicDbRef->{$neurosPath}{$key} = $name;
            $found = 1;
            last;
        }
    }

    if (!$found) {
        push @{$tagCompareInfo{$dbKey}}, $musicDbRef->{$neurosPath}{$key};
    }
}

# Implemented using the Levenshtein Distance Algorithm
# http://www.merriampark.com/ld.htm
sub stridiff($$)
{
    my ($s,$t) = (lc($_[0]),lc($_[1]));
    if ($s eq $t) { return 0; }
    
    my ($n,$m) = (length($s),length($t));
    if ($n == 0) { return $m; }
    if ($m == 0) { return $n; }

    my @d = ();
    my $cost;

    for (my $i = 0; $i <= $n; $i++) { $d[$i][0] = $i; }
    for (my $j = 0; $j <= $m; $j++) { $d[0][$j] = $j; }

    for (my $i = 1; $i <= $n; $i++) {
        my $s_i = substr $s, $i-1, 1;
        for (my $j = 1; $j <= $m; $j++) {
            my $t_j = substr $t, $j-1, 1;
            if ($s_i eq $t_j) {
                $cost = 0;
            } else {
                $cost = 1;
            }
            $d[$i][$j] = min3($d[$i-1][$j]+1, $d[$i][$j-1]+1,
                $d[$i-1][$j-1] + $cost);
        }
    }
    return $d[$n][$m];
}

sub min3($$$)
{
    return $_[0] < $_[1] ?
        $_[0] < $_[2] ? $_[0] : $_[2] :
        $_[1] < $_[2] ? $_[1] : $_[2];
}

sub usage()
{
    my $prog = basename($0);
    print STDERR <<EOF;

Usage: $prog <Command>

Commands:
--add <directory|file>          Add directory or file to Neuros
--backup                        Backup the Neuros database
--gui                           Start the Sorune GUI (default)
--clear <all|add|delete>        Clears all/add/delete/database operations
--delete <Delete Option>        Delete items in the database matching Option
--duplicates                    Prints duplicates (based on artist/title)
--export <filename>             Exports the Sorune database to <filename>
--import <filename>             Imports the Sorune database from <filename>
--file <filename>               Use config file <filename>
--help                          This help screen
--info                          Print Neuros information and exit
--list <List Option>            List items in the database matching Option
--new                           Clear existing database and create new
--restore                       Restore the Neuros backup database
--rebuild                       Rebuild the Sorune database
--rebuild_full                  Clear and then Rebuild the Sorune database
--reset                         Removes all files from the Neuros
--sync                          Synchronize the Sorune database to the Neuros
--validate                      Validate and correct Sorune database
--version                       Version information

Delete Options:
  Album <Album Name>            Deletes song(s) by Album
  Artist <Artist Name>          Deletes song(s) by Artist
  Genre <Genre Name>            Deletes song(s) by Genre
  Playlist <Playlist Name>      Deletes playlist
  Title <Title Name>            Deletes song(s) by Title

List Options:
  Albums                        List all Albums
  Artists                       List all Artists
  Genres                        List all Genres
  Playlists                     List all Playlists
  Titles                        List all Titles
  Add                           List all Files to be added
  Delete                        List all Files to be deleted

Examples:
  Add the directories "Artist 1" and "Artist 2":
    $prog --add "Artist 1" --add "Artist 2"
  List all Artists and Genres:
    $prog --list Artists --list Genres
  Delete Title "Title 1" and Artist "Artist 3":
    $prog --delete "Title Title 1" --delete "Artist Artist 3"
  Synchronize all $prog database additions/deletions to Neuros:
    $prog --sync
  Rebuild the Sorune database and synchronize to Neuros:
    $prog --rebuild --sync

EOF
}

sub version()
{
    my $version = getVersion();
    print STDERR <<EOF;
Sorune $version
Copyright 2004-2005, Darren Smith
All Rights Reserved.
EOF
}

sub getVersion()
{
    return "0.5";
}

1
