///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting/Scripting.h>
#include <mesh/tri/TriMesh.h>
#include "ContainerWrappers.h"

namespace Scripting {

using namespace boost::python;

template<class Container>
class TriMeshVertices : public def_visitor< TriMeshVertices<Container> >
{
public:
    typedef typename Container::value_type data_type;
    typedef typename Container::value_type key_type;
    typedef typename Container::size_type index_type;
    typedef typename Container::size_type size_type;
    typedef typename Container::difference_type difference_type;

    typedef return_internal_reference<> return_policy;
    typedef boost::python::iterator<Container, return_policy> def_iterator;

    template<class Class>
    void visit(Class& cl) const {
        cl
            .def("__len__", &get_size)
            .def("__setitem__", &set_item)
            .def("__delitem__", &delete_item)
            .def("__getitem__", &get_item_wrapper)
            .def("__contains__", &contains)
            .def("__iter__", def_iterator())
        ;
    }

    static size_t get_size(Container& container) { return container.size(); }

    static data_type& get_item(Container& container, index_type i) {
        return container[i];
    }

    template <class DataType>
    static object get_item_helper(DataType const& p, mpl::true_) {
        return object(ptr(p));
    }

    template <class DataType>
    static object get_item_helper(DataType const& x, mpl::false_) {
        return object(x);
    }

    static object get_item_wrapper(back_reference<Container&> container, PyObject* i) {
        if(PySlice_Check(i)) {
        	PyErr_SetString(PyExc_NotImplementedError, "This sequence type does not support slicing.");
        	throw_error_already_set();
        }
        return object(get_item(container.get(), convert_index(container.get(), i)));
    }

    static void set_item(Container& container, PyObject* i, PyObject* v) {
		PyErr_SetString(PyExc_NotImplementedError, "This sequence type is read-only.");
		throw_error_already_set();
    }

	static void delete_item(Container& container, PyObject* i) {
		PyErr_SetString(PyExc_NotImplementedError, "This sequence type is read-only.");
		throw_error_already_set();
    }

    static bool contains(Container& container, key_type const& key) {
		return container.contains(key);
    }

    static index_type convert_index(Container& container, PyObject* i_) {
        extract<long> i(i_);
        if(i.check()) {
            long index = i();
            if (index < 0)
                index += get_size(container);
            if (index >= long(container.size()) || index < 0) {
                PyErr_SetString(PyExc_IndexError, "Index out of range");
                throw_error_already_set();
            }
            return index;
        }

        PyErr_SetString(PyExc_TypeError, "Invalid index type");
        throw_error_already_set();
        return index_type();
    }
};


void ExportTriMesh()
{
	class_<QVector<Point3>, noncopyable>("Point3Vector", no_init)
		.def(QVector_indexing_suite< QVector<Point3>, return_internal_reference<> >())
	;

	// DataChannel types
	enum_<TriMeshFace::MeshFaceFlags>("MeshFaceFlags")
    	.value("NONE", TriMeshFace::NONE)
    	.value("EDGE1", TriMeshFace::EDGE1)
    	.value("EDGE2", TriMeshFace::EDGE2)
    	.value("EDGE3", TriMeshFace::EDGE3)
    	.value("EDGES12", TriMeshFace::EDGES12)
    	.value("EDGES23", TriMeshFace::EDGES23)
    	.value("EDGES13", TriMeshFace::EDGES13)
    	.value("EDGES123", TriMeshFace::EDGES123)
    ;

	class_<TriMeshFace>("TriMeshFace", init<>())
		.def("SetVertices", &TriMeshFace::setVertices)
		.def("SetVertex", &TriMeshFace::setVertex)
		.def("Vertex", &TriMeshFace::vertex)
		.def("SetEdgeVisibility", (void (TriMeshFace::*)(bool,bool,bool))&TriMeshFace::setEdgeVisibility)
		.def("EdgeVisibility", &TriMeshFace::edgeVisibility)
		.add_property("MaterialIndex", &TriMeshFace::materialIndex, &TriMeshFace::setMaterialIndex)
		.add_property("SmoothingGroup", &TriMeshFace::smoothingGroup, &TriMeshFace::setSmoothingGroup)
	;

	class_<QVector<TriMeshFace>, noncopyable>("TriMeshFaceVector", no_init)
		.def(QVector_indexing_suite< QVector<TriMeshFace>, return_internal_reference<> >())
	;

	class_<TriMesh>("TriMesh", init<>())
		.def("ClearMesh", &TriMesh::clearMesh)
		.add_property("BoundingBox", make_function(&TriMesh::boundingBox, return_value_policy<copy_const_reference>()))
		.add_property("VertexCount", &TriMesh::vertexCount, &TriMesh::setVertexCount)
		.add_property("Vertices", make_function((QVector<Point3>& (TriMesh::*)())&TriMesh::vertices, return_internal_reference<>()))
		.def("Vertex", make_function((const Point3& (TriMesh::*)(int) const)&TriMesh::vertex, return_value_policy<copy_const_reference>()))
		.def("SetVertex", &TriMesh::setVertex)
		.def("InvalidateVertices", &TriMesh::invalidateVertices)
		.add_property("FaceCount", &TriMesh::faceCount, &TriMesh::setFaceCount)
		.add_property("Faces", make_function((QVector<TriMeshFace>& (TriMesh::*)())&TriMesh::faces, return_internal_reference<>()))
		.def("Face", make_function((TriMeshFace& (TriMesh::*)(int))&TriMesh::face, return_internal_reference<>()))
		.def("AddFace", make_function(&TriMesh::addFace, return_internal_reference<>()))
		.def("InvalidateFaces", &TriMesh::invalidateFaces)
	;
}

};
