/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "testlib/loggingsink.h"

// Boost C++
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/regex.hpp>

// C++ Standard Library
#include <map>
#include <sstream>
#include <string>
#include <utility>
#include <vector>

namespace mediascanner {

typedef boost::function<bool(const std::string &,
                             const std::string &)> MessageAcceptor;

class CapturingMessageSink : public logging::DefaultMessageSink {
public:
    CapturingMessageSink(logging::MessageSinkPtr parent,
                         MessageAcceptor acceptor,
                         std::ostream *stream)
        : DefaultMessageSink(stream)
        , parent_(parent)
        , acceptor_(acceptor) {
    }

    void Report(const std::string &domain_name, const std::string &message) {
        if (acceptor_(domain_name, message))
            DefaultMessageSink::Report(domain_name, message);

        if (parent_)
            parent_->Report(domain_name, message);
    }

private:
    logging::MessageSinkPtr parent_;
    MessageAcceptor acceptor_;
};

class LoggingSink::Private {
public:
    typedef std::map<logging::Domain *, logging::Domain> DomainSettingsMap;
    typedef std::vector<std::pair<std::string, boost::regex> > IgnorePatterns;

    explicit Private(logging::Domain *level)
        : initial_settings_(collect_domain_settings(level)) {
        for (logging::Domain *ld = level; ld; ld = ld->parent()) {
            const logging::MessageSinkPtr capturing_message_sink
                    (make_message_sink(ld->message_sink()));
            ld->set_message_sink(capturing_message_sink);
        }
    }

    ~Private() {
        for (const auto &p: initial_settings_) {
            p.first->set_message_sink(p.second.message_sink());
            p.first->set_enabled(p.second.enabled());
        }
    }

    CapturingMessageSink* make_message_sink(logging::MessageSinkPtr parent) {
        return new CapturingMessageSink
                (parent, boost::bind(&Private::accept_message, this, _1, _2),
                 &stream_);
    }

    bool accept_message(const std::string &domain_name,
                        const std::string &message) {
        for (const auto &p: ignore_patterns_) {
            if (p.first == domain_name
                    && boost::regex_match(message, p.second))
                return false;
        }

        return true;
    }

    static DomainSettingsMap collect_domain_settings(logging::Domain *level) {
        DomainSettingsMap settings;

        for (logging::Domain *domain = level; domain; domain = domain->parent())
            settings.insert(std::make_pair(domain, *domain));

        return settings;
    }

    const DomainSettingsMap initial_settings_;
    std::stringstream stream_;
    IgnorePatterns ignore_patterns_;
};

LoggingSink::LoggingSink(logging::Domain *level)
    : d(new Private(level)) {
}

LoggingSink::~LoggingSink() {
    delete d;
}

void LoggingSink::reset() {
    d->stream_.str(std::string());
}

std::string LoggingSink::get() const {
    return d->stream_.str();
}

std::string LoggingSink::take() {
    const std::string str = get();
    return reset(), str;
}

void LoggingSink::ignore(const std::string &domain,
                         const std::string &pattern) {
    d->ignore_patterns_.push_back
            (std::make_pair(domain, boost::regex(pattern)));
}


} // namespace mediascanner
