/*
 *  $Id: wrapvalue.c 22340 2019-07-25 10:23:59Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti)
 *  E-mail: yeti@gwyddion.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/stats.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "preview.h"

#define WRAP_VALUE_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

typedef enum {
    WRAP_VALUE_RANGE_USER,
    WRAP_VALUE_RANGE_360_DEG,
    WRAP_VALUE_RANGE_180_DEG,
    WRAP_VALUE_RANGE_2PI,
    WRAP_VALUE_RANGE_PI,
    WRAP_VALUE_RANGE_1,
    WRAP_VALUE_RANGE_NTYPES
} WrapValueRangeType;

typedef struct {
    WrapValueRangeType type;
    gdouble range;
    gdouble offset;
} WrapValueArgs;

typedef struct {
    WrapValueArgs *args;
    GtkWidget *dialog;
    GtkWidget *view;
    GSList *type;
    GtkWidget *range;
    GtkObject *offset;
    GwyContainer *mydata;
    GwyDataField *dfield;
    GwySIValueFormat *format;
    gdouble curr_range;
    gboolean in_update;
} WrapValueControls;

static gboolean module_register         (void);
static void     wrap_value              (GwyContainer *data,
                                         GwyRunType run);
static void     run_noninteractive      (const WrapValueArgs *args,
                                         GwyContainer *data,
                                         GwyDataField *dfield,
                                         GQuark quark);
static void     wrap_value_dialog       (WrapValueArgs *args,
                                         GwyContainer *data,
                                         GwyDataField *dfield,
                                         gint id,
                                         GQuark quark);
static void     type_changed            (GtkWidget *button,
                                         WrapValueControls *controls);
static void     range_changed           (WrapValueControls *controls);
static void     offset_changed          (GtkAdjustment *adj,
                                         WrapValueControls *controls);
static void     update_sensitivity      (WrapValueControls *controls);
static void     preview                 (WrapValueControls *controls);
static void     wrap_value_do           (const WrapValueArgs *args,
                                         gdouble range,
                                         GwyDataField *dfield);
static void     wrap_value_load_args    (GwyContainer *settings,
                                         WrapValueArgs *args);
static void     wrap_value_save_args    (GwyContainer *settings,
                                         WrapValueArgs *args);
static void     wrap_value_sanitize_args(WrapValueArgs *args);


static const gdouble wrap_value_ranges[WRAP_VALUE_RANGE_NTYPES] = {
    0.0,    /* unused */
    360.0,
    180.0,
    2.0*G_PI,
    G_PI,
    1.0,
};

static const WrapValueArgs wrap_value_defaults = {
    WRAP_VALUE_RANGE_USER,
    1.0, 0.5,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Wraps periodic values to a different range."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, wrapvalue)

static gboolean
module_register(void)
{
    gwy_process_func_register("wrapvalue",
                              (GwyProcessFunc)&wrap_value,
                              N_("/_Basic Operations/_Wrap Value..."),
                              GWY_STOCK_WRAP_VALUE,
                              WRAP_VALUE_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Rewrap periodic values"));

    return TRUE;
}

static void
wrap_value(GwyContainer *data, GwyRunType run)
{
    WrapValueArgs args;
    GwyDataField *dfield;
    GQuark quark;
    gint id;

    g_return_if_fail(run & WRAP_VALUE_RUN_MODES);
    wrap_value_load_args(gwy_app_settings_get(), &args);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_DATA_FIELD_KEY, &quark,
                                     0);
    g_return_if_fail(dfield);

    if (run == GWY_RUN_IMMEDIATE) {
        run_noninteractive(&args, data, dfield, quark);
        gwy_app_channel_log_add_proc(data, id, id);
    }
    else
        wrap_value_dialog(&args, data, dfield, id, quark);
}

static void
run_noninteractive(const WrapValueArgs *args,
                   GwyContainer *data,
                   GwyDataField *dfield,
                   GQuark quark)
{
    gdouble range;

    if (args->type == WRAP_VALUE_RANGE_USER)
        range = args->range;
    else
        range = wrap_value_ranges[args->type];

    gwy_app_undo_qcheckpointv(data, 1, &quark);
    wrap_value_do(args, range, dfield);
}

static void
wrap_value_format_value(WrapValueControls *controls,
                        GtkEntry *entry,
                        gdouble value)
{
    gchar *s;

    s = g_strdup_printf("%.*f",
                        controls->format->precision+1,
                        value/controls->format->magnitude);
    gtk_entry_set_text(GTK_ENTRY(entry), s);
    g_free(s);
}

static void
wrap_value_dialog(WrapValueArgs *args,
                  GwyContainer *data,
                  GwyDataField *dfield,
                  gint id,
                  GQuark quark)
{
    GtkWidget *dialog, *hbox, *label, *entry;
    GtkTable *table;
    WrapValueControls controls;
    gint response, row;
    gdouble min, max, r;
    gchar *s;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.dfield = dfield;
    controls.in_update = FALSE;

    /* If remembered range differs wildly from data range, try to guess
     * something of reasonable order of magnitude... */
    gwy_data_field_get_min_max(dfield, &min, &max);
    r = max - min;
    if (!(fabs(log(r/args->range)) < 2.0)) {
        if (r > 0.0)
            args->range = r;
        else
            args->range = 1.0;
    }

    s = gwy_si_unit_get_string(gwy_data_field_get_si_unit_z(dfield),
                               GWY_SI_UNIT_FORMAT_PLAIN);
    controls.format = gwy_data_field_get_value_format_z
                                   (dfield, GWY_SI_UNIT_FORMAT_VFMARKUP, NULL);

    dialog = gtk_dialog_new_with_buttons(_("Wrap Value"), NULL, 0,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 2);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    dfield = gwy_data_field_duplicate(controls.dfield);
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    g_object_unref(dfield);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), controls.view, FALSE, FALSE, 4);

    if (gwy_strequal(s, "deg")) {
        if (args->type != WRAP_VALUE_RANGE_360_DEG
            && args->type != WRAP_VALUE_RANGE_180_DEG)
            args->type = WRAP_VALUE_RANGE_USER;
        controls.type
            = gwy_radio_buttons_createl(G_CALLBACK(type_changed),
                                        &controls, args->type,
                                        _("360 deg"), WRAP_VALUE_RANGE_360_DEG,
                                        _("180 deg"), WRAP_VALUE_RANGE_180_DEG,
                                        _("Specify _range"),
                                        WRAP_VALUE_RANGE_USER,
                                        NULL);
    }
    else if (gwy_strequal(s, "")) {
        if (args->type != WRAP_VALUE_RANGE_2PI
            && args->type != WRAP_VALUE_RANGE_PI
            && args->type != WRAP_VALUE_RANGE_1)
            args->type = WRAP_VALUE_RANGE_USER;
        controls.type
            = gwy_radio_buttons_createl(G_CALLBACK(type_changed),
                                        &controls, args->type,
                                        _("2π"), WRAP_VALUE_RANGE_2PI,
                                        _("π"), WRAP_VALUE_RANGE_PI,
                                        _("1"), WRAP_VALUE_RANGE_1,
                                        _("Specify _range"),
                                        WRAP_VALUE_RANGE_USER,
                                        NULL);
    }
    else {
        args->type = WRAP_VALUE_RANGE_USER;
        controls.type = NULL;
    }
    g_free(s);

    table = GTK_TABLE(gtk_table_new(4 + g_slist_length(controls.type), 4,
                                    FALSE));
    gtk_table_set_row_spacings(table, 2);
    gtk_table_set_col_spacings(table, 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(table), TRUE, TRUE, 4);
    row = 0;

    label = gtk_label_new(_("Data range:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    if (controls.type)
        row = gwy_radio_buttons_attach_to_table(controls.type, table, 3, row);

    controls.range = entry = gtk_entry_new();
    gwy_widget_set_activate_on_unfocus(entry, TRUE);
    wrap_value_format_value(&controls, GTK_ENTRY(entry), args->range);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row,
                            _("_Range:"), controls.format->units,
                            GTK_OBJECT(entry), GWY_HSCALE_WIDGET_NO_EXPAND);
    g_signal_connect_swapped(controls.range, "activate",
                             G_CALLBACK(range_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.offset = gtk_adjustment_new(0.5, 0.0, 1.0, 0.01, 1.0, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row,
                            _("O_ffset:"), controls.format->units,
                            controls.offset, GWY_HSCALE_DEFAULT);
    g_signal_connect(controls.offset, "value-changed",
                     G_CALLBACK(offset_changed), &controls);
    row++;

    type_changed(NULL, &controls);
    update_sensitivity(&controls);
    preview(&controls);

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            g_object_unref(controls.mydata);
            gwy_si_unit_value_format_free(controls.format);
            wrap_value_save_args(gwy_app_settings_get(), args);
            return;
            break;

            case GTK_RESPONSE_OK:
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    wrap_value_save_args(gwy_app_settings_get(), args);

    gtk_widget_destroy(dialog);
    g_object_unref(controls.mydata);
    gwy_si_unit_value_format_free(controls.format);
    run_noninteractive(args, data, controls.dfield, quark);
    gwy_app_channel_log_add_proc(data, id, id);
}

static void
update_offset_range(WrapValueControls *controls,
                       gdouble range)
{
    gdouble offsetval, step;

    controls->curr_range = range;
    offsetval = controls->args->offset*range;
    if (range > 0.0)
        step = 0.001*range;
    else
        step = 0.001;

    controls->in_update = TRUE;
    g_object_set(controls->offset,
                 "lower", -range,
                 "upper", range,
                 "value", offsetval,
                 "step-increment", step,
                 "page-increment", 10.0*step,
                 NULL);
    controls->in_update = FALSE;
}

static void
type_changed(G_GNUC_UNUSED GtkWidget *button,
             WrapValueControls *controls)
{
    WrapValueArgs *args = controls->args;
    gdouble range;

    /* We can be called even when there is just a single range type. */
    if (controls->type)
        args->type = gwy_radio_buttons_get_current(controls->type);
    else
        args->type = WRAP_VALUE_RANGE_USER;

    if (args->type == WRAP_VALUE_RANGE_USER)
        range = args->range;
    else
        range = wrap_value_ranges[args->type];

    update_offset_range(controls, range);
    update_sensitivity(controls);
    preview(controls);
}

static void
update_sensitivity(WrapValueControls *controls)
{
    WrapValueArgs *args = controls->args;
    gboolean sens;

    sens = (args->type == WRAP_VALUE_RANGE_USER);
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->range), sens);
}

static void
range_changed(WrapValueControls *controls)
{
    const gchar *value = gtk_entry_get_text(GTK_ENTRY(controls->range));
    gdouble range = g_strtod(value, NULL);

    if (!(range > 0.0))
        range = controls->args->range;
    else
        range *= controls->format->magnitude;

    wrap_value_format_value(controls, GTK_ENTRY(controls->range), range);
    update_offset_range(controls, range);
    controls->args->range = range;
    preview(controls);
}

static void
offset_changed(GtkAdjustment *adj, WrapValueControls *controls)
{
    WrapValueArgs *args = controls->args;

    args->offset = gtk_adjustment_get_value(adj);
    if (controls->curr_range > 0.0) {
        args->offset /= controls->curr_range;
        args->offset = CLAMP(args->offset, -1.0, 1.0);
    }
    else
        args->offset = 0.5;

    if (controls->in_update)
        return;

    preview(controls);
}

static void
preview(WrapValueControls *controls)
{
    GwyDataField *dfield;

    if (controls->in_update)
        return;

    dfield = GWY_DATA_FIELD(gwy_container_get_object_by_name(controls->mydata,
                                                             "/0/data"));
    gwy_data_field_copy(controls->dfield, dfield, FALSE);
    wrap_value_do(controls->args, controls->curr_range, dfield);
}

static void
wrap_value_do(const WrapValueArgs *args, gdouble range, GwyDataField *dfield)
{
    gdouble offsetval;
    gdouble *d;
    gint n, i;

    if (!(range > 0.0)) {
        gwy_data_field_clear(dfield);
        gwy_data_field_data_changed(dfield);
        return;
    }

    offsetval = args->offset*range;
    n = gwy_data_field_get_xres(dfield)*gwy_data_field_get_yres(dfield);
    d = gwy_data_field_get_data(dfield);
#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            private(i) \
            shared(d,n,offsetval,range)
#endif
    for (i = 0; i < n; i++) {
        d[i] = fmod(d[i] - offsetval, range);
        if (d[i] < 0.0)
            d[i] += range;
        d[i] += offsetval;
    }

    gwy_data_field_data_changed(dfield);
}

static const gchar offset_key[] = "/module/wrapvalue/offset";
static const gchar range_key[]  = "/module/wrapvalue/range";
static const gchar type_key[]   = "/module/wrapvalue/type";

static void
wrap_value_sanitize_args(WrapValueArgs *args)
{
    args->type = MIN(args->type, WRAP_VALUE_RANGE_NTYPES-1);
    args->range = MAX(args->range, G_MINDOUBLE);
    args->offset = CLAMP(args->offset, -1.0, 1.0);
}

static void
wrap_value_load_args(GwyContainer *settings, WrapValueArgs *args)
{
    *args = wrap_value_defaults;

    gwy_container_gis_enum_by_name(settings, type_key, &args->type);
    gwy_container_gis_double_by_name(settings, range_key, &args->range);
    gwy_container_gis_double_by_name(settings, offset_key, &args->offset);
    wrap_value_sanitize_args(args);
}

static void
wrap_value_save_args(GwyContainer *settings, WrapValueArgs *args)
{
    gwy_container_set_enum_by_name(settings, type_key, args->type);
    gwy_container_set_double_by_name(settings, range_key, args->range);
    gwy_container_set_double_by_name(settings, offset_key, args->offset);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
