from gpaw.test import equal
from ase import Atoms
from ase.units import Pascal, m
from ase.data.vdw import vdw_radii
from ase.parallel import rank
from gpaw.solvation import (
    SolvationGPAW,
    EffectivePotentialCavity,
    Power12Potential,
    LinearDielectric,
    KB51Volume,
    GradientSurface,
    VolumeInteraction,
    SurfaceInteraction,
    LeakedDensityInteraction)

import numpy as np

SKIP_ENERGY_CALCULATION = True
F_max_err = 0.01

h = 0.2
u0 = 0.180
epsinf = 80.
T = 298.15
atomic_radii = lambda atoms: [vdw_radii[n] for n in atoms.numbers]

atoms = Atoms('NaCl', positions=((5.6, 5.6, 6.8), (5.6, 5.6, 8.8)))
atoms.set_cell((11.2, 11.2, 14.4))


atoms.calc = SolvationGPAW(
    xc='PBE', h=h, setups={'Na': '1'},
    cavity=EffectivePotentialCavity(
        effective_potential=Power12Potential(atomic_radii, u0),
        temperature=T,
        volume_calculator=KB51Volume(),
        surface_calculator=GradientSurface()),
    dielectric=LinearDielectric(epsinf=epsinf),
    # parameters chosen to give ~ 1eV for each interaction
    interactions=[
        VolumeInteraction(pressure=-1e9 * Pascal),
        SurfaceInteraction(surface_tension=100. * 1e-3 * Pascal * m),
        LeakedDensityInteraction(voltage=10.)
    ]
)


def vac(atoms):
    return min(
        atoms.positions[0][2],
        14.4 - atoms.positions[1][2]
    )

step = .05
if not SKIP_ENERGY_CALCULATION:
    d = []
    E = []
    F = []
    while vac(atoms) >= 5.6:
        d.append(abs(atoms.positions[0][2] - atoms.positions[1][2]))
        E.append(atoms.calc.get_potential_energy(atoms, force_consistent=True))
        F.append(atoms.calc.get_forces(atoms))
        atoms.positions[0][2] -= step

    d = np.array(d)
    E = np.array(E)
    F = np.array(F)

    if rank == 0:
        np.save('d.npy', d)
        np.save('E.npy', E)
        np.save('F.npy', F)
        from pprint import pprint
        print('d')
        pprint(list(d))
        print()
        print('E')
        pprint(list(E))
        print()
        print('F')
        pprint([list([list(l2) for l2 in l1]) for l1 in F])
else:
    # h=0.2, setups: 0.9.11271
    d = [
        2.0000000000000009,
        2.0500000000000007,
        2.1000000000000005,
        2.1500000000000004,
        2.2000000000000002,
        2.25,
        2.2999999999999998,
        2.3499999999999996,
        2.3999999999999995,
        2.4499999999999993,
        2.4999999999999991,
        2.5499999999999989,
        2.5999999999999988,
        2.6499999999999986,
        2.6999999999999984,
        2.7499999999999982,
        2.799999999999998,
        2.8499999999999979,
        2.8999999999999977,
        2.9499999999999975,
        2.9999999999999973,
        3.0499999999999972,
        3.099999999999997,
        3.1499999999999968,
        3.1999999999999966
    ]

    E = [
        -3.562275095496517,
        -3.8473139097013753,
        -4.0700170404316216,
        -4.2411280906015847,
        -4.3696230053988998,
        -4.4631058774925938,
        -4.5278419352994277,
        -4.5689533655049273,
        -4.5909074791427535,
        -4.5973242956988001,
        -4.5911955771799642,
        -4.5750005750743217,
        -4.5508272583082476,
        -4.5204998569534682,
        -4.4853536276501051,
        -4.446576387818217,
        -4.4051492970094319,
        -4.3619637256674846,
        -4.3175923355729191,
        -4.2725810250114824,
        -4.2273623203404105,
        -4.1823745422113907,
        -4.1378435049679414,
        -4.0939327053888253,
        -4.0509012515430012
    ]

    F = [
        [[-6.8649267934919038e-12, -6.8285488255163915e-12, -6.4013896254399594],
         [-2.6300729419344384e-10, -2.6383434154270953e-10, 6.4024961288370834]],
        [[-6.7067934676926484e-12, -6.7827710411030432e-12, -5.035806576943255],
         [-2.6265003631465652e-10, -2.6358273430409411e-10, 5.0363219577200393]],
        [[-6.6452522782472319e-12, -6.7350652481746713e-12, -3.9024047426211013],
         [-2.599941690725449e-10, -2.6427151466339217e-10, 3.9041962651946549]],
        [[-6.6678787127007239e-12, -6.7240667463184095e-12, -2.9653099919755648],
         [-2.6301398163741478e-10, -2.6378537086737464e-10, 2.9677010135504456]],
        [[-6.7321644245704045e-12, -6.7023538748183105e-12, -2.1941739883216349],
         [-2.6323640189803985e-10, -2.6502975875270664e-10, 2.1956072044985082]],
        [[-6.6744556469137517e-12, -6.7342571167849082e-12, -1.5603845691279461],
         [-2.6150037805716431e-10, -2.6494203401997128e-10, 1.5617242524244506]],
        [[-6.6740115916431383e-12, -6.7067692706129593e-12, -1.0394646874434972],
         [-2.657058970775486e-10, -2.6486463588998541e-10, 1.0411305520365366]],
        [[-6.6978937133664486e-12, -6.7559118414194535e-12, -0.6139233210960644],
         [-2.6177850433079895e-10, -2.6417622580023846e-10, 0.61622349293177647]],
        [[-6.6211500466746826e-12, -6.6910274310038701e-12, -0.26943737156761505],
         [-2.6257887811625044e-10, -2.6408833568511208e-10, 0.27225705595193372]],
        [[-6.6850010873431177e-12, -6.7442884174800231e-12, 0.0089081295524152775],
         [-2.6124062311671723e-10, -2.6382235955634513e-10, -0.0076515998408599566]],
        [[-6.7390588871677577e-12, -6.7034381182837725e-12, 0.23393233473731392],
         [-2.6379960167644507e-10, -2.6460665237784466e-10, -0.23162254793656473]],
        [[-6.7478611985265237e-12, -6.7270688826163367e-12, 0.41344114500688628],
         [-2.6307072030825134e-10, -2.6499582985996902e-10, -0.41090256849768725]],
        [[-6.8089864119330214e-12, -6.7665673210298178e-12, 0.55336685925410833],
         [-2.6654711238076438e-10, -2.6500969566112445e-10, -0.5513277496947222]],
        [[-6.724371042066641e-12, -6.8046040813353111e-12, 0.66191904264557422],
         [-2.6690771291888572e-10, -2.6553224854951662e-10, -0.66116131240092779]],
        [[-6.8278815770331904e-12, -6.8044619806595946e-12, 0.74595045809588023],
         [-2.5978727053971925e-10, -2.6560989627796548e-10, -0.74446609663056418]],
        [[-6.7783638939157041e-12, -6.8133286336204628e-12, 0.80852902621501777],
         [-2.6244275687456016e-10, -2.6512375201569421e-10, -0.806217471185949]],
        [[-6.8446384858991989e-12, -6.8432091478293385e-12, 0.85168825894226086],
         [-2.6407844819045985e-10, -2.649684928296529e-10, -0.85040178985847581]],
        [[-6.7066299199700351e-12, -6.8496098228765492e-12, 0.88042067677910474],
         [-2.6392062693018424e-10, -2.6369080594848279e-10, -0.87999233344200489]],
        [[-6.8489522669943444e-12, -6.8984993318559013e-12, 0.89869231350992573],
         [-2.6765853049315407e-10, -2.652701718256956e-10, -0.8971875269274393]],
        [[-6.8160721183558111e-12, -6.8960086592369432e-12, 0.90729473426351359],
         [-2.632560888207301e-10, -2.6406685134015365e-10, -0.90541522611106229]],
        [[-6.986299841419749e-12, -6.9133592118879747e-12, 0.90622371816323821],
         [-2.618130831940452e-10, -2.6463957320603059e-10, -0.90650772392397816]],
        [[-6.9613071942774566e-12, -6.9395424290466483e-12, 0.89897847913208095],
         [-2.6166471205863267e-10, -2.6505752863392422e-10, -0.89825350436859608]],
        [[-6.9015705487362154e-12, -6.9199753382689915e-12, 0.88817599560445226],
         [-2.6640069318482646e-10, -2.656425323100925e-10, -0.88699092835462157]],
        [[-6.8099622202150577e-12, -6.9738365001184978e-12, 0.87343340469998276],
         [-2.6503495062870906e-10, -2.6377908478652655e-10, -0.87163079404060828]],
        [[-6.9867580450747381e-12, -7.0042141182343876e-12, 0.85369504926144513],
         [-2.6442899833727647e-10, -2.6415524503832887e-10, -0.85271756460899106]]
    ]
    d = np.array(d)
    E = np.array(E)
    F = np.array(F)


# test for orthogonal forces equal zero:
equal(F[..., :2], .0, 1e-7)

stencil = 2  # 1 is too rough, 3 does not change compared to 2
FNa, FCl = F[..., 2].T
FNa *= -1.
# test symmetry
equal(FNa, FCl, F_max_err)
dd = np.diff(d)[0]
kernel = {
    1: np.array((0.5, 0, -0.5)),
    2: np.array((-1. / 12., 2. / 3., 0, -2. / 3., 1. / 12.)),
    3: np.array((1. / 60., -0.15, 0.75, 0, -0.75, 0.15, -1. / 60.)),
}

dEdz = np.convolve(E, kernel[stencil] / step, 'valid')

err = np.maximum(
    np.abs(-dEdz - FNa[stencil:-stencil]),
    np.abs(-dEdz - FCl[stencil:-stencil])
)

# test forces against -dE / dd finite difference
equal(err, .0, F_max_err)

if SKIP_ENERGY_CALCULATION:
    # check only selected points:

    def check(index):
        atoms.positions[0][2] = 6.8 - index * step
        F_check = atoms.get_forces()
        equal(F_check[..., :2], .0, 1e-7)
        FNa_check, FCl_check = F_check[..., 2].T
        FNa_check *= -1.
        equal(FNa_check, FCl_check, F_max_err)
        err = np.maximum(
            np.abs(-dEdz[index - stencil] - FNa_check),
            np.abs(-dEdz[index - stencil] - FCl_check)
        )
        equal(err, .0, F_max_err)

    l = len(FNa)
    # check(stencil)
    check(l // 2)
    # check(l - 1 - stencil)
