/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gnome-scan
 * Copyright (C) Étienne Bersac 2007 <bersace03@laposte.net>
 * 
 * gnome-scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

/**
 * SECTION: gnome-scan-module-manager
 * @short_description: Finding installed modules
 *
 * The #GnomeScanModuleManager search in a list of path shared object,
 * and instanciate a #GnomeScanModule for each shared object. Upon
 * initialization, the #GnomeScanModule will register new GTypes. This
 * implementation is meant to be portable, but never tested on other
 * plateform that GNU/Linux.
 **/

#include "gnome-scan-module-manager.h"
#include "gnome-scan-module.h"

enum {
	PROP_0,
	PROP_PATH
};


static GObjectClass* parent_class = NULL;

static gboolean	gsmm_is_valid_module_name (const gchar*name);

G_DEFINE_TYPE (GnomeScanModuleManager, gnome_scan_module_manager, G_TYPE_OBJECT);

static void
gnome_scan_module_manager_init (GnomeScanModuleManager *object)
{
	/* TODO: Add initialization code here */
}

#if 0
static GObject*
gnome_scan_module_manager_constructor (GType type,
									   guint n_params,
									   GObjectConstructParam *params)
{
	GnomeScanModuleManager *manager;
	GObject *object;
	
	object =
		G_OBJECT_CLASS (gnome_scan_module_manager_parent_class)->constructor (type,
																			  n_params,
																			  params);
	manager = GNOME_SCAN_MODULE_MANAGER (object);
	
	if (manager->path)
		gnome_scan_module_manager_query_modules (manager);
	
	return object;
}
#endif							   

static void
gnome_scan_module_manager_finalize (GObject *object)
{
	/* TODO: Add deinitalization code here */

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_module_manager_set_property (GObject *object,
										guint prop_id,
										const GValue *value,
										GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_MODULE_MANAGER (object));

	switch (prop_id)
		{
		case PROP_PATH:
			GNOME_SCAN_MODULE_MANAGER (object)->path = g_value_dup_string (value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
		}
}

static void
gnome_scan_module_manager_get_property (GObject *object,
										guint prop_id,
										GValue *value,
										GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_MODULE_MANAGER (object));
	
	switch (prop_id)
		{
		case PROP_PATH:
			g_value_set_string (value, GNOME_SCAN_MODULE_MANAGER (object)->path);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
		}
}

static void
gnome_scan_module_manager_class_init (GnomeScanModuleManagerClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));

	/*object_class->constructor	= gnome_scan_module_manager_constructor;*/
	object_class->finalize = gnome_scan_module_manager_finalize;
	object_class->set_property = gnome_scan_module_manager_set_property;
	object_class->get_property = gnome_scan_module_manager_get_property;

	/**
	 * GnomeScanModuleManager:path:
	 *
	 * The search path where are installed #GnomeScanModule shared
	 * object. A search path is a string formed by a list of absolute
	 * path to directory separated by #G_SEARCHPATH_SEPARATOR.
	 **/
	g_object_class_install_property (object_class,
	                                 PROP_PATH,
	                                 g_param_spec_string ("path",
	                                                      "Path",
	                                                      "Module path",
	                                                      NULL,
	                                                      G_PARAM_READABLE | G_PARAM_WRITABLE));
}

/**
 * gnome_scan_module_manager_new:
 * @path: Search path.
 * 
 * Create a new #GnomeScanModuleManager which will handle @path.
 * 
 * Returns: a new #GnomeScanModuleManager
 **/
GnomeScanModuleManager*
gnome_scan_module_manager_new (gchar *path)
{
	return g_object_new (GNOME_TYPE_SCAN_MODULE_MANAGER,
						 "path", path,
						 NULL);
}

/**
 * gnome_scan_module_manager_query_modules:
 * @manager: a #GnomeScanModuleManager
 * 
 * Search for shared objects in path, instanciate and initialize a
 * #GnomeScanModule for each shared object. Note that it won't search
 * in subdirectories.
 **/
void
gnome_scan_module_manager_query_modules (GnomeScanModuleManager *manager)
{
	GnomeScanModule *module;
	GDir *dir;
	GError *error = NULL;
	gchar **paths = g_strsplit (manager->path, G_SEARCHPATH_SEPARATOR_S, -1);
	const gchar *name;
	gchar *filename;
	guint i;
	
	for (i = 0; paths[i]; i++) {
		dir = g_dir_open (paths[i], 0, &error);
		if (!dir) {
			/* Show warning only for absolute path, else should be devel path… */
			if (paths[i][0] == '/')
				g_warning ("%s: Error while searching for modules: %s",
						   G_STRLOC, error->message);
			g_clear_error (&error);
			continue;
		}
		
		/* Scan for lib*.{so,la} */
		while ((name = g_dir_read_name (dir))) {
			if (gsmm_is_valid_module_name (name)) {
				filename = g_build_filename (paths[i], name, NULL);
				module = gnome_scan_module_new (filename);

				if (!g_type_module_use (G_TYPE_MODULE (module))) {
					g_warning (g_module_error ());
				}
				manager->modules = g_slist_append (manager->modules, module);
			}
		}
	}
}

/**
 * gnome_scan_module_manager_unload_modules:
 * @manager: a #GnomeScanModuleManager
 * 
 * Search for shared objects in path, instanciate and initialize a
 * #GnomeScanModule for each shared object. Note that it won't search
 * in subdirectories.
 **/
void
gnome_scan_module_manager_unload_modules (GnomeScanModuleManager *manager)
{
	GSList *node;
	
	for (node = manager->modules; node ; node = node->next) {
		g_type_module_unuse (G_TYPE_MODULE (node->data));
	}
}

static gboolean
gsmm_is_valid_module_name (const gchar*name)
{
	return g_str_has_prefix (name, "lib")
		&& (g_str_has_suffix (name, G_MODULE_SUFFIX)
			|| g_str_has_suffix (name, ".la"));
}
