
#line 1 "./zscan_rfc1035.rl"
/* Copyright © 2012 Brandon L Black <blblack@gmail.com> and Jay Reitz <jreitz@gmail.com>
 *
 * This file is part of gdnsd.
 *
 * gdnsd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * gdnsd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gdnsd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "zscan_rfc1035.h"

#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <setjmp.h>

#include "conf.h"
#include "ltree.h"
#include "ltarena.h"
#include "gdnsd/log.h"
#include "gdnsd/misc.h"

#ifndef INET6_ADDRSTRLEN
#define INET6_ADDRSTRLEN 46
#endif

/*
 * The longest possible tstart-based token is a maximum-length
 *  quoted TXT string, which given autosplit, can be up to
 *  roughly 64K, which means buffer size has to be that big
 *  to accommodate that.  There probably wouldn't be any bugs
 *  going down to something reasonable like 4K, but it would
 *  cause parse errors if anyone tried to use longer TXT strings.
 * Another important thing: for integers, we use atoi() directly
 *  on the buffer itself.  In the normal case this works because
 *  there is always some non-integer after it in the buffer, which
 *  halts atoi().  The corner case is if the last digit of an
 *  integer happened to be the last byte of the buffer.  This
 *  is why we allocate one extra buffer byte and set it to zero.
 */
#define MAX_BUFSIZE 65536


#define parse_error(_fmt, ...) \
    do {\
        log_err("rfc1035: Zone %s: Zonefile parse error at line %u: " _fmt,logf_dname(z->zone->dname),z->lcount,__VA_ARGS__);\
        siglongjmp(z->jbuf, 1);\
    } while(0)

#define parse_error_noargs(_fmt) \
    do {\
        log_err("rfc1035: Zone %s: Zonefile parse error at line %u: " _fmt,logf_dname(z->zone->dname),z->lcount);\
        siglongjmp(z->jbuf, 1);\
    } while(0)

typedef struct {
    uint8_t  ipv6[16];
    uint32_t ipv4;
    bool     in_paren;
    bool     zn_err_detect;
    bool     lhs_is_ooz;
    unsigned lcount;
    unsigned num_texts;
    unsigned def_ttl;
    unsigned uval;
    unsigned ttl;
    unsigned uv_1;
    unsigned uv_2;
    unsigned uv_3;
    unsigned uv_4;
    unsigned uv_5;
    unsigned rfc3597_data_len;
    unsigned rfc3597_data_written;
    unsigned limit_v4;
    unsigned limit_v6;
    uint8_t* rfc3597_data;
    zone_t* zone;
    const char* tstart;
    uint8_t  origin[256];
    uint8_t  lhs_dname[256];
    uint8_t  rhs_dname[256];
    uint8_t  eml_dname[256];
    uint8_t** texts;
    sigjmp_buf jbuf;
} zscan_t;

F_NONNULL
static void scanner(zscan_t* z, char* buf, const unsigned bufsize, const int fd);

/******** IP Addresses ********/

F_NONNULL
static void set_ipv4(zscan_t* z, const char* end) {
    dmn_assert(z); dmn_assert(end);
    char txt[16];
    unsigned len = end - z->tstart;
    memcpy(txt, z->tstart, len);
    txt[len] = 0;
    z->tstart = NULL;
    struct in_addr addr;
    int status = inet_pton(AF_INET, txt, &addr);
    if(status > 0)
        z->ipv4 = addr.s_addr;
    else
        parse_error("IPv4 address '%s' invalid", txt);
}

F_NONNULL
static void set_ipv6(zscan_t* z, const char* end) {
    dmn_assert(z);
    dmn_assert(end);
    char txt[INET6_ADDRSTRLEN + 1];
    unsigned len = end - z->tstart;
    memcpy(txt, z->tstart, len);
    txt[len] = 0;
    z->tstart = NULL;
    struct in6_addr v6a;
    int status = inet_pton(AF_INET6, txt, &v6a);
    if(status > 0)
        memcpy(z->ipv6, v6a.s6_addr, 16);
    else
        parse_error("IPv6 address '%s' invalid", txt);
}

F_NONNULL
static void validate_origin_in_zone(zscan_t* z, const uint8_t* origin) {
    dmn_assert(z); dmn_assert(z->zone->dname); dmn_assert(origin);
    if(!dname_isinzone(z->zone->dname, origin))
        parse_error("Origin '%s' is not within this zonefile's zone (%s)", logf_dname(origin), logf_dname(z->zone->dname));
}

F_NONNULL F_PURE
static void validate_lhs_not_ooz(zscan_t* z) {
    dmn_assert(z);
    if(z->lhs_is_ooz)
        parse_error("Domainname '%s' is not within this zonefile's zone (%s)", logf_dname(z->lhs_dname), logf_dname(z->zone->dname));
}

F_NONNULL
static void dname_set(zscan_t* z, uint8_t* dname, unsigned len, bool lhs) {
    dmn_assert(z); dmn_assert(dname); dmn_assert(z->zone->dname);
    dname_status_t catstat;
    dname_status_t status;

    if(len) {
        status = dname_from_string(dname, (const uint8_t*)z->tstart, len);
    }
    else {
        dmn_assert(lhs);
        dname_copy(dname, z->origin);
        status = DNAME_VALID;
    }

    switch(status) {
        case DNAME_INVALID:
            parse_error_noargs("unparseable domainname");
            break;
        case DNAME_VALID:
            if(lhs) {
                const bool inzone = dname_isinzone(z->zone->dname, dname);
                z->lhs_is_ooz = !inzone;
                // in-zone LHS dnames are made relative to zroot
                if(inzone)
                    gdnsd_dname_drop_zone(dname, z->zone->dname);
            }
            break;
        case DNAME_PARTIAL:
            // even though in the lhs case we commonly trim
            //   back most or all of z->origin from dname, we
            //   still have to construct it just for validity checks
            catstat = dname_cat(dname, z->origin);
            if(catstat == DNAME_INVALID)
                parse_error_noargs("illegal domainname");
            dmn_assert(catstat == DNAME_VALID);
            if(lhs) {
                z->lhs_is_ooz = false;
                gdnsd_dname_drop_zone(dname, z->zone->dname);
            }
            break;
    }
}

/********** TXT/SPF ******************/

F_NONNULL
static void text_start(zscan_t* z) {
    dmn_assert(z);
    z->num_texts = 0;
    z->texts = NULL;
}

F_NONNULL
static void text_add_tok(zscan_t* z, const unsigned len, const bool big_ok) {
    dmn_assert(z);

    uint8_t text_temp[len + 1];
    text_temp[0] = 0;
    unsigned newlen = len;
    if(len)
        newlen = dns_unescape(text_temp, (const uint8_t*)z->tstart, len);

    dmn_assert(newlen <= len);

    if(newlen > 255) {
        if(!big_ok || gconfig.disable_text_autosplit)
            parse_error_noargs("Text chunk too long (>255 unescaped)");
        if(newlen > 65500) parse_error_noargs("Text chunk too long (>65500 unescaped)");
        unsigned remainder = newlen % 255;
        unsigned num_whole_chunks = (newlen - remainder) / 255;
        const uint8_t* zptr = text_temp;
        const unsigned new_alloc = 1 + z->num_texts + num_whole_chunks + (remainder ? 1 : 0);
        z->texts = realloc(z->texts, new_alloc * sizeof(uint8_t*));
        for(unsigned i = 0; i < num_whole_chunks; i++) {
            uint8_t* chunk = z->texts[z->num_texts++] = malloc(256);
            *chunk++ = 255;
            memcpy(chunk, zptr, 255);
            zptr += 255;
        }
        if(remainder) {
            uint8_t* chunk = z->texts[z->num_texts++] = malloc(remainder + 1);
            *chunk++ = remainder;
            memcpy(chunk, zptr, remainder);
        }
        z->texts[z->num_texts] = NULL;
    }
    else {
        z->texts = realloc(z->texts, (z->num_texts + 2) * sizeof(uint8_t*));
        uint8_t* chunk = z->texts[z->num_texts++] = malloc(newlen + 1);
        *chunk++ = newlen;
        memcpy(chunk, text_temp, newlen);
        z->texts[z->num_texts] = NULL;
    }

    z->tstart = NULL;
}

// Input must have two bytes of text constrained to [0-9A-Fa-f]
F_NONNULL
static unsigned hexbyte(const char* intxt) {
    dmn_assert(intxt);
    dmn_assert(
        (intxt[0] >= '0' && intxt[0] <= '9')
        || (intxt[0] >= 'A' && intxt[0] <= 'F')
        || (intxt[0] >= 'a' && intxt[0] <= 'f')
    );
    dmn_assert(
        (intxt[1] >= '0' && intxt[1] <= '9')
        || (intxt[1] >= 'A' && intxt[1] <= 'F')
        || (intxt[1] >= 'a' && intxt[1] <= 'f')
    );

    unsigned out;

    if(intxt[0] <= '9')
        out = (intxt[0] - '0') << 4;
    else
        out = ((intxt[0] | 0x20) - ('a' - 10)) << 4;

    if(intxt[1] <= '9')
        out |= (intxt[1] - '0');
    else
        out |= ((intxt[1] | 0x20) - ('a' - 10));

    return out;
}

F_NONNULL
static void mult_uval(zscan_t* z, int fc) {
    dmn_assert(z);
    fc |= 0x20;
    switch(fc) {
        case 'm': z->uval *= 60; break;
        case 'h': z->uval *= 3600; break;
        case 'd': z->uval *= 86400; break;
        case 'w': z->uval *= 604800; break;
    }
}

F_NONNULL
static void set_dyna(zscan_t* z, const char* fpc) {
    dmn_assert(z);
    unsigned dlen = fpc - z->tstart;
    if(dlen > 255)
        parse_error_noargs("DYNA/DYNC plugin!resource string cannot exceed 255 chars");
    memcpy(z->eml_dname, z->tstart, dlen);
    z->eml_dname[dlen] = 0;
    z->tstart = NULL;
}

F_NONNULL
static void rec_soa(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(z->lhs_dname[0] != 1)
        parse_error_noargs("SOA record can only be defined for the root of the zone");
    if(ltree_add_rec_soa(z->zone, z->lhs_dname, z->rhs_dname, z->eml_dname, z->ttl, z->uv_1, z->uv_2, z->uv_3, z->uv_4, z->uv_5))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_a(zscan_t* z) {
    dmn_assert(z);
    if(ltree_add_rec_a(z->zone, z->lhs_dname, z->ipv4, z->ttl, z->limit_v4, z->lhs_is_ooz))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_aaaa(zscan_t* z) {
    dmn_assert(z);
    if(ltree_add_rec_aaaa(z->zone, z->lhs_dname, z->ipv6, z->ttl, z->limit_v6, z->lhs_is_ooz))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_ns(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_ns(z->zone, z->lhs_dname, z->rhs_dname, z->ttl))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_cname(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_cname(z->zone, z->lhs_dname, z->rhs_dname, z->ttl))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_ptr(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_ptr(z->zone, z->lhs_dname, z->rhs_dname, z->ttl))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_mx(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_mx(z->zone, z->lhs_dname, z->rhs_dname, z->ttl, z->uv_1))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_srv(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_srv(z->zone, z->lhs_dname, z->rhs_dname, z->ttl, z->uv_1, z->uv_2, z->uv_3))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void texts_cleanup(zscan_t* z) {
    dmn_assert(z);
    free(z->texts);
    z->texts = NULL;
    z->num_texts = 0;
}

F_NONNULL
static void rec_naptr(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_naptr(z->zone, z->lhs_dname, z->rhs_dname, z->ttl, z->uv_1, z->uv_2, z->num_texts, z->texts))
        siglongjmp(z->jbuf, 1);
    texts_cleanup(z);
}

F_NONNULL
static void rec_txt(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_txt(z->zone, z->lhs_dname, z->num_texts, z->texts, z->ttl))
        siglongjmp(z->jbuf, 1);
    texts_cleanup(z);
}

F_NONNULL
static void rec_spf(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_spf(z->zone, z->lhs_dname, z->num_texts, z->texts, z->ttl))
        siglongjmp(z->jbuf, 1);
    texts_cleanup(z);
}

F_NONNULL
static void rec_spftxt(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_spftxt(z->zone, z->lhs_dname, z->num_texts, z->texts, z->ttl))
        siglongjmp(z->jbuf, 1);
    texts_cleanup(z);
}

F_NONNULL
static void rec_dyna(zscan_t* z) {
    dmn_assert(z);
    if(ltree_add_rec_dynaddr(z->zone, z->lhs_dname, z->eml_dname, z->ttl, z->limit_v4, z->limit_v6, z->lhs_is_ooz))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_dyncname(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_dyncname(z->zone, z->lhs_dname, z->eml_dname, z->origin, z->ttl))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_rfc3597(zscan_t* z) {
    dmn_assert(z);
    if(z->rfc3597_data_written < z->rfc3597_data_len)
        parse_error("RFC3597 generic RR claimed rdata length of %u, but only %u bytes of data present", z->rfc3597_data_len, z->rfc3597_data_written);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_rfc3597(z->zone, z->lhs_dname, z->uv_1, z->ttl, z->rfc3597_data_len, z->rfc3597_data))
        siglongjmp(z->jbuf, 1);
    z->rfc3597_data = NULL;
}

F_NONNULL
static void rfc3597_data_setup(zscan_t* z) {
    dmn_assert(z);
    z->rfc3597_data_len = z->uval;
    z->rfc3597_data_written = 0;
    z->rfc3597_data = malloc(z->uval);
}

F_NONNULL
static void rfc3597_octet(zscan_t* z) {
    dmn_assert(z);
    if(z->rfc3597_data_written == z->rfc3597_data_len)
       parse_error_noargs("RFC3597 generic RR: more rdata is present than the indicated length");
    z->rfc3597_data[z->rfc3597_data_written++] = hexbyte(z->tstart);
}

F_NONNULL
static void set_limit_v4(zscan_t* z) {
    dmn_assert(z);
    if(z->uval > 65535)
        parse_error("$ADDR_LIMIT_V4 value %u out of range (0-65535)", z->uval);
    z->limit_v4 = z->uval;
}

F_NONNULL
static void set_limit_v6(zscan_t* z) {
    dmn_assert(z);
    if(z->uval > 65535)
        parse_error("$ADDR_LIMIT_V6 value %u out of range (0-65535)", z->uval);
    z->limit_v6 = z->uval;
}

F_NONNULL
static void open_paren(zscan_t* z) {
    dmn_assert(z);
    if(z->in_paren)
        parse_error_noargs("Parenthetical error: double-open");
    z->in_paren = true;
}

F_NONNULL
static void close_paren(zscan_t* z) {
    dmn_assert(z);
    if(!z->in_paren)
        parse_error_noargs("Parenthetical error: unnecessary close");
    z->in_paren = false;
}


#line 487 "./zscan_rfc1035.c"
static const int zone_start = 567;
static const int zone_first_final = 567;
static const int zone_error = 0;

static const int zone_en_main = 567;


#line 690 "./zscan_rfc1035.rl"


F_NONNULL
static void scanner(zscan_t* z, char* buf, const unsigned bufsize, const int fd) {
    dmn_assert(z);

    char* read_at;

    const char* pe = NULL;
    const char* eof = NULL;
    int cs = zone_start;

    while(!eof) {
        unsigned have = 0;
        if(z->tstart != NULL) {
            dmn_assert(pe);
            dmn_assert(z->tstart < pe);
            dmn_assert(z->tstart != buf);
            have = pe - z->tstart;
            memmove(buf, z->tstart, have);
            z->tstart = buf;
        }

        const int space = bufsize - have;
        const char* p = read_at = buf + have;

        const int len = read(fd, read_at, space);
        if(len < 0)
            parse_error("read() failed: %s", logf_errno());

        pe = p + len;

        if(len < space)
            eof = pe;

        
#line 532 "./zscan_rfc1035.c"
	{
	short _widec;
	if ( p == pe )
		goto _test_eof;
	switch ( cs )
	{
case 567:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st15;
		case 32: goto st15;
		case 34: goto tr1984;
		case 36: goto st462;
		case 40: goto tr85;
		case 41: goto tr86;
		case 64: goto tr1986;
		case 92: goto tr1987;
		case 266: goto st568;
		case 269: goto st25;
		case 315: goto st26;
		case 522: goto st449;
		case 525: goto st450;
		case 571: goto st451;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1983;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1983;
		} else if ( _widec >= 14 )
			goto tr1983;
	} else
		goto tr1983;
	goto st0;
tr1983:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st1;
tr1907:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st1;
tr1988:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st1;
st1:
	if ( ++p == pe )
		goto _test_eof1;
case 1:
#line 607 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1;
		case 32: goto tr1;
		case 40: goto tr3;
		case 41: goto tr4;
		case 92: goto st521;
		case 522: goto tr6;
		case 525: goto tr7;
		case 571: goto tr8;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st1;
		} else
			goto st1;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st1;
		} else if ( _widec >= 35 )
			goto st1;
	} else
		goto st1;
	goto st0;
tr1:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st2;
tr3:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr4:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr10:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr11:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr1709:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st2;
tr1710:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr1711:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr1703:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
	goto st2;
tr1704:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr1705:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr1898:
#line 489 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
	goto st2;
tr1899:
#line 489 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr1900:
#line 489 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr1908:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st2;
tr1909:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr1910:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
st2:
	if ( ++p == pe )
		goto _test_eof2;
case 2:
#line 752 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st2;
		case 32: goto st2;
		case 40: goto tr10;
		case 41: goto tr11;
		case 65: goto tr13;
		case 67: goto tr14;
		case 68: goto tr15;
		case 73: goto tr16;
		case 77: goto tr17;
		case 78: goto tr18;
		case 80: goto tr19;
		case 83: goto tr20;
		case 84: goto tr21;
		case 97: goto tr13;
		case 99: goto tr14;
		case 100: goto tr15;
		case 105: goto tr16;
		case 109: goto tr17;
		case 110: goto tr18;
		case 112: goto tr19;
		case 115: goto tr20;
		case 116: goto tr21;
		case 522: goto st456;
		case 525: goto st457;
		case 571: goto st458;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr12;
	goto st0;
st0:
cs = 0;
	goto _out;
tr12:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st3;
tr1680:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st3;
st3:
	if ( ++p == pe )
		goto _test_eof3;
case 3:
#line 825 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr25;
		case 32: goto tr25;
		case 40: goto tr26;
		case 41: goto tr27;
		case 68: goto tr29;
		case 72: goto tr29;
		case 77: goto tr29;
		case 87: goto tr29;
		case 100: goto tr29;
		case 104: goto tr29;
		case 109: goto tr29;
		case 119: goto tr29;
		case 522: goto tr30;
		case 525: goto tr31;
		case 571: goto tr32;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st3;
	goto st0;
tr34:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st4;
tr35:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st4;
tr25:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st4;
tr26:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st4;
tr27:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st4;
tr1970:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st4;
tr1971:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st4;
tr1972:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st4;
tr1977:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st4;
tr1978:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st4;
tr1979:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st4;
st4:
	if ( ++p == pe )
		goto _test_eof4;
case 4:
#line 933 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st4;
		case 32: goto st4;
		case 40: goto tr34;
		case 41: goto tr35;
		case 65: goto st5;
		case 67: goto st16;
		case 68: goto st44;
		case 73: goto st561;
		case 77: goto st88;
		case 78: goto st113;
		case 80: goto st216;
		case 83: goto st237;
		case 84: goto st393;
		case 97: goto st5;
		case 99: goto st16;
		case 100: goto st44;
		case 105: goto st561;
		case 109: goto st88;
		case 110: goto st113;
		case 112: goto st216;
		case 115: goto st237;
		case 116: goto st393;
		case 522: goto st563;
		case 525: goto st564;
		case 571: goto st565;
	}
	goto st0;
tr13:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st5;
tr1653:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st5;
tr1681:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st5;
st5:
	if ( ++p == pe )
		goto _test_eof5;
case 5:
#line 1001 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st6;
		case 32: goto st6;
		case 40: goto tr49;
		case 41: goto tr50;
		case 65: goto st537;
		case 97: goto st537;
		case 522: goto st534;
		case 525: goto st535;
		case 571: goto st536;
	}
	goto st0;
tr49:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st6;
tr50:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st6;
tr1920:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st6;
tr1921:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st6;
tr1922:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st6;
st6:
	if ( ++p == pe )
		goto _test_eof6;
case 6:
#line 1063 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st6;
		case 32: goto st6;
		case 40: goto tr49;
		case 41: goto tr50;
		case 522: goto st534;
		case 525: goto st535;
		case 571: goto st536;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr55;
	goto st0;
tr55:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st7;
tr1923:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st7;
st7:
	if ( ++p == pe )
		goto _test_eof7;
case 7:
#line 1111 "./zscan_rfc1035.c"
	if ( (*p) == 46 )
		goto st8;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st532;
	goto st0;
st8:
	if ( ++p == pe )
		goto _test_eof8;
case 8:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st9;
	goto st0;
st9:
	if ( ++p == pe )
		goto _test_eof9;
case 9:
	if ( (*p) == 46 )
		goto st10;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st530;
	goto st0;
st10:
	if ( ++p == pe )
		goto _test_eof10;
case 10:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st11;
	goto st0;
st11:
	if ( ++p == pe )
		goto _test_eof11;
case 11:
	if ( (*p) == 46 )
		goto st12;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st528;
	goto st0;
st12:
	if ( ++p == pe )
		goto _test_eof12;
case 12:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st13;
	goto st0;
st13:
	if ( ++p == pe )
		goto _test_eof13;
case 13:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr65;
		case 32: goto tr65;
		case 40: goto tr66;
		case 41: goto tr67;
		case 266: goto tr69;
		case 269: goto tr70;
		case 315: goto tr71;
		case 522: goto tr72;
		case 525: goto tr73;
		case 571: goto tr74;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st526;
	goto st0;
tr76:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr77:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr65:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st14;
tr66:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr67:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr105:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st14;
tr106:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr107:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr129:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st14;
tr119:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st14;
tr120:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr121:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr130:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr131:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr149:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st14;
tr150:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr151:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr160:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st14;
tr161:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr162:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr192:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st14;
tr194:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr195:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr209:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st14;
tr210:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr211:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr244:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st14;
tr246:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr247:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr261:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st14;
tr262:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr263:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr335:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st14;
tr336:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr337:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr349:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st14;
tr350:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr351:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr370:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st14;
tr371:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr372:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr381:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st14;
tr382:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr383:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr502:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st14;
tr503:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr504:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr516:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st14;
tr517:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr518:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr537:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st14;
tr538:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr539:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr548:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st14;
tr549:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr550:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr709:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st14;
tr710:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr711:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr723:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st14;
tr724:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr725:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr744:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st14;
tr745:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr746:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr755:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st14;
tr756:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr757:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr787:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st14;
tr788:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr789:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr801:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st14;
tr802:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr803:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr822:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st14;
tr823:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr824:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr833:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st14;
tr834:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr835:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr958:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st14;
tr959:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr960:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr969:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st14;
tr970:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr971:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1384:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st14;
tr1385:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1386:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1398:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st14;
tr1399:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1400:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1419:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st14;
tr1420:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1421:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1430:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st14;
tr1431:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1432:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1740:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st14;
tr1741:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1742:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1764:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st14;
tr1765:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1766:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1797:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st14;
tr1798:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1799:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1811:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st14;
tr1812:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1813:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1832:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st14;
tr1833:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1834:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1843:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st14;
tr1844:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1845:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1871:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st14;
tr1872:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1873:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1882:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st14;
tr1883:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1884:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1936:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st14;
tr1937:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1938:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
st14:
	if ( ++p == pe )
		goto _test_eof14;
case 14:
#line 2194 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st14;
		case 32: goto st14;
		case 40: goto tr76;
		case 41: goto tr77;
		case 266: goto st568;
		case 269: goto st25;
		case 315: goto st26;
		case 522: goto st27;
		case 525: goto st28;
		case 571: goto st29;
	}
	goto st0;
tr69:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st568;
tr109:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st568;
tr132:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st568;
tr123:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st568;
tr152:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st568;
tr163:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st568;
tr197:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st568;
tr213:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st568;
tr249:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st568;
tr265:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st568;
tr339:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st568;
tr353:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st568;
tr373:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st568;
tr384:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st568;
tr506:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st568;
tr520:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st568;
tr540:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st568;
tr551:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st568;
tr713:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st568;
tr727:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st568;
tr747:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st568;
tr758:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st568;
tr791:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st568;
tr805:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st568;
tr825:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st568;
tr836:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st568;
tr963:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st568;
tr972:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st568;
tr1145:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st568;
tr1157:
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st568;
tr1179:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st568;
tr1194:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st568;
tr1206:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st568;
tr1248:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st568;
tr1260:
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st568;
tr1282:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st568;
tr1297:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st568;
tr1309:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st568;
tr1388:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st568;
tr1402:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st568;
tr1422:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st568;
tr1433:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st568;
tr1494:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st568;
tr1506:
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st568;
tr1528:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st568;
tr1543:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st568;
tr1555:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st568;
tr1605:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 543 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st568;
tr1623:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 543 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st568;
tr1744:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st568;
tr1768:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st568;
tr1801:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st568;
tr1815:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st568;
tr1835:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st568;
tr1846:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st568;
tr1876:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st568;
tr1885:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st568;
tr1941:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st568;
st568:
	if ( ++p == pe )
		goto _test_eof568;
case 568:
#line 2611 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1677;
		case 32: goto tr1677;
		case 34: goto tr1989;
		case 36: goto tr1990;
		case 40: goto tr1678;
		case 41: goto tr1679;
		case 64: goto tr1991;
		case 92: goto tr1992;
		case 266: goto tr132;
		case 269: goto tr133;
		case 315: goto tr134;
		case 522: goto tr1690;
		case 525: goto tr1691;
		case 571: goto tr1692;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1988;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1988;
		} else if ( _widec >= 14 )
			goto tr1988;
	} else
		goto tr1988;
	goto st0;
tr85:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st15;
tr86:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st15;
tr1677:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st15;
tr1678:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st15;
tr1679:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st15;
st15:
	if ( ++p == pe )
		goto _test_eof15;
case 15:
#line 2689 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st15;
		case 32: goto st15;
		case 40: goto tr85;
		case 41: goto tr86;
		case 65: goto tr13;
		case 67: goto tr14;
		case 68: goto tr15;
		case 73: goto tr16;
		case 77: goto tr17;
		case 78: goto tr18;
		case 80: goto tr19;
		case 83: goto tr20;
		case 84: goto tr21;
		case 97: goto tr13;
		case 99: goto tr14;
		case 100: goto tr15;
		case 105: goto tr16;
		case 109: goto tr17;
		case 110: goto tr18;
		case 112: goto tr19;
		case 115: goto tr20;
		case 116: goto tr21;
		case 266: goto st568;
		case 269: goto st25;
		case 315: goto st26;
		case 522: goto st449;
		case 525: goto st450;
		case 571: goto st451;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr12;
	goto st0;
tr14:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st16;
tr1654:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st16;
tr1682:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st16;
st16:
	if ( ++p == pe )
		goto _test_eof16;
case 16:
#line 2762 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 78: goto st17;
		case 110: goto st17;
	}
	goto st0;
st17:
	if ( ++p == pe )
		goto _test_eof17;
case 17:
	switch( (*p) ) {
		case 65: goto st18;
		case 97: goto st18;
	}
	goto st0;
st18:
	if ( ++p == pe )
		goto _test_eof18;
case 18:
	switch( (*p) ) {
		case 77: goto st19;
		case 109: goto st19;
	}
	goto st0;
st19:
	if ( ++p == pe )
		goto _test_eof19;
case 19:
	switch( (*p) ) {
		case 69: goto st20;
		case 101: goto st20;
	}
	goto st0;
st20:
	if ( ++p == pe )
		goto _test_eof20;
case 20:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st21;
		case 32: goto st21;
		case 40: goto tr95;
		case 41: goto tr96;
		case 522: goto st41;
		case 525: goto st42;
		case 571: goto st43;
	}
	goto st0;
tr95:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st21;
tr96:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st21;
tr170:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st21;
tr172:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st21;
tr173:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st21;
st21:
	if ( ++p == pe )
		goto _test_eof21;
case 21:
#line 2858 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st21;
		case 32: goto st21;
		case 34: goto tr101;
		case 40: goto tr95;
		case 41: goto tr96;
		case 64: goto tr102;
		case 92: goto tr103;
		case 522: goto st41;
		case 525: goto st42;
		case 571: goto st43;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr100;
		} else
			goto tr100;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr100;
		} else if ( _widec >= 37 )
			goto tr100;
	} else
		goto tr100;
	goto st0;
tr100:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st22;
tr118:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st22;
tr169:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st22;
st22:
	if ( ++p == pe )
		goto _test_eof22;
case 22:
#line 2925 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr105;
		case 32: goto tr105;
		case 40: goto tr106;
		case 41: goto tr107;
		case 92: goto st23;
		case 266: goto tr109;
		case 269: goto tr110;
		case 315: goto tr111;
		case 522: goto tr112;
		case 525: goto tr113;
		case 571: goto tr114;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st22;
		} else
			goto st22;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st22;
		} else if ( _widec >= 35 )
			goto st22;
	} else
		goto st22;
	goto st0;
tr103:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st23;
tr122:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st23;
tr175:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st23;
st23:
	if ( ++p == pe )
		goto _test_eof23;
case 23:
#line 2993 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st24;
		case 13: goto st30;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st31;
	goto st22;
st24:
	if ( ++p == pe )
		goto _test_eof24;
case 24:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr119;
		case 32: goto tr119;
		case 40: goto tr120;
		case 41: goto tr121;
		case 92: goto tr122;
		case 266: goto tr123;
		case 269: goto tr124;
		case 315: goto tr125;
		case 522: goto tr126;
		case 525: goto tr127;
		case 571: goto tr128;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr118;
		} else
			goto tr118;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr118;
		} else if ( _widec >= 35 )
			goto tr118;
	} else
		goto tr118;
	goto st0;
tr70:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st25;
tr110:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st25;
tr133:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st25;
tr124:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st25;
tr153:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st25;
tr164:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st25;
tr198:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st25;
tr214:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st25;
tr250:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st25;
tr266:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st25;
tr340:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st25;
tr354:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st25;
tr374:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st25;
tr385:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st25;
tr507:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st25;
tr521:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st25;
tr541:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st25;
tr552:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st25;
tr714:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st25;
tr728:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st25;
tr748:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st25;
tr759:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st25;
tr792:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st25;
tr806:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st25;
tr826:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st25;
tr837:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st25;
tr964:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st25;
tr973:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st25;
tr1146:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st25;
tr1158:
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st25;
tr1180:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st25;
tr1195:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st25;
tr1207:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st25;
tr1249:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st25;
tr1261:
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st25;
tr1283:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st25;
tr1298:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st25;
tr1310:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st25;
tr1389:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st25;
tr1403:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st25;
tr1423:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st25;
tr1434:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st25;
tr1495:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1507:
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1529:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1544:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1556:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1606:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 543 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st25;
tr1624:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 543 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st25;
tr1745:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st25;
tr1769:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st25;
tr1802:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st25;
tr1816:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st25;
tr1836:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st25;
tr1847:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st25;
tr1877:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st25;
tr1886:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st25;
tr1942:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st25;
st25:
	if ( ++p == pe )
		goto _test_eof25;
case 25:
#line 3439 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 266 )
		goto st568;
	goto st0;
tr71:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st26;
tr111:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st26;
tr134:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st26;
tr125:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st26;
tr154:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st26;
tr165:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st26;
tr199:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st26;
tr215:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st26;
tr251:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st26;
tr267:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st26;
tr341:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st26;
tr355:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st26;
tr375:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st26;
tr386:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st26;
tr508:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st26;
tr522:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st26;
tr542:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st26;
tr553:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st26;
tr715:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st26;
tr729:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st26;
tr749:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st26;
tr760:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st26;
tr793:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st26;
tr807:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st26;
tr827:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st26;
tr838:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st26;
tr965:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st26;
tr974:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st26;
tr1147:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st26;
tr1159:
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st26;
tr1181:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st26;
tr1196:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st26;
tr1208:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 539 "./zscan_rfc1035.rl"
	{ rec_spf(z); }
	goto st26;
tr1250:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st26;
tr1262:
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st26;
tr1284:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st26;
tr1299:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st26;
tr1311:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 540 "./zscan_rfc1035.rl"
	{ rec_spftxt(z); }
	goto st26;
tr1390:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st26;
tr1404:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st26;
tr1424:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st26;
tr1435:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st26;
tr1496:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1508:
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1530:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1545:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1557:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 538 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1607:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 543 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st26;
tr1625:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 543 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st26;
tr1746:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st26;
tr1770:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st26;
tr1803:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st26;
tr1817:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st26;
tr1837:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st26;
tr1848:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st26;
tr1878:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st26;
tr1887:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st26;
tr1943:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st26;
st26:
	if ( ++p == pe )
		goto _test_eof26;
case 26:
#line 3832 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 266: goto st568;
		case 269: goto st25;
	}
	if ( 128 <= _widec && _widec <= 383 )
		goto st26;
	goto st0;
tr72:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st27;
tr112:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st27;
tr135:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st27;
tr126:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st27;
tr155:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st27;
tr166:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st27;
tr200:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st27;
tr216:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st27;
tr252:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st27;
tr268:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st27;
tr342:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st27;
tr356:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st27;
tr376:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st27;
tr387:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st27;
tr509:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st27;
tr523:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st27;
tr543:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st27;
tr554:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st27;
tr716:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st27;
tr730:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st27;
tr750:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st27;
tr761:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st27;
tr794:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st27;
tr808:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st27;
tr828:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st27;
tr839:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st27;
tr966:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st27;
tr975:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st27;
tr1391:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st27;
tr1405:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st27;
tr1425:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st27;
tr1436:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st27;
tr1747:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st27;
tr1771:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st27;
tr1804:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st27;
tr1818:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st27;
tr1838:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st27;
tr1849:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st27;
tr1879:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st27;
tr1888:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st27;
tr1944:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st27;
st27:
	if ( ++p == pe )
		goto _test_eof27;
case 27:
#line 4153 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr129;
		case 32: goto tr129;
		case 40: goto tr130;
		case 41: goto tr131;
		case 266: goto tr132;
		case 269: goto tr133;
		case 315: goto tr134;
		case 522: goto tr135;
		case 525: goto tr136;
		case 571: goto tr137;
	}
	goto st0;
tr73:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st28;
tr113:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st28;
tr136:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st28;
tr127:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st28;
tr156:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st28;
tr167:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st28;
tr201:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st28;
tr217:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st28;
tr253:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st28;
tr269:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st28;
tr343:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st28;
tr357:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st28;
tr377:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st28;
tr388:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st28;
tr510:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st28;
tr524:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st28;
tr544:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st28;
tr555:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st28;
tr717:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st28;
tr731:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st28;
tr751:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st28;
tr762:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st28;
tr795:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st28;
tr809:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st28;
tr829:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st28;
tr840:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st28;
tr967:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st28;
tr976:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st28;
tr1392:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st28;
tr1406:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st28;
tr1426:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st28;
tr1437:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st28;
tr1748:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st28;
tr1772:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st28;
tr1805:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st28;
tr1819:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st28;
tr1839:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st28;
tr1850:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st28;
tr1880:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st28;
tr1889:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st28;
tr1945:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st28;
st28:
	if ( ++p == pe )
		goto _test_eof28;
case 28:
#line 4466 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st27;
	goto st0;
tr74:
#line 509 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st29;
tr114:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st29;
tr137:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st29;
tr128:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st29;
tr157:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st29;
tr168:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st29;
tr202:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st29;
tr218:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 541 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st29;
tr254:
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st29;
tr270:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 527 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 542 "./zscan_rfc1035.rl"
	{ rec_dyncname(z); }
	goto st29;
tr344:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st29;
tr358:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st29;
tr378:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st29;
tr389:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st29;
tr511:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st29;
tr525:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st29;
tr545:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st29;
tr556:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st29;
tr718:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st29;
tr732:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st29;
tr752:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st29;
tr763:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st29;
tr796:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st29;
tr810:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st29;
tr830:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st29;
tr841:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st29;
tr968:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st29;
tr977:
#line 521 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 529 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st29;
tr1393:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st29;
tr1407:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st29;
tr1427:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st29;
tr1438:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st29;
tr1749:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st29;
tr1773:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 524 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st29;
tr1806:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st29;
tr1820:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st29;
tr1840:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st29;
tr1851:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 498 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st29;
tr1881:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st29;
tr1890:
#line 515 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st29;
tr1946:
#line 510 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st29;
st29:
	if ( ++p == pe )
		goto _test_eof29;
case 29:
#line 4755 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st27;
		case 525: goto st28;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st29;
	goto st0;
st30:
	if ( ++p == pe )
		goto _test_eof30;
case 30:
	if ( (*p) == 10 )
		goto st24;
	goto st0;
st31:
	if ( ++p == pe )
		goto _test_eof31;
case 31:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st32;
	goto st0;
st32:
	if ( ++p == pe )
		goto _test_eof32;
case 32:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st22;
	goto st0;
tr101:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st33;
tr144:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st33;
tr171:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st33;
st33:
	if ( ++p == pe )
		goto _test_eof33;
case 33:
#line 4837 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st34;
		case 13: goto st35;
		case 34: goto st36;
		case 92: goto st37;
	}
	goto st33;
tr145:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st34;
st34:
	if ( ++p == pe )
		goto _test_eof34;
case 34:
#line 4853 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr145;
		case 13: goto tr146;
		case 34: goto tr147;
		case 92: goto tr148;
	}
	goto tr144;
tr146:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st35;
st35:
	if ( ++p == pe )
		goto _test_eof35;
case 35:
#line 4869 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st34;
	goto st0;
tr147:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st36;
st36:
	if ( ++p == pe )
		goto _test_eof36;
case 36:
#line 4881 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr149;
		case 32: goto tr149;
		case 40: goto tr150;
		case 41: goto tr151;
		case 266: goto tr152;
		case 269: goto tr153;
		case 315: goto tr154;
		case 522: goto tr155;
		case 525: goto tr156;
		case 571: goto tr157;
	}
	goto st0;
tr148:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st37;
st37:
	if ( ++p == pe )
		goto _test_eof37;
case 37:
#line 4924 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st34;
		case 13: goto st35;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st38;
	goto st33;
st38:
	if ( ++p == pe )
		goto _test_eof38;
case 38:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st39;
	goto st0;
st39:
	if ( ++p == pe )
		goto _test_eof39;
case 39:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st33;
	goto st0;
tr102:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st40;
tr174:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st40;
st40:
	if ( ++p == pe )
		goto _test_eof40;
case 40:
#line 4963 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr160;
		case 32: goto tr160;
		case 40: goto tr161;
		case 41: goto tr162;
		case 266: goto tr163;
		case 269: goto tr164;
		case 315: goto tr165;
		case 522: goto tr166;
		case 525: goto tr167;
		case 571: goto tr168;
	}
	goto st0;
tr176:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st41;
st41:
	if ( ++p == pe )
		goto _test_eof41;
case 41:
#line 5006 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr170;
		case 32: goto tr170;
		case 34: goto tr171;
		case 40: goto tr172;
		case 41: goto tr173;
		case 64: goto tr174;
		case 92: goto tr175;
		case 522: goto tr176;
		case 525: goto tr177;
		case 571: goto tr178;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr169;
		} else
			goto tr169;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr169;
		} else if ( _widec >= 37 )
			goto tr169;
	} else
		goto tr169;
	goto st0;
tr177:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st42;
st42:
	if ( ++p == pe )
		goto _test_eof42;
case 42:
#line 5063 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st41;
	goto st0;
tr178:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st43;
st43:
	if ( ++p == pe )
		goto _test_eof43;
case 43:
#line 5082 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st41;
		case 525: goto st42;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st43;
	goto st0;
tr15:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st44;
tr1655:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st44;
tr1683:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st44;
st44:
	if ( ++p == pe )
		goto _test_eof44;
case 44:
#line 5143 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 89: goto st45;
		case 121: goto st45;
	}
	goto st0;
st45:
	if ( ++p == pe )
		goto _test_eof45;
case 45:
	switch( (*p) ) {
		case 78: goto st46;
		case 110: goto st46;
	}
	goto st0;
st46:
	if ( ++p == pe )
		goto _test_eof46;
case 46:
	switch( (*p) ) {
		case 65: goto st47;
		case 67: goto st65;
		case 97: goto st47;
		case 99: goto st65;
	}
	goto st0;
st47:
	if ( ++p == pe )
		goto _test_eof47;
case 47:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st48;
		case 32: goto st48;
		case 40: goto tr184;
		case 41: goto tr185;
		case 522: goto st62;
		case 525: goto st63;
		case 571: goto st64;
	}
	goto st0;
tr184:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st48;
tr185:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st48;
tr228:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st48;
tr229:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st48;
tr230:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st48;
st48:
	if ( ++p == pe )
		goto _test_eof48;
case 48:
#line 5232 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st48;
		case 32: goto st48;
		case 40: goto tr184;
		case 41: goto tr185;
		case 92: goto tr190;
		case 522: goto st62;
		case 525: goto st63;
		case 571: goto st64;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr189;
		} else
			goto tr189;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr189;
		} else if ( _widec >= 35 )
			goto tr189;
	} else
		goto tr189;
	goto st0;
tr189:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st49;
tr223:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st49;
tr227:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st49;
st49:
	if ( ++p == pe )
		goto _test_eof49;
case 49:
#line 5297 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr192;
		case 32: goto tr192;
		case 33: goto st50;
		case 40: goto tr194;
		case 41: goto tr195;
		case 92: goto st57;
		case 266: goto tr197;
		case 269: goto tr198;
		case 315: goto tr199;
		case 522: goto tr200;
		case 525: goto tr201;
		case 571: goto tr202;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st49;
		} else
			goto st49;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st49;
		} else if ( _widec >= 35 )
			goto st49;
	} else
		goto st49;
	goto st0;
tr224:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st50;
st50:
	if ( ++p == pe )
		goto _test_eof50;
case 50:
#line 5356 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 13: goto st0;
		case 59: goto st0;
		case 92: goto st52;
	}
	if ( (*p) < 32 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto st0;
	} else if ( (*p) > 34 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto st0;
	} else
		goto st0;
	goto st51;
tr208:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st51;
st51:
	if ( ++p == pe )
		goto _test_eof51;
case 51:
#line 5379 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr192;
		case 32: goto tr192;
		case 40: goto tr194;
		case 41: goto tr195;
		case 92: goto st52;
		case 266: goto tr197;
		case 269: goto tr198;
		case 315: goto tr199;
		case 522: goto tr200;
		case 525: goto tr201;
		case 571: goto tr202;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st51;
		} else
			goto st51;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st51;
		} else if ( _widec >= 35 )
			goto st51;
	} else
		goto st51;
	goto st0;
tr212:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st52;
st52:
	if ( ++p == pe )
		goto _test_eof52;
case 52:
#line 5437 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st53;
		case 13: goto st54;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st55;
	goto st51;
st53:
	if ( ++p == pe )
		goto _test_eof53;
case 53:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr209;
		case 32: goto tr209;
		case 40: goto tr210;
		case 41: goto tr211;
		case 92: goto tr212;
		case 266: goto tr213;
		case 269: goto tr214;
		case 315: goto tr215;
		case 522: goto tr216;
		case 525: goto tr217;
		case 571: goto tr218;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr208;
		} else
			goto tr208;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr208;
		} else if ( _widec >= 35 )
			goto tr208;
	} else
		goto tr208;
	goto st0;
st54:
	if ( ++p == pe )
		goto _test_eof54;
case 54:
	if ( (*p) == 10 )
		goto st53;
	goto st0;
st55:
	if ( ++p == pe )
		goto _test_eof55;
case 55:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st56;
	goto st0;
st56:
	if ( ++p == pe )
		goto _test_eof56;
case 56:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st51;
	goto st0;
tr190:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st57;
tr225:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st57;
tr231:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st57;
st57:
	if ( ++p == pe )
		goto _test_eof57;
case 57:
#line 5540 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st58;
		case 13: goto st59;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st60;
	goto st49;
st58:
	if ( ++p == pe )
		goto _test_eof58;
case 58:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr209;
		case 32: goto tr209;
		case 33: goto tr224;
		case 40: goto tr210;
		case 41: goto tr211;
		case 92: goto tr225;
		case 266: goto tr213;
		case 269: goto tr214;
		case 315: goto tr215;
		case 522: goto tr216;
		case 525: goto tr217;
		case 571: goto tr218;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr223;
		} else
			goto tr223;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr223;
		} else if ( _widec >= 35 )
			goto tr223;
	} else
		goto tr223;
	goto st0;
st59:
	if ( ++p == pe )
		goto _test_eof59;
case 59:
	if ( (*p) == 10 )
		goto st58;
	goto st0;
st60:
	if ( ++p == pe )
		goto _test_eof60;
case 60:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st61;
	goto st0;
st61:
	if ( ++p == pe )
		goto _test_eof61;
case 61:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st49;
	goto st0;
tr232:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st62;
st62:
	if ( ++p == pe )
		goto _test_eof62;
case 62:
#line 5634 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr228;
		case 32: goto tr228;
		case 40: goto tr229;
		case 41: goto tr230;
		case 92: goto tr231;
		case 522: goto tr232;
		case 525: goto tr233;
		case 571: goto tr234;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr227;
		} else
			goto tr227;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr227;
		} else if ( _widec >= 35 )
			goto tr227;
	} else
		goto tr227;
	goto st0;
tr233:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st63;
st63:
	if ( ++p == pe )
		goto _test_eof63;
case 63:
#line 5689 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st62;
	goto st0;
tr234:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st64;
st64:
	if ( ++p == pe )
		goto _test_eof64;
case 64:
#line 5708 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st62;
		case 525: goto st63;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st64;
	goto st0;
st65:
	if ( ++p == pe )
		goto _test_eof65;
case 65:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st66;
		case 32: goto st66;
		case 40: goto tr236;
		case 41: goto tr237;
		case 522: goto st80;
		case 525: goto st81;
		case 571: goto st82;
	}
	goto st0;
tr236:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st66;
tr237:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st66;
tr280:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st66;
tr281:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st66;
tr282:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st66;
st66:
	if ( ++p == pe )
		goto _test_eof66;
case 66:
#line 5814 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st66;
		case 32: goto st66;
		case 40: goto tr236;
		case 41: goto tr237;
		case 92: goto tr242;
		case 522: goto st80;
		case 525: goto st81;
		case 571: goto st82;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr241;
		} else
			goto tr241;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr241;
		} else if ( _widec >= 35 )
			goto tr241;
	} else
		goto tr241;
	goto st0;
tr241:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st67;
tr275:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st67;
tr279:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st67;
st67:
	if ( ++p == pe )
		goto _test_eof67;
case 67:
#line 5879 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr244;
		case 32: goto tr244;
		case 33: goto st68;
		case 40: goto tr246;
		case 41: goto tr247;
		case 92: goto st75;
		case 266: goto tr249;
		case 269: goto tr250;
		case 315: goto tr251;
		case 522: goto tr252;
		case 525: goto tr253;
		case 571: goto tr254;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st67;
		} else
			goto st67;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st67;
		} else if ( _widec >= 35 )
			goto st67;
	} else
		goto st67;
	goto st0;
tr276:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st68;
st68:
	if ( ++p == pe )
		goto _test_eof68;
case 68:
#line 5938 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 13: goto st0;
		case 59: goto st0;
		case 92: goto st70;
	}
	if ( (*p) < 32 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto st0;
	} else if ( (*p) > 34 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto st0;
	} else
		goto st0;
	goto st69;
tr260:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st69;
st69:
	if ( ++p == pe )
		goto _test_eof69;
case 69:
#line 5961 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr244;
		case 32: goto tr244;
		case 40: goto tr246;
		case 41: goto tr247;
		case 92: goto st70;
		case 266: goto tr249;
		case 269: goto tr250;
		case 315: goto tr251;
		case 522: goto tr252;
		case 525: goto tr253;
		case 571: goto tr254;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st69;
		} else
			goto st69;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st69;
		} else if ( _widec >= 35 )
			goto st69;
	} else
		goto st69;
	goto st0;
tr264:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st70;
st70:
	if ( ++p == pe )
		goto _test_eof70;
case 70:
#line 6019 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st71;
		case 13: goto st72;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st73;
	goto st69;
st71:
	if ( ++p == pe )
		goto _test_eof71;
case 71:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr261;
		case 32: goto tr261;
		case 40: goto tr262;
		case 41: goto tr263;
		case 92: goto tr264;
		case 266: goto tr265;
		case 269: goto tr266;
		case 315: goto tr267;
		case 522: goto tr268;
		case 525: goto tr269;
		case 571: goto tr270;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr260;
		} else
			goto tr260;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr260;
		} else if ( _widec >= 35 )
			goto tr260;
	} else
		goto tr260;
	goto st0;
st72:
	if ( ++p == pe )
		goto _test_eof72;
case 72:
	if ( (*p) == 10 )
		goto st71;
	goto st0;
st73:
	if ( ++p == pe )
		goto _test_eof73;
case 73:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st74;
	goto st0;
st74:
	if ( ++p == pe )
		goto _test_eof74;
case 74:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st69;
	goto st0;
tr242:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st75;
tr277:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st75;
tr283:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st75;
st75:
	if ( ++p == pe )
		goto _test_eof75;
case 75:
#line 6122 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st76;
		case 13: goto st77;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st78;
	goto st67;
st76:
	if ( ++p == pe )
		goto _test_eof76;
case 76:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr261;
		case 32: goto tr261;
		case 33: goto tr276;
		case 40: goto tr262;
		case 41: goto tr263;
		case 92: goto tr277;
		case 266: goto tr265;
		case 269: goto tr266;
		case 315: goto tr267;
		case 522: goto tr268;
		case 525: goto tr269;
		case 571: goto tr270;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr275;
		} else
			goto tr275;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr275;
		} else if ( _widec >= 35 )
			goto tr275;
	} else
		goto tr275;
	goto st0;
st77:
	if ( ++p == pe )
		goto _test_eof77;
case 77:
	if ( (*p) == 10 )
		goto st76;
	goto st0;
st78:
	if ( ++p == pe )
		goto _test_eof78;
case 78:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st79;
	goto st0;
st79:
	if ( ++p == pe )
		goto _test_eof79;
case 79:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st67;
	goto st0;
tr284:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st80;
st80:
	if ( ++p == pe )
		goto _test_eof80;
case 80:
#line 6216 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr280;
		case 32: goto tr280;
		case 40: goto tr281;
		case 41: goto tr282;
		case 92: goto tr283;
		case 522: goto tr284;
		case 525: goto tr285;
		case 571: goto tr286;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr279;
		} else
			goto tr279;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr279;
		} else if ( _widec >= 35 )
			goto tr279;
	} else
		goto tr279;
	goto st0;
tr285:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st81;
st81:
	if ( ++p == pe )
		goto _test_eof81;
case 81:
#line 6271 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st80;
	goto st0;
tr286:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st82;
st82:
	if ( ++p == pe )
		goto _test_eof82;
case 82:
#line 6290 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st80;
		case 525: goto st81;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st82;
	goto st0;
tr16:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st83;
tr1684:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st83;
st83:
	if ( ++p == pe )
		goto _test_eof83;
case 83:
#line 6347 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 78: goto st84;
		case 110: goto st84;
	}
	goto st0;
st84:
	if ( ++p == pe )
		goto _test_eof84;
case 84:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st85;
		case 32: goto st85;
		case 40: goto tr289;
		case 41: goto tr290;
		case 522: goto st446;
		case 525: goto st447;
		case 571: goto st448;
	}
	goto st0;
tr289:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st85;
tr290:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st85;
tr1670:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st85;
tr1671:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st85;
tr1672:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st85;
st85:
	if ( ++p == pe )
		goto _test_eof85;
case 85:
#line 6416 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st85;
		case 32: goto st85;
		case 40: goto tr289;
		case 41: goto tr290;
		case 65: goto st5;
		case 67: goto st16;
		case 68: goto st44;
		case 77: goto st88;
		case 78: goto st113;
		case 80: goto st216;
		case 83: goto st237;
		case 84: goto st393;
		case 97: goto st5;
		case 99: goto st16;
		case 100: goto st44;
		case 109: goto st88;
		case 110: goto st113;
		case 112: goto st216;
		case 115: goto st237;
		case 116: goto st393;
		case 522: goto st446;
		case 525: goto st447;
		case 571: goto st448;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr294;
	goto st0;
tr294:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st86;
tr1673:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st86;
st86:
	if ( ++p == pe )
		goto _test_eof86;
case 86:
#line 6480 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr295;
		case 32: goto tr295;
		case 40: goto tr296;
		case 41: goto tr297;
		case 68: goto tr299;
		case 72: goto tr299;
		case 77: goto tr299;
		case 87: goto tr299;
		case 100: goto tr299;
		case 104: goto tr299;
		case 109: goto tr299;
		case 119: goto tr299;
		case 522: goto tr300;
		case 525: goto tr301;
		case 571: goto tr302;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st86;
	goto st0;
tr304:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st87;
tr305:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st87;
tr295:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st87;
tr296:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st87;
tr297:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st87;
tr1650:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st87;
tr1651:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st87;
tr1652:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st87;
tr1664:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st87;
tr1665:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st87;
tr1666:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st87;
st87:
	if ( ++p == pe )
		goto _test_eof87;
case 87:
#line 6588 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st87;
		case 32: goto st87;
		case 40: goto tr304;
		case 41: goto tr305;
		case 65: goto st5;
		case 67: goto st16;
		case 68: goto st44;
		case 77: goto st88;
		case 78: goto st113;
		case 80: goto st216;
		case 83: goto st237;
		case 84: goto st393;
		case 97: goto st5;
		case 99: goto st16;
		case 100: goto st44;
		case 109: goto st88;
		case 110: goto st113;
		case 112: goto st216;
		case 115: goto st237;
		case 116: goto st393;
		case 522: goto st442;
		case 525: goto st443;
		case 571: goto st444;
	}
	goto st0;
tr17:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st88;
tr1656:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st88;
tr1685:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st88;
st88:
	if ( ++p == pe )
		goto _test_eof88;
case 88:
#line 6654 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 88: goto st89;
		case 120: goto st89;
	}
	goto st0;
st89:
	if ( ++p == pe )
		goto _test_eof89;
case 89:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st90;
		case 32: goto st90;
		case 40: goto tr311;
		case 41: goto tr312;
		case 522: goto st110;
		case 525: goto st111;
		case 571: goto st112;
	}
	goto st0;
tr311:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st90;
tr312:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st90;
tr400:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st90;
tr401:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st90;
tr402:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st90;
st90:
	if ( ++p == pe )
		goto _test_eof90;
case 90:
#line 6723 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st90;
		case 32: goto st90;
		case 40: goto tr311;
		case 41: goto tr312;
		case 522: goto st110;
		case 525: goto st111;
		case 571: goto st112;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr316;
	goto st0;
tr316:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st91;
tr403:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st91;
st91:
	if ( ++p == pe )
		goto _test_eof91;
case 91:
#line 6771 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr317;
		case 32: goto tr317;
		case 40: goto tr318;
		case 41: goto tr319;
		case 522: goto tr321;
		case 525: goto tr322;
		case 571: goto tr323;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st91;
	goto st0;
tr327:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st92;
tr328:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st92;
tr391:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st92;
tr393:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st92;
tr394:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st92;
tr317:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st92;
tr318:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st92;
tr319:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st92;
st92:
	if ( ++p == pe )
		goto _test_eof92;
case 92:
#line 6855 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st92;
		case 32: goto st92;
		case 34: goto tr326;
		case 40: goto tr327;
		case 41: goto tr328;
		case 64: goto tr329;
		case 92: goto tr330;
		case 522: goto st107;
		case 525: goto st108;
		case 571: goto st109;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr324;
		} else
			goto tr324;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr324;
		} else if ( _widec >= 37 )
			goto tr324;
	} else
		goto tr324;
	goto st0;
tr324:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st93;
tr348:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st93;
tr390:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st93;
st93:
	if ( ++p == pe )
		goto _test_eof93;
case 93:
#line 6922 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr335;
		case 32: goto tr335;
		case 40: goto tr336;
		case 41: goto tr337;
		case 92: goto st94;
		case 266: goto tr339;
		case 269: goto tr340;
		case 315: goto tr341;
		case 522: goto tr342;
		case 525: goto tr343;
		case 571: goto tr344;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st93;
		} else
			goto st93;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st93;
		} else if ( _widec >= 35 )
			goto st93;
	} else
		goto st93;
	goto st0;
tr330:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st94;
tr352:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st94;
tr396:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st94;
st94:
	if ( ++p == pe )
		goto _test_eof94;
case 94:
#line 6990 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st95;
		case 13: goto st96;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st97;
	goto st93;
st95:
	if ( ++p == pe )
		goto _test_eof95;
case 95:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr349;
		case 32: goto tr349;
		case 40: goto tr350;
		case 41: goto tr351;
		case 92: goto tr352;
		case 266: goto tr353;
		case 269: goto tr354;
		case 315: goto tr355;
		case 522: goto tr356;
		case 525: goto tr357;
		case 571: goto tr358;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr348;
		} else
			goto tr348;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr348;
		} else if ( _widec >= 35 )
			goto tr348;
	} else
		goto tr348;
	goto st0;
st96:
	if ( ++p == pe )
		goto _test_eof96;
case 96:
	if ( (*p) == 10 )
		goto st95;
	goto st0;
st97:
	if ( ++p == pe )
		goto _test_eof97;
case 97:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st98;
	goto st0;
st98:
	if ( ++p == pe )
		goto _test_eof98;
case 98:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st93;
	goto st0;
tr326:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st99;
tr365:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st99;
tr392:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st99;
st99:
	if ( ++p == pe )
		goto _test_eof99;
case 99:
#line 7093 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st100;
		case 13: goto st101;
		case 34: goto st102;
		case 92: goto st103;
	}
	goto st99;
tr366:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st100;
st100:
	if ( ++p == pe )
		goto _test_eof100;
case 100:
#line 7109 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr366;
		case 13: goto tr367;
		case 34: goto tr368;
		case 92: goto tr369;
	}
	goto tr365;
tr367:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st101;
st101:
	if ( ++p == pe )
		goto _test_eof101;
case 101:
#line 7125 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st100;
	goto st0;
tr368:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st102;
st102:
	if ( ++p == pe )
		goto _test_eof102;
case 102:
#line 7137 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr370;
		case 32: goto tr370;
		case 40: goto tr371;
		case 41: goto tr372;
		case 266: goto tr373;
		case 269: goto tr374;
		case 315: goto tr375;
		case 522: goto tr376;
		case 525: goto tr377;
		case 571: goto tr378;
	}
	goto st0;
tr369:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st103;
st103:
	if ( ++p == pe )
		goto _test_eof103;
case 103:
#line 7180 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st100;
		case 13: goto st101;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st104;
	goto st99;
st104:
	if ( ++p == pe )
		goto _test_eof104;
case 104:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st105;
	goto st0;
st105:
	if ( ++p == pe )
		goto _test_eof105;
case 105:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st99;
	goto st0;
tr329:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st106;
tr395:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st106;
st106:
	if ( ++p == pe )
		goto _test_eof106;
case 106:
#line 7219 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr381;
		case 32: goto tr381;
		case 40: goto tr382;
		case 41: goto tr383;
		case 266: goto tr384;
		case 269: goto tr385;
		case 315: goto tr386;
		case 522: goto tr387;
		case 525: goto tr388;
		case 571: goto tr389;
	}
	goto st0;
tr397:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st107;
tr321:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st107;
st107:
	if ( ++p == pe )
		goto _test_eof107;
case 107:
#line 7268 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr391;
		case 32: goto tr391;
		case 34: goto tr392;
		case 40: goto tr393;
		case 41: goto tr394;
		case 64: goto tr395;
		case 92: goto tr396;
		case 522: goto tr397;
		case 525: goto tr398;
		case 571: goto tr399;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr390;
		} else
			goto tr390;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr390;
		} else if ( _widec >= 37 )
			goto tr390;
	} else
		goto tr390;
	goto st0;
tr398:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st108;
tr322:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st108;
st108:
	if ( ++p == pe )
		goto _test_eof108;
case 108:
#line 7331 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st107;
	goto st0;
tr399:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st109;
tr323:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st109;
st109:
	if ( ++p == pe )
		goto _test_eof109;
case 109:
#line 7356 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st107;
		case 525: goto st108;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st109;
	goto st0;
tr404:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st110;
st110:
	if ( ++p == pe )
		goto _test_eof110;
case 110:
#line 7407 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr400;
		case 32: goto tr400;
		case 40: goto tr401;
		case 41: goto tr402;
		case 522: goto tr404;
		case 525: goto tr405;
		case 571: goto tr406;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr403;
	goto st0;
tr405:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st111;
st111:
	if ( ++p == pe )
		goto _test_eof111;
case 111:
#line 7449 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st110;
	goto st0;
tr406:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st112;
st112:
	if ( ++p == pe )
		goto _test_eof112;
case 112:
#line 7468 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st110;
		case 525: goto st111;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st112;
	goto st0;
tr18:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st113;
tr1657:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st113;
tr1686:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st113;
st113:
	if ( ++p == pe )
		goto _test_eof113;
case 113:
#line 7529 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 65: goto st114;
		case 83: goto st197;
		case 97: goto st114;
		case 115: goto st197;
	}
	goto st0;
st114:
	if ( ++p == pe )
		goto _test_eof114;
case 114:
	switch( (*p) ) {
		case 80: goto st115;
		case 112: goto st115;
	}
	goto st0;
st115:
	if ( ++p == pe )
		goto _test_eof115;
case 115:
	switch( (*p) ) {
		case 84: goto st116;
		case 116: goto st116;
	}
	goto st0;
st116:
	if ( ++p == pe )
		goto _test_eof116;
case 116:
	switch( (*p) ) {
		case 82: goto st117;
		case 114: goto st117;
	}
	goto st0;
st117:
	if ( ++p == pe )
		goto _test_eof117;
case 117:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st118;
		case 32: goto st118;
		case 40: goto tr413;
		case 41: goto tr414;
		case 522: goto st194;
		case 525: goto st195;
		case 571: goto st196;
	}
	goto st0;
tr413:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st118;
tr414:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st118;
tr691:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st118;
tr692:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st118;
tr693:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st118;
st118:
	if ( ++p == pe )
		goto _test_eof118;
case 118:
#line 7627 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st118;
		case 32: goto st118;
		case 40: goto tr413;
		case 41: goto tr414;
		case 522: goto st194;
		case 525: goto st195;
		case 571: goto st196;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr418;
	goto st0;
tr418:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st119;
tr694:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st119;
st119:
	if ( ++p == pe )
		goto _test_eof119;
case 119:
#line 7675 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr419;
		case 32: goto tr419;
		case 40: goto tr420;
		case 41: goto tr421;
		case 522: goto tr423;
		case 525: goto tr424;
		case 571: goto tr425;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st119;
	goto st0;
tr427:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st120;
tr428:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st120;
tr684:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st120;
tr685:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st120;
tr686:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st120;
tr419:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st120;
tr420:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st120;
tr421:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st120;
st120:
	if ( ++p == pe )
		goto _test_eof120;
case 120:
#line 7759 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st120;
		case 32: goto st120;
		case 40: goto tr427;
		case 41: goto tr428;
		case 522: goto st191;
		case 525: goto st192;
		case 571: goto st193;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr429;
	goto st0;
tr429:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st121;
tr687:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st121;
st121:
	if ( ++p == pe )
		goto _test_eof121;
case 121:
#line 7807 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr433;
		case 32: goto tr433;
		case 40: goto tr434;
		case 41: goto tr435;
		case 522: goto tr437;
		case 525: goto tr438;
		case 571: goto tr439;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st121;
	goto st0;
tr443:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st122;
tr444:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st122;
tr676:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st122;
tr678:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st122;
tr679:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st122;
tr433:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st122;
tr434:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st122;
tr435:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st122;
st122:
	if ( ++p == pe )
		goto _test_eof122;
case 122:
#line 7891 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st122;
		case 32: goto st122;
		case 34: goto tr442;
		case 40: goto tr443;
		case 41: goto tr444;
		case 92: goto tr445;
		case 522: goto st188;
		case 525: goto st189;
		case 571: goto st190;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr440;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr440;
		} else if ( _widec >= 14 )
			goto tr440;
	} else
		goto tr440;
	goto st0;
tr648:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st123;
tr440:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st123;
tr675:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st123;
st123:
	if ( ++p == pe )
		goto _test_eof123;
case 123:
#line 7958 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr450;
		case 32: goto tr450;
		case 40: goto tr451;
		case 41: goto tr452;
		case 92: goto st176;
		case 522: goto tr454;
		case 525: goto tr455;
		case 571: goto tr456;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st123;
		} else
			goto st123;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st123;
		} else if ( _widec >= 35 )
			goto st123;
	} else
		goto st123;
	goto st0;
tr460:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr461:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
tr637:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st124;
tr639:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr640:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
tr450:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st124;
tr451:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr452:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
tr649:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st124;
tr650:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr651:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
tr667:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st124;
tr668:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr669:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
st124:
	if ( ++p == pe )
		goto _test_eof124;
case 124:
#line 8087 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st124;
		case 32: goto st124;
		case 34: goto tr459;
		case 40: goto tr460;
		case 41: goto tr461;
		case 92: goto tr462;
		case 522: goto st173;
		case 525: goto st174;
		case 571: goto st175;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr457;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr457;
		} else if ( _widec >= 14 )
			goto tr457;
	} else
		goto tr457;
	goto st0;
tr457:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st125;
tr609:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st125;
tr636:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st125;
st125:
	if ( ++p == pe )
		goto _test_eof125;
case 125:
#line 8150 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr467;
		case 32: goto tr467;
		case 40: goto tr468;
		case 41: goto tr469;
		case 92: goto st161;
		case 522: goto tr471;
		case 525: goto tr472;
		case 571: goto tr473;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st125;
		} else
			goto st125;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st125;
		} else if ( _widec >= 35 )
			goto st125;
	} else
		goto st125;
	goto st0;
tr477:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr478:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
tr598:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st126;
tr600:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr601:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
tr467:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st126;
tr468:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr469:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
tr610:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st126;
tr611:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr612:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
tr628:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st126;
tr629:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr630:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
st126:
	if ( ++p == pe )
		goto _test_eof126;
case 126:
#line 8279 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st126;
		case 32: goto st126;
		case 34: goto tr476;
		case 40: goto tr477;
		case 41: goto tr478;
		case 92: goto tr479;
		case 522: goto st158;
		case 525: goto st159;
		case 571: goto st160;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr474;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr474;
		} else if ( _widec >= 14 )
			goto tr474;
	} else
		goto tr474;
	goto st0;
tr474:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st127;
tr570:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st127;
tr597:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st127;
st127:
	if ( ++p == pe )
		goto _test_eof127;
case 127:
#line 8342 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr484;
		case 32: goto tr484;
		case 40: goto tr485;
		case 41: goto tr486;
		case 92: goto st146;
		case 522: goto tr488;
		case 525: goto tr489;
		case 571: goto tr490;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st127;
		} else
			goto st127;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st127;
		} else if ( _widec >= 35 )
			goto st127;
	} else
		goto st127;
	goto st0;
tr494:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr495:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
tr558:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st128;
tr560:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr561:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
tr484:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st128;
tr485:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr486:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
tr571:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st128;
tr572:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr573:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
tr589:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st128;
tr590:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr591:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
st128:
	if ( ++p == pe )
		goto _test_eof128;
case 128:
#line 8471 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st128;
		case 32: goto st128;
		case 34: goto tr493;
		case 40: goto tr494;
		case 41: goto tr495;
		case 64: goto tr496;
		case 92: goto tr497;
		case 522: goto st143;
		case 525: goto st144;
		case 571: goto st145;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr491;
		} else
			goto tr491;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr491;
		} else if ( _widec >= 37 )
			goto tr491;
	} else
		goto tr491;
	goto st0;
tr491:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st129;
tr515:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st129;
tr557:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st129;
st129:
	if ( ++p == pe )
		goto _test_eof129;
case 129:
#line 8538 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr502;
		case 32: goto tr502;
		case 40: goto tr503;
		case 41: goto tr504;
		case 92: goto st130;
		case 266: goto tr506;
		case 269: goto tr507;
		case 315: goto tr508;
		case 522: goto tr509;
		case 525: goto tr510;
		case 571: goto tr511;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st129;
		} else
			goto st129;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st129;
		} else if ( _widec >= 35 )
			goto st129;
	} else
		goto st129;
	goto st0;
tr497:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st130;
tr519:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st130;
tr563:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st130;
st130:
	if ( ++p == pe )
		goto _test_eof130;
case 130:
#line 8606 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st131;
		case 13: goto st132;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st133;
	goto st129;
st131:
	if ( ++p == pe )
		goto _test_eof131;
case 131:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr516;
		case 32: goto tr516;
		case 40: goto tr517;
		case 41: goto tr518;
		case 92: goto tr519;
		case 266: goto tr520;
		case 269: goto tr521;
		case 315: goto tr522;
		case 522: goto tr523;
		case 525: goto tr524;
		case 571: goto tr525;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr515;
		} else
			goto tr515;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr515;
		} else if ( _widec >= 35 )
			goto tr515;
	} else
		goto tr515;
	goto st0;
st132:
	if ( ++p == pe )
		goto _test_eof132;
case 132:
	if ( (*p) == 10 )
		goto st131;
	goto st0;
st133:
	if ( ++p == pe )
		goto _test_eof133;
case 133:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st134;
	goto st0;
st134:
	if ( ++p == pe )
		goto _test_eof134;
case 134:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st129;
	goto st0;
tr493:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st135;
tr532:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st135;
tr559:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st135;
st135:
	if ( ++p == pe )
		goto _test_eof135;
case 135:
#line 8709 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st136;
		case 13: goto st137;
		case 34: goto st138;
		case 92: goto st139;
	}
	goto st135;
tr533:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st136;
st136:
	if ( ++p == pe )
		goto _test_eof136;
case 136:
#line 8725 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr533;
		case 13: goto tr534;
		case 34: goto tr535;
		case 92: goto tr536;
	}
	goto tr532;
tr534:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st137;
st137:
	if ( ++p == pe )
		goto _test_eof137;
case 137:
#line 8741 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st136;
	goto st0;
tr535:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st138;
st138:
	if ( ++p == pe )
		goto _test_eof138;
case 138:
#line 8753 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr537;
		case 32: goto tr537;
		case 40: goto tr538;
		case 41: goto tr539;
		case 266: goto tr540;
		case 269: goto tr541;
		case 315: goto tr542;
		case 522: goto tr543;
		case 525: goto tr544;
		case 571: goto tr545;
	}
	goto st0;
tr536:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st139;
st139:
	if ( ++p == pe )
		goto _test_eof139;
case 139:
#line 8796 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st136;
		case 13: goto st137;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st140;
	goto st135;
st140:
	if ( ++p == pe )
		goto _test_eof140;
case 140:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st141;
	goto st0;
st141:
	if ( ++p == pe )
		goto _test_eof141;
case 141:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st135;
	goto st0;
tr496:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st142;
tr562:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st142;
st142:
	if ( ++p == pe )
		goto _test_eof142;
case 142:
#line 8835 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr548;
		case 32: goto tr548;
		case 40: goto tr549;
		case 41: goto tr550;
		case 266: goto tr551;
		case 269: goto tr552;
		case 315: goto tr553;
		case 522: goto tr554;
		case 525: goto tr555;
		case 571: goto tr556;
	}
	goto st0;
tr564:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st143;
tr488:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st143;
tr575:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st143;
tr592:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st143;
st143:
	if ( ++p == pe )
		goto _test_eof143;
case 143:
#line 8892 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr558;
		case 32: goto tr558;
		case 34: goto tr559;
		case 40: goto tr560;
		case 41: goto tr561;
		case 64: goto tr562;
		case 92: goto tr563;
		case 522: goto tr564;
		case 525: goto tr565;
		case 571: goto tr566;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr557;
		} else
			goto tr557;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr557;
		} else if ( _widec >= 37 )
			goto tr557;
	} else
		goto tr557;
	goto st0;
tr565:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st144;
tr489:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st144;
tr576:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st144;
tr593:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st144;
st144:
	if ( ++p == pe )
		goto _test_eof144;
case 144:
#line 8963 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st143;
	goto st0;
tr566:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st145;
tr490:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st145;
tr577:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st145;
tr594:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st145;
st145:
	if ( ++p == pe )
		goto _test_eof145;
case 145:
#line 8996 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st143;
		case 525: goto st144;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st145;
	goto st0;
tr479:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st146;
tr574:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st146;
tr602:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st146;
st146:
	if ( ++p == pe )
		goto _test_eof146;
case 146:
#line 9057 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st147;
		case 13: goto st148;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st149;
	goto st127;
st147:
	if ( ++p == pe )
		goto _test_eof147;
case 147:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr571;
		case 32: goto tr571;
		case 40: goto tr572;
		case 41: goto tr573;
		case 92: goto tr574;
		case 522: goto tr575;
		case 525: goto tr576;
		case 571: goto tr577;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr570;
		} else
			goto tr570;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr570;
		} else if ( _widec >= 35 )
			goto tr570;
	} else
		goto tr570;
	goto st0;
st148:
	if ( ++p == pe )
		goto _test_eof148;
case 148:
	if ( (*p) == 10 )
		goto st147;
	goto st0;
st149:
	if ( ++p == pe )
		goto _test_eof149;
case 149:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st150;
	goto st0;
st150:
	if ( ++p == pe )
		goto _test_eof150;
case 150:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st127;
	goto st0;
tr476:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st151;
tr584:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st151;
tr599:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st151;
st151:
	if ( ++p == pe )
		goto _test_eof151;
case 151:
#line 9157 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st152;
		case 13: goto st153;
		case 34: goto st154;
		case 92: goto st155;
	}
	goto st151;
tr585:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st152;
st152:
	if ( ++p == pe )
		goto _test_eof152;
case 152:
#line 9173 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr585;
		case 13: goto tr586;
		case 34: goto tr587;
		case 92: goto tr588;
	}
	goto tr584;
tr586:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st153;
st153:
	if ( ++p == pe )
		goto _test_eof153;
case 153:
#line 9189 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st152;
	goto st0;
tr587:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st154;
st154:
	if ( ++p == pe )
		goto _test_eof154;
case 154:
#line 9201 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr589;
		case 32: goto tr589;
		case 40: goto tr590;
		case 41: goto tr591;
		case 522: goto tr592;
		case 525: goto tr593;
		case 571: goto tr594;
	}
	goto st0;
tr588:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st155;
st155:
	if ( ++p == pe )
		goto _test_eof155;
case 155:
#line 9241 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st152;
		case 13: goto st153;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st156;
	goto st151;
st156:
	if ( ++p == pe )
		goto _test_eof156;
case 156:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st157;
	goto st0;
st157:
	if ( ++p == pe )
		goto _test_eof157;
case 157:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st151;
	goto st0;
tr603:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st158;
tr471:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st158;
tr614:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st158;
tr631:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st158;
st158:
	if ( ++p == pe )
		goto _test_eof158;
case 158:
#line 9288 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr598;
		case 32: goto tr598;
		case 34: goto tr599;
		case 40: goto tr600;
		case 41: goto tr601;
		case 92: goto tr602;
		case 522: goto tr603;
		case 525: goto tr604;
		case 571: goto tr605;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr597;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr597;
		} else if ( _widec >= 14 )
			goto tr597;
	} else
		goto tr597;
	goto st0;
tr604:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st159;
tr472:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st159;
tr615:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st159;
tr632:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st159;
st159:
	if ( ++p == pe )
		goto _test_eof159;
case 159:
#line 9355 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st158;
	goto st0;
tr605:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st160;
tr473:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st160;
tr616:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st160;
tr633:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st160;
st160:
	if ( ++p == pe )
		goto _test_eof160;
case 160:
#line 9388 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st158;
		case 525: goto st159;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st160;
	goto st0;
tr462:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st161;
tr613:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st161;
tr641:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st161;
st161:
	if ( ++p == pe )
		goto _test_eof161;
case 161:
#line 9449 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st162;
		case 13: goto st163;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st164;
	goto st125;
st162:
	if ( ++p == pe )
		goto _test_eof162;
case 162:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr610;
		case 32: goto tr610;
		case 40: goto tr611;
		case 41: goto tr612;
		case 92: goto tr613;
		case 522: goto tr614;
		case 525: goto tr615;
		case 571: goto tr616;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr609;
		} else
			goto tr609;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr609;
		} else if ( _widec >= 35 )
			goto tr609;
	} else
		goto tr609;
	goto st0;
st163:
	if ( ++p == pe )
		goto _test_eof163;
case 163:
	if ( (*p) == 10 )
		goto st162;
	goto st0;
st164:
	if ( ++p == pe )
		goto _test_eof164;
case 164:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st165;
	goto st0;
st165:
	if ( ++p == pe )
		goto _test_eof165;
case 165:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st125;
	goto st0;
tr459:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st166;
tr623:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st166;
tr638:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st166;
st166:
	if ( ++p == pe )
		goto _test_eof166;
case 166:
#line 9549 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st167;
		case 13: goto st168;
		case 34: goto st169;
		case 92: goto st170;
	}
	goto st166;
tr624:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st167;
st167:
	if ( ++p == pe )
		goto _test_eof167;
case 167:
#line 9565 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr624;
		case 13: goto tr625;
		case 34: goto tr626;
		case 92: goto tr627;
	}
	goto tr623;
tr625:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st168;
st168:
	if ( ++p == pe )
		goto _test_eof168;
case 168:
#line 9581 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st167;
	goto st0;
tr626:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st169;
st169:
	if ( ++p == pe )
		goto _test_eof169;
case 169:
#line 9593 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr628;
		case 32: goto tr628;
		case 40: goto tr629;
		case 41: goto tr630;
		case 522: goto tr631;
		case 525: goto tr632;
		case 571: goto tr633;
	}
	goto st0;
tr627:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st170;
st170:
	if ( ++p == pe )
		goto _test_eof170;
case 170:
#line 9633 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st167;
		case 13: goto st168;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st171;
	goto st166;
st171:
	if ( ++p == pe )
		goto _test_eof171;
case 171:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st172;
	goto st0;
st172:
	if ( ++p == pe )
		goto _test_eof172;
case 172:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st166;
	goto st0;
tr642:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st173;
tr454:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st173;
tr653:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st173;
tr670:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st173;
st173:
	if ( ++p == pe )
		goto _test_eof173;
case 173:
#line 9680 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr637;
		case 32: goto tr637;
		case 34: goto tr638;
		case 40: goto tr639;
		case 41: goto tr640;
		case 92: goto tr641;
		case 522: goto tr642;
		case 525: goto tr643;
		case 571: goto tr644;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr636;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr636;
		} else if ( _widec >= 14 )
			goto tr636;
	} else
		goto tr636;
	goto st0;
tr643:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st174;
tr455:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st174;
tr654:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st174;
tr671:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st174;
st174:
	if ( ++p == pe )
		goto _test_eof174;
case 174:
#line 9747 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st173;
	goto st0;
tr644:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st175;
tr456:
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st175;
tr655:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 506 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st175;
tr672:
#line 507 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st175;
st175:
	if ( ++p == pe )
		goto _test_eof175;
case 175:
#line 9780 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st173;
		case 525: goto st174;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st175;
	goto st0;
tr652:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st176;
tr445:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st176;
tr680:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st176;
st176:
	if ( ++p == pe )
		goto _test_eof176;
case 176:
#line 9845 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st177;
		case 13: goto st178;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st179;
	goto st123;
st177:
	if ( ++p == pe )
		goto _test_eof177;
case 177:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr649;
		case 32: goto tr649;
		case 40: goto tr650;
		case 41: goto tr651;
		case 92: goto tr652;
		case 522: goto tr653;
		case 525: goto tr654;
		case 571: goto tr655;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr648;
		} else
			goto tr648;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr648;
		} else if ( _widec >= 35 )
			goto tr648;
	} else
		goto tr648;
	goto st0;
st178:
	if ( ++p == pe )
		goto _test_eof178;
case 178:
	if ( (*p) == 10 )
		goto st177;
	goto st0;
st179:
	if ( ++p == pe )
		goto _test_eof179;
case 179:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st180;
	goto st0;
st180:
	if ( ++p == pe )
		goto _test_eof180;
case 180:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st123;
	goto st0;
tr662:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st181;
tr442:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st181;
tr677:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st181;
st181:
	if ( ++p == pe )
		goto _test_eof181;
case 181:
#line 9949 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st182;
		case 13: goto st183;
		case 34: goto st184;
		case 92: goto st185;
	}
	goto st181;
tr663:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st182;
st182:
	if ( ++p == pe )
		goto _test_eof182;
case 182:
#line 9965 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr663;
		case 13: goto tr664;
		case 34: goto tr665;
		case 92: goto tr666;
	}
	goto tr662;
tr664:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st183;
st183:
	if ( ++p == pe )
		goto _test_eof183;
case 183:
#line 9981 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st182;
	goto st0;
tr665:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st184;
st184:
	if ( ++p == pe )
		goto _test_eof184;
case 184:
#line 9993 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr667;
		case 32: goto tr667;
		case 40: goto tr668;
		case 41: goto tr669;
		case 522: goto tr670;
		case 525: goto tr671;
		case 571: goto tr672;
	}
	goto st0;
tr666:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st185;
st185:
	if ( ++p == pe )
		goto _test_eof185;
case 185:
#line 10033 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st182;
		case 13: goto st183;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st186;
	goto st181;
st186:
	if ( ++p == pe )
		goto _test_eof186;
case 186:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st187;
	goto st0;
st187:
	if ( ++p == pe )
		goto _test_eof187;
case 187:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st181;
	goto st0;
tr681:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st188;
tr437:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st188;
st188:
	if ( ++p == pe )
		goto _test_eof188;
case 188:
#line 10072 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr676;
		case 32: goto tr676;
		case 34: goto tr677;
		case 40: goto tr678;
		case 41: goto tr679;
		case 92: goto tr680;
		case 522: goto tr681;
		case 525: goto tr682;
		case 571: goto tr683;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr675;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr675;
		} else if ( _widec >= 14 )
			goto tr675;
	} else
		goto tr675;
	goto st0;
tr682:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st189;
tr438:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st189;
st189:
	if ( ++p == pe )
		goto _test_eof189;
case 189:
#line 10131 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st188;
	goto st0;
tr683:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st190;
tr439:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st190;
st190:
	if ( ++p == pe )
		goto _test_eof190;
case 190:
#line 10156 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st188;
		case 525: goto st189;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st190;
	goto st0;
tr688:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st191;
tr423:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st191;
st191:
	if ( ++p == pe )
		goto _test_eof191;
case 191:
#line 10213 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr684;
		case 32: goto tr684;
		case 40: goto tr685;
		case 41: goto tr686;
		case 522: goto tr688;
		case 525: goto tr689;
		case 571: goto tr690;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr687;
	goto st0;
tr689:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st192;
tr424:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st192;
st192:
	if ( ++p == pe )
		goto _test_eof192;
case 192:
#line 10261 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st191;
	goto st0;
tr690:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st193;
tr425:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st193;
st193:
	if ( ++p == pe )
		goto _test_eof193;
case 193:
#line 10286 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st191;
		case 525: goto st192;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st193;
	goto st0;
tr695:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st194;
st194:
	if ( ++p == pe )
		goto _test_eof194;
case 194:
#line 10337 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr691;
		case 32: goto tr691;
		case 40: goto tr692;
		case 41: goto tr693;
		case 522: goto tr695;
		case 525: goto tr696;
		case 571: goto tr697;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr694;
	goto st0;
tr696:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st195;
st195:
	if ( ++p == pe )
		goto _test_eof195;
case 195:
#line 10379 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st194;
	goto st0;
tr697:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st196;
st196:
	if ( ++p == pe )
		goto _test_eof196;
case 196:
#line 10398 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st194;
		case 525: goto st195;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st196;
	goto st0;
st197:
	if ( ++p == pe )
		goto _test_eof197;
case 197:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st198;
		case 32: goto st198;
		case 40: goto tr699;
		case 41: goto tr700;
		case 522: goto st213;
		case 525: goto st214;
		case 571: goto st215;
	}
	goto st0;
tr699:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st198;
tr700:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st198;
tr765:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st198;
tr767:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st198;
tr768:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st198;
st198:
	if ( ++p == pe )
		goto _test_eof198;
case 198:
#line 10504 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st198;
		case 32: goto st198;
		case 34: goto tr705;
		case 40: goto tr699;
		case 41: goto tr700;
		case 64: goto tr706;
		case 92: goto tr707;
		case 522: goto st213;
		case 525: goto st214;
		case 571: goto st215;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr704;
		} else
			goto tr704;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr704;
		} else if ( _widec >= 37 )
			goto tr704;
	} else
		goto tr704;
	goto st0;
tr704:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st199;
tr722:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st199;
tr764:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st199;
st199:
	if ( ++p == pe )
		goto _test_eof199;
case 199:
#line 10571 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr709;
		case 32: goto tr709;
		case 40: goto tr710;
		case 41: goto tr711;
		case 92: goto st200;
		case 266: goto tr713;
		case 269: goto tr714;
		case 315: goto tr715;
		case 522: goto tr716;
		case 525: goto tr717;
		case 571: goto tr718;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st199;
		} else
			goto st199;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st199;
		} else if ( _widec >= 35 )
			goto st199;
	} else
		goto st199;
	goto st0;
tr707:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st200;
tr726:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st200;
tr770:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st200;
st200:
	if ( ++p == pe )
		goto _test_eof200;
case 200:
#line 10639 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st201;
		case 13: goto st202;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st203;
	goto st199;
st201:
	if ( ++p == pe )
		goto _test_eof201;
case 201:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr723;
		case 32: goto tr723;
		case 40: goto tr724;
		case 41: goto tr725;
		case 92: goto tr726;
		case 266: goto tr727;
		case 269: goto tr728;
		case 315: goto tr729;
		case 522: goto tr730;
		case 525: goto tr731;
		case 571: goto tr732;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr722;
		} else
			goto tr722;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr722;
		} else if ( _widec >= 35 )
			goto tr722;
	} else
		goto tr722;
	goto st0;
st202:
	if ( ++p == pe )
		goto _test_eof202;
case 202:
	if ( (*p) == 10 )
		goto st201;
	goto st0;
st203:
	if ( ++p == pe )
		goto _test_eof203;
case 203:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st204;
	goto st0;
st204:
	if ( ++p == pe )
		goto _test_eof204;
case 204:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st199;
	goto st0;
tr705:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st205;
tr739:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st205;
tr766:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st205;
st205:
	if ( ++p == pe )
		goto _test_eof205;
case 205:
#line 10742 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st206;
		case 13: goto st207;
		case 34: goto st208;
		case 92: goto st209;
	}
	goto st205;
tr740:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st206;
st206:
	if ( ++p == pe )
		goto _test_eof206;
case 206:
#line 10758 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr740;
		case 13: goto tr741;
		case 34: goto tr742;
		case 92: goto tr743;
	}
	goto tr739;
tr741:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st207;
st207:
	if ( ++p == pe )
		goto _test_eof207;
case 207:
#line 10774 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st206;
	goto st0;
tr742:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st208;
st208:
	if ( ++p == pe )
		goto _test_eof208;
case 208:
#line 10786 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr744;
		case 32: goto tr744;
		case 40: goto tr745;
		case 41: goto tr746;
		case 266: goto tr747;
		case 269: goto tr748;
		case 315: goto tr749;
		case 522: goto tr750;
		case 525: goto tr751;
		case 571: goto tr752;
	}
	goto st0;
tr743:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st209;
st209:
	if ( ++p == pe )
		goto _test_eof209;
case 209:
#line 10829 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st206;
		case 13: goto st207;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st210;
	goto st205;
st210:
	if ( ++p == pe )
		goto _test_eof210;
case 210:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st211;
	goto st0;
st211:
	if ( ++p == pe )
		goto _test_eof211;
case 211:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st205;
	goto st0;
tr706:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st212;
tr769:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st212;
st212:
	if ( ++p == pe )
		goto _test_eof212;
case 212:
#line 10868 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr755;
		case 32: goto tr755;
		case 40: goto tr756;
		case 41: goto tr757;
		case 266: goto tr758;
		case 269: goto tr759;
		case 315: goto tr760;
		case 522: goto tr761;
		case 525: goto tr762;
		case 571: goto tr763;
	}
	goto st0;
tr771:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st213;
st213:
	if ( ++p == pe )
		goto _test_eof213;
case 213:
#line 10911 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr765;
		case 32: goto tr765;
		case 34: goto tr766;
		case 40: goto tr767;
		case 41: goto tr768;
		case 64: goto tr769;
		case 92: goto tr770;
		case 522: goto tr771;
		case 525: goto tr772;
		case 571: goto tr773;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr764;
		} else
			goto tr764;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr764;
		} else if ( _widec >= 37 )
			goto tr764;
	} else
		goto tr764;
	goto st0;
tr772:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st214;
st214:
	if ( ++p == pe )
		goto _test_eof214;
case 214:
#line 10968 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st213;
	goto st0;
tr773:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st215;
st215:
	if ( ++p == pe )
		goto _test_eof215;
case 215:
#line 10987 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st213;
		case 525: goto st214;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st215;
	goto st0;
tr19:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st216;
tr1658:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st216;
tr1687:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st216;
st216:
	if ( ++p == pe )
		goto _test_eof216;
case 216:
#line 11048 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 84: goto st217;
		case 116: goto st217;
	}
	goto st0;
st217:
	if ( ++p == pe )
		goto _test_eof217;
case 217:
	switch( (*p) ) {
		case 82: goto st218;
		case 114: goto st218;
	}
	goto st0;
st218:
	if ( ++p == pe )
		goto _test_eof218;
case 218:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st219;
		case 32: goto st219;
		case 40: goto tr777;
		case 41: goto tr778;
		case 522: goto st234;
		case 525: goto st235;
		case 571: goto st236;
	}
	goto st0;
tr777:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st219;
tr778:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st219;
tr843:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st219;
tr845:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st219;
tr846:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st219;
st219:
	if ( ++p == pe )
		goto _test_eof219;
case 219:
#line 11126 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st219;
		case 32: goto st219;
		case 34: goto tr783;
		case 40: goto tr777;
		case 41: goto tr778;
		case 64: goto tr784;
		case 92: goto tr785;
		case 522: goto st234;
		case 525: goto st235;
		case 571: goto st236;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr782;
		} else
			goto tr782;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr782;
		} else if ( _widec >= 37 )
			goto tr782;
	} else
		goto tr782;
	goto st0;
tr782:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st220;
tr800:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st220;
tr842:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st220;
st220:
	if ( ++p == pe )
		goto _test_eof220;
case 220:
#line 11193 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr787;
		case 32: goto tr787;
		case 40: goto tr788;
		case 41: goto tr789;
		case 92: goto st221;
		case 266: goto tr791;
		case 269: goto tr792;
		case 315: goto tr793;
		case 522: goto tr794;
		case 525: goto tr795;
		case 571: goto tr796;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st220;
		} else
			goto st220;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st220;
		} else if ( _widec >= 35 )
			goto st220;
	} else
		goto st220;
	goto st0;
tr785:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st221;
tr804:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st221;
tr848:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st221;
st221:
	if ( ++p == pe )
		goto _test_eof221;
case 221:
#line 11261 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st222;
		case 13: goto st223;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st224;
	goto st220;
st222:
	if ( ++p == pe )
		goto _test_eof222;
case 222:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr801;
		case 32: goto tr801;
		case 40: goto tr802;
		case 41: goto tr803;
		case 92: goto tr804;
		case 266: goto tr805;
		case 269: goto tr806;
		case 315: goto tr807;
		case 522: goto tr808;
		case 525: goto tr809;
		case 571: goto tr810;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr800;
		} else
			goto tr800;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr800;
		} else if ( _widec >= 35 )
			goto tr800;
	} else
		goto tr800;
	goto st0;
st223:
	if ( ++p == pe )
		goto _test_eof223;
case 223:
	if ( (*p) == 10 )
		goto st222;
	goto st0;
st224:
	if ( ++p == pe )
		goto _test_eof224;
case 224:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st225;
	goto st0;
st225:
	if ( ++p == pe )
		goto _test_eof225;
case 225:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st220;
	goto st0;
tr783:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st226;
tr817:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st226;
tr844:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st226;
st226:
	if ( ++p == pe )
		goto _test_eof226;
case 226:
#line 11364 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st227;
		case 13: goto st228;
		case 34: goto st229;
		case 92: goto st230;
	}
	goto st226;
tr818:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st227;
st227:
	if ( ++p == pe )
		goto _test_eof227;
case 227:
#line 11380 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr818;
		case 13: goto tr819;
		case 34: goto tr820;
		case 92: goto tr821;
	}
	goto tr817;
tr819:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st228;
st228:
	if ( ++p == pe )
		goto _test_eof228;
case 228:
#line 11396 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st227;
	goto st0;
tr820:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st229;
st229:
	if ( ++p == pe )
		goto _test_eof229;
case 229:
#line 11408 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr822;
		case 32: goto tr822;
		case 40: goto tr823;
		case 41: goto tr824;
		case 266: goto tr825;
		case 269: goto tr826;
		case 315: goto tr827;
		case 522: goto tr828;
		case 525: goto tr829;
		case 571: goto tr830;
	}
	goto st0;
tr821:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st230;
st230:
	if ( ++p == pe )
		goto _test_eof230;
case 230:
#line 11451 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st227;
		case 13: goto st228;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st231;
	goto st226;
st231:
	if ( ++p == pe )
		goto _test_eof231;
case 231:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st232;
	goto st0;
st232:
	if ( ++p == pe )
		goto _test_eof232;
case 232:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st226;
	goto st0;
tr784:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st233;
tr847:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st233;
st233:
	if ( ++p == pe )
		goto _test_eof233;
case 233:
#line 11490 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr833;
		case 32: goto tr833;
		case 40: goto tr834;
		case 41: goto tr835;
		case 266: goto tr836;
		case 269: goto tr837;
		case 315: goto tr838;
		case 522: goto tr839;
		case 525: goto tr840;
		case 571: goto tr841;
	}
	goto st0;
tr849:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st234;
st234:
	if ( ++p == pe )
		goto _test_eof234;
case 234:
#line 11533 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr843;
		case 32: goto tr843;
		case 34: goto tr844;
		case 40: goto tr845;
		case 41: goto tr846;
		case 64: goto tr847;
		case 92: goto tr848;
		case 522: goto tr849;
		case 525: goto tr850;
		case 571: goto tr851;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr842;
		} else
			goto tr842;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr842;
		} else if ( _widec >= 37 )
			goto tr842;
	} else
		goto tr842;
	goto st0;
tr850:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st235;
st235:
	if ( ++p == pe )
		goto _test_eof235;
case 235:
#line 11590 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st234;
	goto st0;
tr851:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st236;
st236:
	if ( ++p == pe )
		goto _test_eof236;
case 236:
#line 11609 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st234;
		case 525: goto st235;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st236;
	goto st0;
tr20:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st237;
tr1659:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st237;
tr1688:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st237;
st237:
	if ( ++p == pe )
		goto _test_eof237;
case 237:
#line 11670 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 79: goto st238;
		case 80: goto st305;
		case 82: goto st358;
		case 111: goto st238;
		case 112: goto st305;
		case 114: goto st358;
	}
	goto st0;
st238:
	if ( ++p == pe )
		goto _test_eof238;
case 238:
	switch( (*p) ) {
		case 65: goto st239;
		case 97: goto st239;
	}
	goto st0;
st239:
	if ( ++p == pe )
		goto _test_eof239;
case 239:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st240;
		case 32: goto st240;
		case 40: goto tr857;
		case 41: goto tr858;
		case 522: goto st302;
		case 525: goto st303;
		case 571: goto st304;
	}
	goto st0;
tr857:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st240;
tr858:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st240;
tr1113:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st240;
tr1115:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st240;
tr1116:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st240;
st240:
	if ( ++p == pe )
		goto _test_eof240;
case 240:
#line 11752 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st240;
		case 32: goto st240;
		case 34: goto tr863;
		case 40: goto tr857;
		case 41: goto tr858;
		case 64: goto tr864;
		case 92: goto tr865;
		case 522: goto st302;
		case 525: goto st303;
		case 571: goto st304;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr862;
		} else
			goto tr862;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr862;
		} else if ( _widec >= 37 )
			goto tr862;
	} else
		goto tr862;
	goto st0;
tr862:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st241;
tr1079:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st241;
tr1112:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st241;
st241:
	if ( ++p == pe )
		goto _test_eof241;
case 241:
#line 11819 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr867;
		case 32: goto tr867;
		case 40: goto tr868;
		case 41: goto tr869;
		case 92: goto st289;
		case 522: goto tr871;
		case 525: goto tr872;
		case 571: goto tr873;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st241;
		} else
			goto st241;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st241;
		} else if ( _widec >= 35 )
			goto st241;
	} else
		goto st241;
	goto st0;
tr877:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr878:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr1068:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st242;
tr1070:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr1071:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr867:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st242;
tr868:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr869:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr1080:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st242;
tr1081:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr1082:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr1098:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
	goto st242;
tr1099:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr1100:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr1106:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
	goto st242;
tr1107:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr1108:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
st242:
	if ( ++p == pe )
		goto _test_eof242;
case 242:
#line 11964 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st242;
		case 32: goto st242;
		case 34: goto tr876;
		case 40: goto tr877;
		case 41: goto tr878;
		case 92: goto tr879;
		case 522: goto st286;
		case 525: goto st287;
		case 571: goto st288;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr874;
		} else
			goto tr874;
	} else if ( _widec > 35 ) {
		if ( _widec < 60 ) {
			if ( 37 <= _widec && _widec <= 58 )
				goto tr874;
		} else if ( _widec > 63 ) {
			if ( 65 <= _widec )
				goto tr874;
		} else
			goto tr874;
	} else
		goto tr874;
	goto st0;
tr874:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st243;
tr1040:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st243;
tr1067:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st243;
st243:
	if ( ++p == pe )
		goto _test_eof243;
case 243:
#line 12033 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr884;
		case 32: goto tr884;
		case 40: goto tr885;
		case 41: goto tr886;
		case 92: goto st274;
		case 522: goto tr888;
		case 525: goto tr889;
		case 571: goto tr890;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st243;
		} else
			goto st243;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st243;
		} else if ( _widec >= 35 )
			goto st243;
	} else
		goto st243;
	goto st0;
tr892:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr893:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
tr1030:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st244;
tr1031:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr1032:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
tr884:
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st244;
tr885:
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr886:
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
tr1041:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st244;
tr1042:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr1043:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
tr1059:
#line 496 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
	goto st244;
tr1060:
#line 496 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr1061:
#line 496 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
st244:
	if ( ++p == pe )
		goto _test_eof244;
case 244:
#line 12162 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st244;
		case 32: goto st244;
		case 40: goto tr892;
		case 41: goto tr893;
		case 522: goto st271;
		case 525: goto st272;
		case 571: goto st273;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr894;
	goto st0;
tr894:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st245;
tr1033:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st245;
st245:
	if ( ++p == pe )
		goto _test_eof245;
case 245:
#line 12210 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr898;
		case 32: goto tr898;
		case 40: goto tr899;
		case 41: goto tr900;
		case 68: goto tr902;
		case 72: goto tr902;
		case 77: goto tr902;
		case 87: goto tr902;
		case 100: goto tr902;
		case 104: goto tr902;
		case 109: goto tr902;
		case 119: goto tr902;
		case 522: goto tr903;
		case 525: goto tr904;
		case 571: goto tr905;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st245;
	goto st0;
tr907:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st246;
tr908:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st246;
tr1017:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st246;
tr1018:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st246;
tr1019:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st246;
tr898:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st246;
tr899:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st246;
tr900:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st246;
tr1024:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st246;
tr1025:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st246;
tr1026:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st246;
st246:
	if ( ++p == pe )
		goto _test_eof246;
case 246:
#line 12318 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st246;
		case 32: goto st246;
		case 40: goto tr907;
		case 41: goto tr908;
		case 522: goto st267;
		case 525: goto st268;
		case 571: goto st269;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr909;
	goto st0;
tr909:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st247;
tr1020:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st247;
st247:
	if ( ++p == pe )
		goto _test_eof247;
case 247:
#line 12366 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr913;
		case 32: goto tr913;
		case 40: goto tr914;
		case 41: goto tr915;
		case 68: goto tr917;
		case 72: goto tr917;
		case 77: goto tr917;
		case 87: goto tr917;
		case 100: goto tr917;
		case 104: goto tr917;
		case 109: goto tr917;
		case 119: goto tr917;
		case 522: goto tr918;
		case 525: goto tr919;
		case 571: goto tr920;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st247;
	goto st0;
tr922:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st248;
tr923:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st248;
tr1004:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st248;
tr1005:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st248;
tr1006:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st248;
tr913:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st248;
tr914:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st248;
tr915:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st248;
tr1011:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st248;
tr1012:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st248;
tr1013:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st248;
st248:
	if ( ++p == pe )
		goto _test_eof248;
case 248:
#line 12474 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st248;
		case 32: goto st248;
		case 40: goto tr922;
		case 41: goto tr923;
		case 522: goto st263;
		case 525: goto st264;
		case 571: goto st265;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr924;
	goto st0;
tr924:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st249;
tr1007:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st249;
st249:
	if ( ++p == pe )
		goto _test_eof249;
case 249:
#line 12522 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr928;
		case 32: goto tr928;
		case 40: goto tr929;
		case 41: goto tr930;
		case 68: goto tr932;
		case 72: goto tr932;
		case 77: goto tr932;
		case 87: goto tr932;
		case 100: goto tr932;
		case 104: goto tr932;
		case 109: goto tr932;
		case 119: goto tr932;
		case 522: goto tr933;
		case 525: goto tr934;
		case 571: goto tr935;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st249;
	goto st0;
tr937:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st250;
tr938:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st250;
tr991:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st250;
tr992:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st250;
tr993:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st250;
tr928:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st250;
tr929:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st250;
tr930:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st250;
tr998:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st250;
tr999:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st250;
tr1000:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st250;
st250:
	if ( ++p == pe )
		goto _test_eof250;
case 250:
#line 12630 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st250;
		case 32: goto st250;
		case 40: goto tr937;
		case 41: goto tr938;
		case 522: goto st259;
		case 525: goto st260;
		case 571: goto st261;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr939;
	goto st0;
tr939:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st251;
tr994:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st251;
st251:
	if ( ++p == pe )
		goto _test_eof251;
case 251:
#line 12678 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr943;
		case 32: goto tr943;
		case 40: goto tr944;
		case 41: goto tr945;
		case 68: goto tr947;
		case 72: goto tr947;
		case 77: goto tr947;
		case 87: goto tr947;
		case 100: goto tr947;
		case 104: goto tr947;
		case 109: goto tr947;
		case 119: goto tr947;
		case 522: goto tr948;
		case 525: goto tr949;
		case 571: goto tr950;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st251;
	goto st0;
tr952:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st252;
tr953:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st252;
tr978:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st252;
tr979:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st252;
tr980:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st252;
tr943:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st252;
tr944:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st252;
tr945:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st252;
tr985:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st252;
tr986:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st252;
tr987:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st252;
st252:
	if ( ++p == pe )
		goto _test_eof252;
case 252:
#line 12786 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st252;
		case 32: goto st252;
		case 40: goto tr952;
		case 41: goto tr953;
		case 522: goto st255;
		case 525: goto st256;
		case 571: goto st257;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr954;
	goto st0;
tr954:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st253;
tr981:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st253;
st253:
	if ( ++p == pe )
		goto _test_eof253;
case 253:
#line 12834 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr958;
		case 32: goto tr958;
		case 40: goto tr959;
		case 41: goto tr960;
		case 68: goto tr962;
		case 72: goto tr962;
		case 77: goto tr962;
		case 87: goto tr962;
		case 100: goto tr962;
		case 104: goto tr962;
		case 109: goto tr962;
		case 119: goto tr962;
		case 266: goto tr963;
		case 269: goto tr964;
		case 315: goto tr965;
		case 522: goto tr966;
		case 525: goto tr967;
		case 571: goto tr968;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st253;
	goto st0;
tr962:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 512 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st254;
st254:
	if ( ++p == pe )
		goto _test_eof254;
case 254:
#line 12889 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr969;
		case 32: goto tr969;
		case 40: goto tr970;
		case 41: goto tr971;
		case 266: goto tr972;
		case 269: goto tr973;
		case 315: goto tr974;
		case 522: goto tr975;
		case 525: goto tr976;
		case 571: goto tr977;
	}
	goto st0;
tr982:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st255;
tr948:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st255;
tr988:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st255;
st255:
	if ( ++p == pe )
		goto _test_eof255;
case 255:
#line 12942 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr978;
		case 32: goto tr978;
		case 40: goto tr979;
		case 41: goto tr980;
		case 522: goto tr982;
		case 525: goto tr983;
		case 571: goto tr984;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr981;
	goto st0;
tr983:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st256;
tr949:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st256;
tr989:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st256;
st256:
	if ( ++p == pe )
		goto _test_eof256;
case 256:
#line 12994 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st255;
	goto st0;
tr984:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st257;
tr950:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st257;
tr990:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st257;
st257:
	if ( ++p == pe )
		goto _test_eof257;
case 257:
#line 13023 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st255;
		case 525: goto st256;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st257;
	goto st0;
tr947:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 512 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st258;
st258:
	if ( ++p == pe )
		goto _test_eof258;
case 258:
#line 13076 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr985;
		case 32: goto tr985;
		case 40: goto tr986;
		case 41: goto tr987;
		case 522: goto tr988;
		case 525: goto tr989;
		case 571: goto tr990;
	}
	goto st0;
tr995:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st259;
tr933:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st259;
tr1001:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st259;
st259:
	if ( ++p == pe )
		goto _test_eof259;
case 259:
#line 13126 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr991;
		case 32: goto tr991;
		case 40: goto tr992;
		case 41: goto tr993;
		case 522: goto tr995;
		case 525: goto tr996;
		case 571: goto tr997;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr994;
	goto st0;
tr996:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st260;
tr934:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st260;
tr1002:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st260;
st260:
	if ( ++p == pe )
		goto _test_eof260;
case 260:
#line 13178 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st259;
	goto st0;
tr997:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st261;
tr935:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st261;
tr1003:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st261;
st261:
	if ( ++p == pe )
		goto _test_eof261;
case 261:
#line 13207 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st259;
		case 525: goto st260;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st261;
	goto st0;
tr932:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 512 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st262;
st262:
	if ( ++p == pe )
		goto _test_eof262;
case 262:
#line 13260 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr998;
		case 32: goto tr998;
		case 40: goto tr999;
		case 41: goto tr1000;
		case 522: goto tr1001;
		case 525: goto tr1002;
		case 571: goto tr1003;
	}
	goto st0;
tr1008:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st263;
tr918:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st263;
tr1014:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st263;
st263:
	if ( ++p == pe )
		goto _test_eof263;
case 263:
#line 13310 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1004;
		case 32: goto tr1004;
		case 40: goto tr1005;
		case 41: goto tr1006;
		case 522: goto tr1008;
		case 525: goto tr1009;
		case 571: goto tr1010;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1007;
	goto st0;
tr1009:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st264;
tr919:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st264;
tr1015:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st264;
st264:
	if ( ++p == pe )
		goto _test_eof264;
case 264:
#line 13362 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st263;
	goto st0;
tr1010:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st265;
tr920:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st265;
tr1016:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st265;
st265:
	if ( ++p == pe )
		goto _test_eof265;
case 265:
#line 13391 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st263;
		case 525: goto st264;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st265;
	goto st0;
tr917:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 512 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st266;
st266:
	if ( ++p == pe )
		goto _test_eof266;
case 266:
#line 13444 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1011;
		case 32: goto tr1011;
		case 40: goto tr1012;
		case 41: goto tr1013;
		case 522: goto tr1014;
		case 525: goto tr1015;
		case 571: goto tr1016;
	}
	goto st0;
tr1021:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st267;
tr903:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st267;
tr1027:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st267;
st267:
	if ( ++p == pe )
		goto _test_eof267;
case 267:
#line 13494 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1017;
		case 32: goto tr1017;
		case 40: goto tr1018;
		case 41: goto tr1019;
		case 522: goto tr1021;
		case 525: goto tr1022;
		case 571: goto tr1023;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1020;
	goto st0;
tr1022:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st268;
tr904:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st268;
tr1028:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st268;
st268:
	if ( ++p == pe )
		goto _test_eof268;
case 268:
#line 13546 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st267;
	goto st0;
tr1023:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st269;
tr905:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st269;
tr1029:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st269;
st269:
	if ( ++p == pe )
		goto _test_eof269;
case 269:
#line 13575 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st267;
		case 525: goto st268;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st269;
	goto st0;
tr902:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 512 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st270;
st270:
	if ( ++p == pe )
		goto _test_eof270;
case 270:
#line 13628 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1024;
		case 32: goto tr1024;
		case 40: goto tr1025;
		case 41: goto tr1026;
		case 522: goto tr1027;
		case 525: goto tr1028;
		case 571: goto tr1029;
	}
	goto st0;
tr1034:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st271;
tr888:
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st271;
tr1045:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st271;
tr1062:
#line 496 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
	goto st271;
st271:
	if ( ++p == pe )
		goto _test_eof271;
case 271:
#line 13682 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1030;
		case 32: goto tr1030;
		case 40: goto tr1031;
		case 41: goto tr1032;
		case 522: goto tr1034;
		case 525: goto tr1035;
		case 571: goto tr1036;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1033;
	goto st0;
tr1035:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st272;
tr889:
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st272;
tr1046:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st272;
tr1063:
#line 496 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
	goto st272;
st272:
	if ( ++p == pe )
		goto _test_eof272;
case 272:
#line 13738 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st271;
	goto st0;
tr1036:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st273;
tr890:
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st273;
tr1047:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 495 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st273;
tr1064:
#line 496 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
	goto st273;
st273:
	if ( ++p == pe )
		goto _test_eof273;
case 273:
#line 13771 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st271;
		case 525: goto st272;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st273;
	goto st0;
tr879:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st274;
tr1044:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st274;
tr1072:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st274;
st274:
	if ( ++p == pe )
		goto _test_eof274;
case 274:
#line 13832 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st275;
		case 13: goto st276;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st277;
	goto st243;
st275:
	if ( ++p == pe )
		goto _test_eof275;
case 275:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1041;
		case 32: goto tr1041;
		case 40: goto tr1042;
		case 41: goto tr1043;
		case 92: goto tr1044;
		case 522: goto tr1045;
		case 525: goto tr1046;
		case 571: goto tr1047;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1040;
		} else
			goto tr1040;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1040;
		} else if ( _widec >= 35 )
			goto tr1040;
	} else
		goto tr1040;
	goto st0;
st276:
	if ( ++p == pe )
		goto _test_eof276;
case 276:
	if ( (*p) == 10 )
		goto st275;
	goto st0;
st277:
	if ( ++p == pe )
		goto _test_eof277;
case 277:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st278;
	goto st0;
st278:
	if ( ++p == pe )
		goto _test_eof278;
case 278:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st243;
	goto st0;
tr876:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st279;
tr1054:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st279;
tr1069:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st279;
st279:
	if ( ++p == pe )
		goto _test_eof279;
case 279:
#line 13932 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st280;
		case 13: goto st281;
		case 34: goto st282;
		case 92: goto st283;
	}
	goto st279;
tr1055:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st280;
st280:
	if ( ++p == pe )
		goto _test_eof280;
case 280:
#line 13948 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1055;
		case 13: goto tr1056;
		case 34: goto tr1057;
		case 92: goto tr1058;
	}
	goto tr1054;
tr1056:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st281;
st281:
	if ( ++p == pe )
		goto _test_eof281;
case 281:
#line 13964 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st280;
	goto st0;
tr1057:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st282;
st282:
	if ( ++p == pe )
		goto _test_eof282;
case 282:
#line 13976 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1059;
		case 32: goto tr1059;
		case 40: goto tr1060;
		case 41: goto tr1061;
		case 522: goto tr1062;
		case 525: goto tr1063;
		case 571: goto tr1064;
	}
	goto st0;
tr1058:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st283;
st283:
	if ( ++p == pe )
		goto _test_eof283;
case 283:
#line 14016 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st280;
		case 13: goto st281;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st284;
	goto st279;
st284:
	if ( ++p == pe )
		goto _test_eof284;
case 284:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st285;
	goto st0;
st285:
	if ( ++p == pe )
		goto _test_eof285;
case 285:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st279;
	goto st0;
tr1073:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st286;
tr871:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st286;
tr1084:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st286;
tr1101:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
	goto st286;
tr1109:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
	goto st286;
st286:
	if ( ++p == pe )
		goto _test_eof286;
case 286:
#line 14067 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1068;
		case 32: goto tr1068;
		case 34: goto tr1069;
		case 40: goto tr1070;
		case 41: goto tr1071;
		case 92: goto tr1072;
		case 522: goto tr1073;
		case 525: goto tr1074;
		case 571: goto tr1075;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1067;
		} else
			goto tr1067;
	} else if ( _widec > 35 ) {
		if ( _widec < 60 ) {
			if ( 37 <= _widec && _widec <= 58 )
				goto tr1067;
		} else if ( _widec > 63 ) {
			if ( 65 <= _widec )
				goto tr1067;
		} else
			goto tr1067;
	} else
		goto tr1067;
	goto st0;
tr1074:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st287;
tr872:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st287;
tr1085:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st287;
tr1102:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
	goto st287;
tr1110:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
	goto st287;
st287:
	if ( ++p == pe )
		goto _test_eof287;
case 287:
#line 14144 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st286;
	goto st0;
tr1075:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st288;
tr873:
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st288;
tr1086:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 493 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st288;
tr1103:
#line 494 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
	goto st288;
tr1111:
#line 492 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
	goto st288;
st288:
	if ( ++p == pe )
		goto _test_eof288;
case 288:
#line 14181 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st286;
		case 525: goto st287;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st288;
	goto st0;
tr865:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st289;
tr1083:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st289;
tr1118:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st289;
st289:
	if ( ++p == pe )
		goto _test_eof289;
case 289:
#line 14242 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st290;
		case 13: goto st291;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st292;
	goto st241;
st290:
	if ( ++p == pe )
		goto _test_eof290;
case 290:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1080;
		case 32: goto tr1080;
		case 40: goto tr1081;
		case 41: goto tr1082;
		case 92: goto tr1083;
		case 522: goto tr1084;
		case 525: goto tr1085;
		case 571: goto tr1086;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1079;
		} else
			goto tr1079;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1079;
		} else if ( _widec >= 35 )
			goto tr1079;
	} else
		goto tr1079;
	goto st0;
st291:
	if ( ++p == pe )
		goto _test_eof291;
case 291:
	if ( (*p) == 10 )
		goto st290;
	goto st0;
st292:
	if ( ++p == pe )
		goto _test_eof292;
case 292:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st293;
	goto st0;
st293:
	if ( ++p == pe )
		goto _test_eof293;
case 293:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st241;
	goto st0;
tr863:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st294;
tr1093:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st294;
tr1114:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st294;
st294:
	if ( ++p == pe )
		goto _test_eof294;
case 294:
#line 14342 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st295;
		case 13: goto st296;
		case 34: goto st297;
		case 92: goto st298;
	}
	goto st294;
tr1094:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st295;
st295:
	if ( ++p == pe )
		goto _test_eof295;
case 295:
#line 14358 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1094;
		case 13: goto tr1095;
		case 34: goto tr1096;
		case 92: goto tr1097;
	}
	goto tr1093;
tr1095:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st296;
st296:
	if ( ++p == pe )
		goto _test_eof296;
case 296:
#line 14374 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st295;
	goto st0;
tr1096:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st297;
st297:
	if ( ++p == pe )
		goto _test_eof297;
case 297:
#line 14386 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1098;
		case 32: goto tr1098;
		case 40: goto tr1099;
		case 41: goto tr1100;
		case 522: goto tr1101;
		case 525: goto tr1102;
		case 571: goto tr1103;
	}
	goto st0;
tr1097:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st298;
st298:
	if ( ++p == pe )
		goto _test_eof298;
case 298:
#line 14426 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st295;
		case 13: goto st296;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st299;
	goto st294;
st299:
	if ( ++p == pe )
		goto _test_eof299;
case 299:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st300;
	goto st0;
st300:
	if ( ++p == pe )
		goto _test_eof300;
case 300:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st294;
	goto st0;
tr864:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st301;
tr1117:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st301;
st301:
	if ( ++p == pe )
		goto _test_eof301;
case 301:
#line 14465 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1106;
		case 32: goto tr1106;
		case 40: goto tr1107;
		case 41: goto tr1108;
		case 522: goto tr1109;
		case 525: goto tr1110;
		case 571: goto tr1111;
	}
	goto st0;
tr1119:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st302;
st302:
	if ( ++p == pe )
		goto _test_eof302;
case 302:
#line 14505 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1113;
		case 32: goto tr1113;
		case 34: goto tr1114;
		case 40: goto tr1115;
		case 41: goto tr1116;
		case 64: goto tr1117;
		case 92: goto tr1118;
		case 522: goto tr1119;
		case 525: goto tr1120;
		case 571: goto tr1121;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1112;
		} else
			goto tr1112;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1112;
		} else if ( _widec >= 37 )
			goto tr1112;
	} else
		goto tr1112;
	goto st0;
tr1120:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st303;
st303:
	if ( ++p == pe )
		goto _test_eof303;
case 303:
#line 14562 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st302;
	goto st0;
tr1121:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st304;
st304:
	if ( ++p == pe )
		goto _test_eof304;
case 304:
#line 14581 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st302;
		case 525: goto st303;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st304;
	goto st0;
st305:
	if ( ++p == pe )
		goto _test_eof305;
case 305:
	switch( (*p) ) {
		case 70: goto st306;
		case 102: goto st306;
	}
	goto st0;
st306:
	if ( ++p == pe )
		goto _test_eof306;
case 306:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st307;
		case 32: goto st307;
		case 40: goto tr1124;
		case 41: goto tr1125;
		case 43: goto st329;
		case 522: goto st355;
		case 525: goto st356;
		case 571: goto st357;
	}
	goto st0;
tr1124:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st307;
tr1125:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st307;
st307:
	if ( ++p == pe )
		goto _test_eof307;
case 307:
#line 14681 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1131;
		case 32: goto tr1131;
		case 34: goto tr1132;
		case 40: goto tr1133;
		case 41: goto tr1134;
		case 92: goto tr1135;
		case 522: goto tr1136;
		case 525: goto tr1137;
		case 571: goto tr1138;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1130;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1130;
		} else if ( _widec >= 14 )
			goto tr1130;
	} else
		goto tr1130;
	goto st0;
tr1151:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st308;
tr1188:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st308;
tr1200:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st308;
tr1130:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st308;
tr1215:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st308;
tr1173:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st308;
st308:
	if ( ++p == pe )
		goto _test_eof308;
case 308:
#line 14764 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1140;
		case 32: goto tr1140;
		case 34: goto tr1141;
		case 40: goto tr1142;
		case 41: goto tr1143;
		case 92: goto st314;
		case 266: goto tr1145;
		case 269: goto tr1146;
		case 315: goto tr1147;
		case 522: goto tr1148;
		case 525: goto tr1149;
		case 571: goto tr1150;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto st308;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st308;
		} else if ( _widec >= 14 )
			goto st308;
	} else
		goto st308;
	goto st0;
tr1154:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st309;
tr1155:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st309;
tr1201:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st309;
tr1203:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st309;
tr1204:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st309;
tr1140:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st309;
tr1142:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st309;
tr1143:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st309;
tr1174:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st309;
tr1176:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st309;
tr1177:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st309;
tr1189:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st309;
tr1191:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st309;
tr1192:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st309;
st309:
	if ( ++p == pe )
		goto _test_eof309;
case 309:
#line 14894 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st309;
		case 32: goto st309;
		case 34: goto tr1153;
		case 40: goto tr1154;
		case 41: goto tr1155;
		case 92: goto tr1156;
		case 266: goto tr1157;
		case 269: goto tr1158;
		case 315: goto tr1159;
		case 522: goto st316;
		case 525: goto st317;
		case 571: goto st318;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1151;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1151;
		} else if ( _widec >= 14 )
			goto tr1151;
	} else
		goto tr1151;
	goto st0;
tr1153:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
tr1168:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st310;
tr1202:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
tr1132:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
tr1217:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
tr1141:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
tr1175:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
tr1190:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
st310:
	if ( ++p == pe )
		goto _test_eof310;
case 310:
#line 14994 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st311;
		case 13: goto st312;
		case 34: goto st313;
		case 92: goto st322;
	}
	goto st310;
tr1169:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st311;
st311:
	if ( ++p == pe )
		goto _test_eof311;
case 311:
#line 15010 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1169;
		case 13: goto tr1170;
		case 34: goto tr1171;
		case 92: goto tr1172;
	}
	goto tr1168;
tr1170:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st312;
st312:
	if ( ++p == pe )
		goto _test_eof312;
case 312:
#line 15026 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st311;
	goto st0;
tr1171:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st313;
st313:
	if ( ++p == pe )
		goto _test_eof313;
case 313:
#line 15038 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1174;
		case 32: goto tr1174;
		case 34: goto tr1175;
		case 40: goto tr1176;
		case 41: goto tr1177;
		case 92: goto tr1178;
		case 266: goto tr1179;
		case 269: goto tr1180;
		case 315: goto tr1181;
		case 522: goto tr1182;
		case 525: goto tr1183;
		case 571: goto tr1184;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1173;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1173;
		} else if ( _widec >= 14 )
			goto tr1173;
	} else
		goto tr1173;
	goto st0;
tr1156:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st314;
tr1193:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st314;
tr1205:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st314;
tr1135:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st314;
tr1220:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st314;
tr1178:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st314;
st314:
	if ( ++p == pe )
		goto _test_eof314;
case 314:
#line 15124 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st315;
		case 13: goto st319;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st320;
	goto st308;
st315:
	if ( ++p == pe )
		goto _test_eof315;
case 315:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1189;
		case 32: goto tr1189;
		case 34: goto tr1190;
		case 40: goto tr1191;
		case 41: goto tr1192;
		case 92: goto tr1193;
		case 266: goto tr1194;
		case 269: goto tr1195;
		case 315: goto tr1196;
		case 522: goto tr1197;
		case 525: goto tr1198;
		case 571: goto tr1199;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1188;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1188;
		} else if ( _widec >= 14 )
			goto tr1188;
	} else
		goto tr1188;
	goto st0;
tr1209:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st316;
tr1148:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st316;
tr1182:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st316;
tr1197:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st316;
st316:
	if ( ++p == pe )
		goto _test_eof316;
case 316:
#line 15208 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1201;
		case 32: goto tr1201;
		case 34: goto tr1202;
		case 40: goto tr1203;
		case 41: goto tr1204;
		case 92: goto tr1205;
		case 266: goto tr1206;
		case 269: goto tr1207;
		case 315: goto tr1208;
		case 522: goto tr1209;
		case 525: goto tr1210;
		case 571: goto tr1211;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1200;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1200;
		} else if ( _widec >= 14 )
			goto tr1200;
	} else
		goto tr1200;
	goto st0;
tr1210:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st317;
tr1149:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st317;
tr1183:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st317;
tr1198:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st317;
st317:
	if ( ++p == pe )
		goto _test_eof317;
case 317:
#line 15278 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st316;
	goto st0;
tr1211:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st318;
tr1150:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st318;
tr1184:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st318;
tr1199:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st318;
st318:
	if ( ++p == pe )
		goto _test_eof318;
case 318:
#line 15311 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st316;
		case 525: goto st317;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st318;
	goto st0;
st319:
	if ( ++p == pe )
		goto _test_eof319;
case 319:
	if ( (*p) == 10 )
		goto st315;
	goto st0;
st320:
	if ( ++p == pe )
		goto _test_eof320;
case 320:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st321;
	goto st0;
st321:
	if ( ++p == pe )
		goto _test_eof321;
case 321:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st308;
	goto st0;
tr1172:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st322;
st322:
	if ( ++p == pe )
		goto _test_eof322;
case 322:
#line 15383 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st311;
		case 13: goto st312;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st323;
	goto st310;
st323:
	if ( ++p == pe )
		goto _test_eof323;
case 323:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st324;
	goto st0;
st324:
	if ( ++p == pe )
		goto _test_eof324;
case 324:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st310;
	goto st0;
tr1131:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st325;
tr1133:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st325;
tr1134:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st325;
tr1216:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st325;
tr1218:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st325;
tr1219:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st325;
st325:
	if ( ++p == pe )
		goto _test_eof325;
case 325:
#line 15450 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1131;
		case 32: goto tr1131;
		case 34: goto tr1132;
		case 40: goto tr1133;
		case 41: goto tr1134;
		case 92: goto tr1135;
		case 266: goto tr1157;
		case 269: goto tr1158;
		case 315: goto tr1159;
		case 522: goto tr1136;
		case 525: goto tr1137;
		case 571: goto tr1138;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1130;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1130;
		} else if ( _widec >= 14 )
			goto tr1130;
	} else
		goto tr1130;
	goto st0;
tr1136:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st326;
tr1221:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st326;
st326:
	if ( ++p == pe )
		goto _test_eof326;
case 326:
#line 15512 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1216;
		case 32: goto tr1216;
		case 34: goto tr1217;
		case 40: goto tr1218;
		case 41: goto tr1219;
		case 92: goto tr1220;
		case 266: goto tr1206;
		case 269: goto tr1207;
		case 315: goto tr1208;
		case 522: goto tr1221;
		case 525: goto tr1222;
		case 571: goto tr1223;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1215;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1215;
		} else if ( _widec >= 14 )
			goto tr1215;
	} else
		goto tr1215;
	goto st0;
tr1137:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st327;
tr1222:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st327;
st327:
	if ( ++p == pe )
		goto _test_eof327;
case 327:
#line 15574 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st326;
	goto st0;
tr1138:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st328;
tr1223:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st328;
st328:
	if ( ++p == pe )
		goto _test_eof328;
case 328:
#line 15599 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st326;
		case 525: goto st327;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st328;
	goto st0;
st329:
	if ( ++p == pe )
		goto _test_eof329;
case 329:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st330;
		case 32: goto st330;
		case 40: goto tr1228;
		case 41: goto tr1229;
		case 522: goto st352;
		case 525: goto st353;
		case 571: goto st354;
	}
	goto st0;
tr1228:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st330;
tr1229:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st330;
st330:
	if ( ++p == pe )
		goto _test_eof330;
case 330:
#line 15689 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1234;
		case 32: goto tr1234;
		case 34: goto tr1235;
		case 40: goto tr1236;
		case 41: goto tr1237;
		case 92: goto tr1238;
		case 522: goto tr1239;
		case 525: goto tr1240;
		case 571: goto tr1241;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1233;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1233;
		} else if ( _widec >= 14 )
			goto tr1233;
	} else
		goto tr1233;
	goto st0;
tr1254:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st331;
tr1291:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st331;
tr1303:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st331;
tr1233:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st331;
tr1318:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st331;
tr1276:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st331;
st331:
	if ( ++p == pe )
		goto _test_eof331;
case 331:
#line 15772 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1243;
		case 32: goto tr1243;
		case 34: goto tr1244;
		case 40: goto tr1245;
		case 41: goto tr1246;
		case 92: goto st337;
		case 266: goto tr1248;
		case 269: goto tr1249;
		case 315: goto tr1250;
		case 522: goto tr1251;
		case 525: goto tr1252;
		case 571: goto tr1253;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto st331;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st331;
		} else if ( _widec >= 14 )
			goto st331;
	} else
		goto st331;
	goto st0;
tr1257:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st332;
tr1258:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st332;
tr1304:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st332;
tr1306:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st332;
tr1307:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st332;
tr1243:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st332;
tr1245:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st332;
tr1246:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st332;
tr1277:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st332;
tr1279:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st332;
tr1280:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st332;
tr1292:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st332;
tr1294:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st332;
tr1295:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st332;
st332:
	if ( ++p == pe )
		goto _test_eof332;
case 332:
#line 15902 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st332;
		case 32: goto st332;
		case 34: goto tr1256;
		case 40: goto tr1257;
		case 41: goto tr1258;
		case 92: goto tr1259;
		case 266: goto tr1260;
		case 269: goto tr1261;
		case 315: goto tr1262;
		case 522: goto st339;
		case 525: goto st340;
		case 571: goto st341;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1254;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1254;
		} else if ( _widec >= 14 )
			goto tr1254;
	} else
		goto tr1254;
	goto st0;
tr1256:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st333;
tr1271:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st333;
tr1305:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st333;
tr1235:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st333;
tr1320:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st333;
tr1244:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st333;
tr1278:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st333;
tr1293:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st333;
st333:
	if ( ++p == pe )
		goto _test_eof333;
case 333:
#line 16002 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st334;
		case 13: goto st335;
		case 34: goto st336;
		case 92: goto st345;
	}
	goto st333;
tr1272:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st334;
st334:
	if ( ++p == pe )
		goto _test_eof334;
case 334:
#line 16018 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1272;
		case 13: goto tr1273;
		case 34: goto tr1274;
		case 92: goto tr1275;
	}
	goto tr1271;
tr1273:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st335;
st335:
	if ( ++p == pe )
		goto _test_eof335;
case 335:
#line 16034 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st334;
	goto st0;
tr1274:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st336;
st336:
	if ( ++p == pe )
		goto _test_eof336;
case 336:
#line 16046 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1277;
		case 32: goto tr1277;
		case 34: goto tr1278;
		case 40: goto tr1279;
		case 41: goto tr1280;
		case 92: goto tr1281;
		case 266: goto tr1282;
		case 269: goto tr1283;
		case 315: goto tr1284;
		case 522: goto tr1285;
		case 525: goto tr1286;
		case 571: goto tr1287;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1276;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1276;
		} else if ( _widec >= 14 )
			goto tr1276;
	} else
		goto tr1276;
	goto st0;
tr1259:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st337;
tr1296:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st337;
tr1308:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st337;
tr1238:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st337;
tr1323:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st337;
tr1281:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st337;
st337:
	if ( ++p == pe )
		goto _test_eof337;
case 337:
#line 16132 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st338;
		case 13: goto st342;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st343;
	goto st331;
st338:
	if ( ++p == pe )
		goto _test_eof338;
case 338:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1292;
		case 32: goto tr1292;
		case 34: goto tr1293;
		case 40: goto tr1294;
		case 41: goto tr1295;
		case 92: goto tr1296;
		case 266: goto tr1297;
		case 269: goto tr1298;
		case 315: goto tr1299;
		case 522: goto tr1300;
		case 525: goto tr1301;
		case 571: goto tr1302;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1291;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1291;
		} else if ( _widec >= 14 )
			goto tr1291;
	} else
		goto tr1291;
	goto st0;
tr1312:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st339;
tr1251:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st339;
tr1285:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st339;
tr1300:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st339;
st339:
	if ( ++p == pe )
		goto _test_eof339;
case 339:
#line 16216 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1304;
		case 32: goto tr1304;
		case 34: goto tr1305;
		case 40: goto tr1306;
		case 41: goto tr1307;
		case 92: goto tr1308;
		case 266: goto tr1309;
		case 269: goto tr1310;
		case 315: goto tr1311;
		case 522: goto tr1312;
		case 525: goto tr1313;
		case 571: goto tr1314;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1303;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1303;
		} else if ( _widec >= 14 )
			goto tr1303;
	} else
		goto tr1303;
	goto st0;
tr1313:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st340;
tr1252:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st340;
tr1286:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st340;
tr1301:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st340;
st340:
	if ( ++p == pe )
		goto _test_eof340;
case 340:
#line 16286 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st339;
	goto st0;
tr1314:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st341;
tr1253:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st341;
tr1287:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st341;
tr1302:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st341;
st341:
	if ( ++p == pe )
		goto _test_eof341;
case 341:
#line 16319 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st339;
		case 525: goto st340;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st341;
	goto st0;
st342:
	if ( ++p == pe )
		goto _test_eof342;
case 342:
	if ( (*p) == 10 )
		goto st338;
	goto st0;
st343:
	if ( ++p == pe )
		goto _test_eof343;
case 343:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st344;
	goto st0;
st344:
	if ( ++p == pe )
		goto _test_eof344;
case 344:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st331;
	goto st0;
tr1275:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st345;
st345:
	if ( ++p == pe )
		goto _test_eof345;
case 345:
#line 16391 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st334;
		case 13: goto st335;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st346;
	goto st333;
st346:
	if ( ++p == pe )
		goto _test_eof346;
case 346:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st347;
	goto st0;
st347:
	if ( ++p == pe )
		goto _test_eof347;
case 347:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st333;
	goto st0;
tr1234:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st348;
tr1236:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st348;
tr1237:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st348;
tr1319:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st348;
tr1321:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st348;
tr1322:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st348;
st348:
	if ( ++p == pe )
		goto _test_eof348;
case 348:
#line 16458 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1234;
		case 32: goto tr1234;
		case 34: goto tr1235;
		case 40: goto tr1236;
		case 41: goto tr1237;
		case 92: goto tr1238;
		case 266: goto tr1260;
		case 269: goto tr1261;
		case 315: goto tr1262;
		case 522: goto tr1239;
		case 525: goto tr1240;
		case 571: goto tr1241;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1233;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1233;
		} else if ( _widec >= 14 )
			goto tr1233;
	} else
		goto tr1233;
	goto st0;
tr1239:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st349;
tr1324:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st349;
st349:
	if ( ++p == pe )
		goto _test_eof349;
case 349:
#line 16520 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1319;
		case 32: goto tr1319;
		case 34: goto tr1320;
		case 40: goto tr1321;
		case 41: goto tr1322;
		case 92: goto tr1323;
		case 266: goto tr1309;
		case 269: goto tr1310;
		case 315: goto tr1311;
		case 522: goto tr1324;
		case 525: goto tr1325;
		case 571: goto tr1326;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1318;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1318;
		} else if ( _widec >= 14 )
			goto tr1318;
	} else
		goto tr1318;
	goto st0;
tr1240:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st350;
tr1325:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st350;
st350:
	if ( ++p == pe )
		goto _test_eof350;
case 350:
#line 16582 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st349;
	goto st0;
tr1241:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st351;
tr1326:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st351;
st351:
	if ( ++p == pe )
		goto _test_eof351;
case 351:
#line 16607 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st349;
		case 525: goto st350;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st351;
	goto st0;
st352:
	if ( ++p == pe )
		goto _test_eof352;
case 352:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1319;
		case 32: goto tr1319;
		case 34: goto tr1320;
		case 40: goto tr1321;
		case 41: goto tr1322;
		case 92: goto tr1323;
		case 522: goto tr1324;
		case 525: goto tr1325;
		case 571: goto tr1326;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1318;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1318;
		} else if ( _widec >= 14 )
			goto tr1318;
	} else
		goto tr1318;
	goto st0;
st353:
	if ( ++p == pe )
		goto _test_eof353;
case 353:
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st352;
	goto st0;
st354:
	if ( ++p == pe )
		goto _test_eof354;
case 354:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st352;
		case 525: goto st353;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st354;
	goto st0;
st355:
	if ( ++p == pe )
		goto _test_eof355;
case 355:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1216;
		case 32: goto tr1216;
		case 34: goto tr1217;
		case 40: goto tr1218;
		case 41: goto tr1219;
		case 92: goto tr1220;
		case 522: goto tr1221;
		case 525: goto tr1222;
		case 571: goto tr1223;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1215;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1215;
		} else if ( _widec >= 14 )
			goto tr1215;
	} else
		goto tr1215;
	goto st0;
st356:
	if ( ++p == pe )
		goto _test_eof356;
case 356:
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st355;
	goto st0;
st357:
	if ( ++p == pe )
		goto _test_eof357;
case 357:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st355;
		case 525: goto st356;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st357;
	goto st0;
st358:
	if ( ++p == pe )
		goto _test_eof358;
case 358:
	switch( (*p) ) {
		case 86: goto st359;
		case 118: goto st359;
	}
	goto st0;
st359:
	if ( ++p == pe )
		goto _test_eof359;
case 359:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st360;
		case 32: goto st360;
		case 40: goto tr1332;
		case 41: goto tr1333;
		case 522: goto st390;
		case 525: goto st391;
		case 571: goto st392;
	}
	goto st0;
tr1332:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st360;
tr1333:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st360;
tr1463:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st360;
tr1464:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st360;
tr1465:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st360;
st360:
	if ( ++p == pe )
		goto _test_eof360;
case 360:
#line 16938 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st360;
		case 32: goto st360;
		case 40: goto tr1332;
		case 41: goto tr1333;
		case 522: goto st390;
		case 525: goto st391;
		case 571: goto st392;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1337;
	goto st0;
tr1337:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st361;
tr1466:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st361;
st361:
	if ( ++p == pe )
		goto _test_eof361;
case 361:
#line 16986 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1338;
		case 32: goto tr1338;
		case 40: goto tr1339;
		case 41: goto tr1340;
		case 522: goto tr1342;
		case 525: goto tr1343;
		case 571: goto tr1344;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st361;
	goto st0;
tr1346:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st362;
tr1347:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st362;
tr1456:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st362;
tr1457:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st362;
tr1458:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st362;
tr1338:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st362;
tr1339:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st362;
tr1340:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st362;
st362:
	if ( ++p == pe )
		goto _test_eof362;
case 362:
#line 17070 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st362;
		case 32: goto st362;
		case 40: goto tr1346;
		case 41: goto tr1347;
		case 522: goto st387;
		case 525: goto st388;
		case 571: goto st389;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1348;
	goto st0;
tr1348:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st363;
tr1459:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st363;
st363:
	if ( ++p == pe )
		goto _test_eof363;
case 363:
#line 17118 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1352;
		case 32: goto tr1352;
		case 40: goto tr1353;
		case 41: goto tr1354;
		case 522: goto tr1356;
		case 525: goto tr1357;
		case 571: goto tr1358;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st363;
	goto st0;
tr1360:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st364;
tr1361:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st364;
tr1449:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st364;
tr1450:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st364;
tr1451:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st364;
tr1352:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st364;
tr1353:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st364;
tr1354:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st364;
st364:
	if ( ++p == pe )
		goto _test_eof364;
case 364:
#line 17202 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st364;
		case 32: goto st364;
		case 40: goto tr1360;
		case 41: goto tr1361;
		case 522: goto st384;
		case 525: goto st385;
		case 571: goto st386;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1362;
	goto st0;
tr1362:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st365;
tr1452:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st365;
st365:
	if ( ++p == pe )
		goto _test_eof365;
case 365:
#line 17250 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1366;
		case 32: goto tr1366;
		case 40: goto tr1367;
		case 41: goto tr1368;
		case 522: goto tr1370;
		case 525: goto tr1371;
		case 571: goto tr1372;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st365;
	goto st0;
tr1376:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st366;
tr1377:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st366;
tr1440:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st366;
tr1442:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st366;
tr1443:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st366;
tr1366:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st366;
tr1367:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st366;
tr1368:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st366;
st366:
	if ( ++p == pe )
		goto _test_eof366;
case 366:
#line 17334 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st366;
		case 32: goto st366;
		case 34: goto tr1375;
		case 40: goto tr1376;
		case 41: goto tr1377;
		case 64: goto tr1378;
		case 92: goto tr1379;
		case 522: goto st381;
		case 525: goto st382;
		case 571: goto st383;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1373;
		} else
			goto tr1373;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1373;
		} else if ( _widec >= 37 )
			goto tr1373;
	} else
		goto tr1373;
	goto st0;
tr1373:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st367;
tr1397:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st367;
tr1439:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st367;
st367:
	if ( ++p == pe )
		goto _test_eof367;
case 367:
#line 17401 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1384;
		case 32: goto tr1384;
		case 40: goto tr1385;
		case 41: goto tr1386;
		case 92: goto st368;
		case 266: goto tr1388;
		case 269: goto tr1389;
		case 315: goto tr1390;
		case 522: goto tr1391;
		case 525: goto tr1392;
		case 571: goto tr1393;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st367;
		} else
			goto st367;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st367;
		} else if ( _widec >= 35 )
			goto st367;
	} else
		goto st367;
	goto st0;
tr1379:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st368;
tr1401:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st368;
tr1445:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st368;
st368:
	if ( ++p == pe )
		goto _test_eof368;
case 368:
#line 17469 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st369;
		case 13: goto st370;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st371;
	goto st367;
st369:
	if ( ++p == pe )
		goto _test_eof369;
case 369:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1398;
		case 32: goto tr1398;
		case 40: goto tr1399;
		case 41: goto tr1400;
		case 92: goto tr1401;
		case 266: goto tr1402;
		case 269: goto tr1403;
		case 315: goto tr1404;
		case 522: goto tr1405;
		case 525: goto tr1406;
		case 571: goto tr1407;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1397;
		} else
			goto tr1397;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1397;
		} else if ( _widec >= 35 )
			goto tr1397;
	} else
		goto tr1397;
	goto st0;
st370:
	if ( ++p == pe )
		goto _test_eof370;
case 370:
	if ( (*p) == 10 )
		goto st369;
	goto st0;
st371:
	if ( ++p == pe )
		goto _test_eof371;
case 371:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st372;
	goto st0;
st372:
	if ( ++p == pe )
		goto _test_eof372;
case 372:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st367;
	goto st0;
tr1375:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st373;
tr1414:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st373;
tr1441:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st373;
st373:
	if ( ++p == pe )
		goto _test_eof373;
case 373:
#line 17572 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st374;
		case 13: goto st375;
		case 34: goto st376;
		case 92: goto st377;
	}
	goto st373;
tr1415:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st374;
st374:
	if ( ++p == pe )
		goto _test_eof374;
case 374:
#line 17588 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1415;
		case 13: goto tr1416;
		case 34: goto tr1417;
		case 92: goto tr1418;
	}
	goto tr1414;
tr1416:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st375;
st375:
	if ( ++p == pe )
		goto _test_eof375;
case 375:
#line 17604 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st374;
	goto st0;
tr1417:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st376;
st376:
	if ( ++p == pe )
		goto _test_eof376;
case 376:
#line 17616 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1419;
		case 32: goto tr1419;
		case 40: goto tr1420;
		case 41: goto tr1421;
		case 266: goto tr1422;
		case 269: goto tr1423;
		case 315: goto tr1424;
		case 522: goto tr1425;
		case 525: goto tr1426;
		case 571: goto tr1427;
	}
	goto st0;
tr1418:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st377;
st377:
	if ( ++p == pe )
		goto _test_eof377;
case 377:
#line 17659 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st374;
		case 13: goto st375;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st378;
	goto st373;
st378:
	if ( ++p == pe )
		goto _test_eof378;
case 378:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st379;
	goto st0;
st379:
	if ( ++p == pe )
		goto _test_eof379;
case 379:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st373;
	goto st0;
tr1378:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st380;
tr1444:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st380;
st380:
	if ( ++p == pe )
		goto _test_eof380;
case 380:
#line 17698 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1430;
		case 32: goto tr1430;
		case 40: goto tr1431;
		case 41: goto tr1432;
		case 266: goto tr1433;
		case 269: goto tr1434;
		case 315: goto tr1435;
		case 522: goto tr1436;
		case 525: goto tr1437;
		case 571: goto tr1438;
	}
	goto st0;
tr1446:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st381;
tr1370:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st381;
st381:
	if ( ++p == pe )
		goto _test_eof381;
case 381:
#line 17747 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1440;
		case 32: goto tr1440;
		case 34: goto tr1441;
		case 40: goto tr1442;
		case 41: goto tr1443;
		case 64: goto tr1444;
		case 92: goto tr1445;
		case 522: goto tr1446;
		case 525: goto tr1447;
		case 571: goto tr1448;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1439;
		} else
			goto tr1439;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1439;
		} else if ( _widec >= 37 )
			goto tr1439;
	} else
		goto tr1439;
	goto st0;
tr1447:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st382;
tr1371:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st382;
st382:
	if ( ++p == pe )
		goto _test_eof382;
case 382:
#line 17810 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st381;
	goto st0;
tr1448:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st383;
tr1372:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st383;
st383:
	if ( ++p == pe )
		goto _test_eof383;
case 383:
#line 17835 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st381;
		case 525: goto st382;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st383;
	goto st0;
tr1453:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st384;
tr1356:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st384;
st384:
	if ( ++p == pe )
		goto _test_eof384;
case 384:
#line 17892 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1449;
		case 32: goto tr1449;
		case 40: goto tr1450;
		case 41: goto tr1451;
		case 522: goto tr1453;
		case 525: goto tr1454;
		case 571: goto tr1455;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1452;
	goto st0;
tr1454:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st385;
tr1357:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st385;
st385:
	if ( ++p == pe )
		goto _test_eof385;
case 385:
#line 17940 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st384;
	goto st0;
tr1455:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st386;
tr1358:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st386;
st386:
	if ( ++p == pe )
		goto _test_eof386;
case 386:
#line 17965 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st384;
		case 525: goto st385;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st386;
	goto st0;
tr1460:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st387;
tr1342:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st387;
st387:
	if ( ++p == pe )
		goto _test_eof387;
case 387:
#line 18022 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1456;
		case 32: goto tr1456;
		case 40: goto tr1457;
		case 41: goto tr1458;
		case 522: goto tr1460;
		case 525: goto tr1461;
		case 571: goto tr1462;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1459;
	goto st0;
tr1461:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st388;
tr1343:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st388;
st388:
	if ( ++p == pe )
		goto _test_eof388;
case 388:
#line 18070 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st387;
	goto st0;
tr1462:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st389;
tr1344:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st389;
st389:
	if ( ++p == pe )
		goto _test_eof389;
case 389:
#line 18095 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st387;
		case 525: goto st388;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st389;
	goto st0;
tr1467:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st390;
st390:
	if ( ++p == pe )
		goto _test_eof390;
case 390:
#line 18146 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1463;
		case 32: goto tr1463;
		case 40: goto tr1464;
		case 41: goto tr1465;
		case 522: goto tr1467;
		case 525: goto tr1468;
		case 571: goto tr1469;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1466;
	goto st0;
tr1468:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st391;
st391:
	if ( ++p == pe )
		goto _test_eof391;
case 391:
#line 18188 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st390;
	goto st0;
tr1469:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st392;
st392:
	if ( ++p == pe )
		goto _test_eof392;
case 392:
#line 18207 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st390;
		case 525: goto st391;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st392;
	goto st0;
tr21:
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st393;
tr1660:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st393;
tr1689:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 516 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st393;
st393:
	if ( ++p == pe )
		goto _test_eof393;
case 393:
#line 18268 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 88: goto st394;
		case 89: goto st421;
		case 120: goto st394;
		case 121: goto st421;
	}
	goto st0;
st394:
	if ( ++p == pe )
		goto _test_eof394;
case 394:
	switch( (*p) ) {
		case 84: goto st395;
		case 116: goto st395;
	}
	goto st0;
st395:
	if ( ++p == pe )
		goto _test_eof395;
case 395:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st396;
		case 32: goto st396;
		case 40: goto tr1474;
		case 41: goto tr1475;
		case 522: goto st418;
		case 525: goto st419;
		case 571: goto st420;
	}
	goto st0;
tr1474:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st396;
tr1475:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st396;
st396:
	if ( ++p == pe )
		goto _test_eof396;
case 396:
#line 18332 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1480;
		case 32: goto tr1480;
		case 34: goto tr1481;
		case 40: goto tr1482;
		case 41: goto tr1483;
		case 92: goto tr1484;
		case 522: goto tr1485;
		case 525: goto tr1486;
		case 571: goto tr1487;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1479;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1479;
		} else if ( _widec >= 14 )
			goto tr1479;
	} else
		goto tr1479;
	goto st0;
tr1500:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st397;
tr1537:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st397;
tr1549:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st397;
tr1479:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st397;
tr1564:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st397;
tr1522:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st397;
st397:
	if ( ++p == pe )
		goto _test_eof397;
case 397:
#line 18415 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1489;
		case 32: goto tr1489;
		case 34: goto tr1490;
		case 40: goto tr1491;
		case 41: goto tr1492;
		case 92: goto st403;
		case 266: goto tr1494;
		case 269: goto tr1495;
		case 315: goto tr1496;
		case 522: goto tr1497;
		case 525: goto tr1498;
		case 571: goto tr1499;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto st397;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st397;
		} else if ( _widec >= 14 )
			goto st397;
	} else
		goto st397;
	goto st0;
tr1503:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st398;
tr1504:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st398;
tr1550:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st398;
tr1552:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st398;
tr1553:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st398;
tr1489:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st398;
tr1491:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st398;
tr1492:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st398;
tr1523:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st398;
tr1525:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st398;
tr1526:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st398;
tr1538:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st398;
tr1540:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st398;
tr1541:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st398;
st398:
	if ( ++p == pe )
		goto _test_eof398;
case 398:
#line 18545 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st398;
		case 32: goto st398;
		case 34: goto tr1502;
		case 40: goto tr1503;
		case 41: goto tr1504;
		case 92: goto tr1505;
		case 266: goto tr1506;
		case 269: goto tr1507;
		case 315: goto tr1508;
		case 522: goto st405;
		case 525: goto st406;
		case 571: goto st407;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1500;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1500;
		} else if ( _widec >= 14 )
			goto tr1500;
	} else
		goto tr1500;
	goto st0;
tr1502:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st399;
tr1517:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st399;
tr1551:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st399;
tr1481:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st399;
tr1566:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st399;
tr1490:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st399;
tr1524:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st399;
tr1539:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st399;
st399:
	if ( ++p == pe )
		goto _test_eof399;
case 399:
#line 18645 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st400;
		case 13: goto st401;
		case 34: goto st402;
		case 92: goto st411;
	}
	goto st399;
tr1518:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st400;
st400:
	if ( ++p == pe )
		goto _test_eof400;
case 400:
#line 18661 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1518;
		case 13: goto tr1519;
		case 34: goto tr1520;
		case 92: goto tr1521;
	}
	goto tr1517;
tr1519:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st401;
st401:
	if ( ++p == pe )
		goto _test_eof401;
case 401:
#line 18677 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st400;
	goto st0;
tr1520:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st402;
st402:
	if ( ++p == pe )
		goto _test_eof402;
case 402:
#line 18689 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1523;
		case 32: goto tr1523;
		case 34: goto tr1524;
		case 40: goto tr1525;
		case 41: goto tr1526;
		case 92: goto tr1527;
		case 266: goto tr1528;
		case 269: goto tr1529;
		case 315: goto tr1530;
		case 522: goto tr1531;
		case 525: goto tr1532;
		case 571: goto tr1533;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1522;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1522;
		} else if ( _widec >= 14 )
			goto tr1522;
	} else
		goto tr1522;
	goto st0;
tr1505:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st403;
tr1542:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st403;
tr1554:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st403;
tr1484:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st403;
tr1569:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st403;
tr1527:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st403;
st403:
	if ( ++p == pe )
		goto _test_eof403;
case 403:
#line 18775 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st404;
		case 13: goto st408;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st409;
	goto st397;
st404:
	if ( ++p == pe )
		goto _test_eof404;
case 404:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1538;
		case 32: goto tr1538;
		case 34: goto tr1539;
		case 40: goto tr1540;
		case 41: goto tr1541;
		case 92: goto tr1542;
		case 266: goto tr1543;
		case 269: goto tr1544;
		case 315: goto tr1545;
		case 522: goto tr1546;
		case 525: goto tr1547;
		case 571: goto tr1548;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1537;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1537;
		} else if ( _widec >= 14 )
			goto tr1537;
	} else
		goto tr1537;
	goto st0;
tr1558:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st405;
tr1497:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st405;
tr1531:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st405;
tr1546:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st405;
st405:
	if ( ++p == pe )
		goto _test_eof405;
case 405:
#line 18859 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1550;
		case 32: goto tr1550;
		case 34: goto tr1551;
		case 40: goto tr1552;
		case 41: goto tr1553;
		case 92: goto tr1554;
		case 266: goto tr1555;
		case 269: goto tr1556;
		case 315: goto tr1557;
		case 522: goto tr1558;
		case 525: goto tr1559;
		case 571: goto tr1560;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1549;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1549;
		} else if ( _widec >= 14 )
			goto tr1549;
	} else
		goto tr1549;
	goto st0;
tr1559:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st406;
tr1498:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st406;
tr1532:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st406;
tr1547:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st406;
st406:
	if ( ++p == pe )
		goto _test_eof406;
case 406:
#line 18929 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st405;
	goto st0;
tr1560:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st407;
tr1499:
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st407;
tr1533:
#line 505 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st407;
tr1548:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 504 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st407;
st407:
	if ( ++p == pe )
		goto _test_eof407;
case 407:
#line 18962 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st405;
		case 525: goto st406;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st407;
	goto st0;
st408:
	if ( ++p == pe )
		goto _test_eof408;
case 408:
	if ( (*p) == 10 )
		goto st404;
	goto st0;
st409:
	if ( ++p == pe )
		goto _test_eof409;
case 409:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st410;
	goto st0;
st410:
	if ( ++p == pe )
		goto _test_eof410;
case 410:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st397;
	goto st0;
tr1521:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st411;
st411:
	if ( ++p == pe )
		goto _test_eof411;
case 411:
#line 19034 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st400;
		case 13: goto st401;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st412;
	goto st399;
st412:
	if ( ++p == pe )
		goto _test_eof412;
case 412:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st413;
	goto st0;
st413:
	if ( ++p == pe )
		goto _test_eof413;
case 413:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st399;
	goto st0;
tr1480:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st414;
tr1482:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st414;
tr1483:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st414;
tr1565:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st414;
tr1567:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st414;
tr1568:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st414;
st414:
	if ( ++p == pe )
		goto _test_eof414;
case 414:
#line 19101 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1480;
		case 32: goto tr1480;
		case 34: goto tr1481;
		case 40: goto tr1482;
		case 41: goto tr1483;
		case 92: goto tr1484;
		case 266: goto tr1506;
		case 269: goto tr1507;
		case 315: goto tr1508;
		case 522: goto tr1485;
		case 525: goto tr1486;
		case 571: goto tr1487;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1479;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1479;
		} else if ( _widec >= 14 )
			goto tr1479;
	} else
		goto tr1479;
	goto st0;
tr1485:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st415;
tr1570:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st415;
st415:
	if ( ++p == pe )
		goto _test_eof415;
case 415:
#line 19163 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1565;
		case 32: goto tr1565;
		case 34: goto tr1566;
		case 40: goto tr1567;
		case 41: goto tr1568;
		case 92: goto tr1569;
		case 266: goto tr1555;
		case 269: goto tr1556;
		case 315: goto tr1557;
		case 522: goto tr1570;
		case 525: goto tr1571;
		case 571: goto tr1572;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1564;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1564;
		} else if ( _widec >= 14 )
			goto tr1564;
	} else
		goto tr1564;
	goto st0;
tr1486:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st416;
tr1571:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st416;
st416:
	if ( ++p == pe )
		goto _test_eof416;
case 416:
#line 19225 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st415;
	goto st0;
tr1487:
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st417;
tr1572:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st417;
st417:
	if ( ++p == pe )
		goto _test_eof417;
case 417:
#line 19250 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st415;
		case 525: goto st416;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st417;
	goto st0;
st418:
	if ( ++p == pe )
		goto _test_eof418;
case 418:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1565;
		case 32: goto tr1565;
		case 34: goto tr1566;
		case 40: goto tr1567;
		case 41: goto tr1568;
		case 92: goto tr1569;
		case 522: goto tr1570;
		case 525: goto tr1571;
		case 571: goto tr1572;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1564;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1564;
		} else if ( _widec >= 14 )
			goto tr1564;
	} else
		goto tr1564;
	goto st0;
st419:
	if ( ++p == pe )
		goto _test_eof419;
case 419:
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st418;
	goto st0;
st420:
	if ( ++p == pe )
		goto _test_eof420;
case 420:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st418;
		case 525: goto st419;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st420;
	goto st0;
st421:
	if ( ++p == pe )
		goto _test_eof421;
case 421:
	switch( (*p) ) {
		case 80: goto st422;
		case 112: goto st422;
	}
	goto st0;
st422:
	if ( ++p == pe )
		goto _test_eof422;
case 422:
	switch( (*p) ) {
		case 69: goto st423;
		case 101: goto st423;
	}
	goto st0;
st423:
	if ( ++p == pe )
		goto _test_eof423;
case 423:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1578;
	goto st0;
tr1578:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st424;
st424:
	if ( ++p == pe )
		goto _test_eof424;
case 424:
#line 19434 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1579;
		case 32: goto tr1579;
		case 40: goto tr1580;
		case 41: goto tr1581;
		case 522: goto tr1583;
		case 525: goto tr1584;
		case 571: goto tr1585;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st424;
	goto st0;
tr1587:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st425;
tr1588:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st425;
tr1643:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st425;
tr1644:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st425;
tr1645:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st425;
tr1579:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st425;
tr1580:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st425;
tr1581:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st425;
st425:
	if ( ++p == pe )
		goto _test_eof425;
case 425:
#line 19518 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st425;
		case 32: goto st425;
		case 40: goto tr1587;
		case 41: goto tr1588;
		case 92: goto st426;
		case 522: goto st439;
		case 525: goto st440;
		case 571: goto st441;
	}
	goto st0;
tr1646:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st426;
st426:
	if ( ++p == pe )
		goto _test_eof426;
case 426:
#line 19559 "./zscan_rfc1035.c"
	if ( (*p) == 35 )
		goto st427;
	goto st0;
st427:
	if ( ++p == pe )
		goto _test_eof427;
case 427:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st428;
		case 32: goto st428;
		case 40: goto tr1595;
		case 41: goto tr1596;
		case 522: goto st436;
		case 525: goto st437;
		case 571: goto st438;
	}
	goto st0;
tr1595:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st428;
tr1596:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st428;
tr1636:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st428;
tr1637:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st428;
tr1638:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st428;
st428:
	if ( ++p == pe )
		goto _test_eof428;
case 428:
#line 19626 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st428;
		case 32: goto st428;
		case 40: goto tr1595;
		case 41: goto tr1596;
		case 522: goto st436;
		case 525: goto st437;
		case 571: goto st438;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1600;
	goto st0;
tr1600:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st429;
tr1639:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st429;
st429:
	if ( ++p == pe )
		goto _test_eof429;
case 429:
#line 19674 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1601;
		case 32: goto tr1601;
		case 40: goto tr1602;
		case 41: goto tr1603;
		case 266: goto tr1605;
		case 269: goto tr1606;
		case 315: goto tr1607;
		case 522: goto tr1608;
		case 525: goto tr1609;
		case 571: goto tr1610;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st429;
	goto st0;
tr1612:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st430;
tr1613:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st430;
tr1629:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st430;
tr1630:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st430;
tr1631:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st430;
tr1601:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
	goto st430;
tr1602:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st430;
tr1603:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st430;
tr1619:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
	goto st430;
tr1620:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st430;
tr1621:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st430;
st430:
	if ( ++p == pe )
		goto _test_eof430;
case 430:
#line 19777 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st430;
		case 32: goto st430;
		case 40: goto tr1612;
		case 41: goto tr1613;
		case 522: goto st433;
		case 525: goto st434;
		case 571: goto st435;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1614;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1614;
	} else
		goto tr1614;
	goto st0;
tr1614:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st431;
tr1632:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st431;
tr1622:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st431;
st431:
	if ( ++p == pe )
		goto _test_eof431;
case 431:
#line 19837 "./zscan_rfc1035.c"
	if ( (*p) < 65 ) {
		if ( 48 <= (*p) && (*p) <= 57 )
			goto st432;
	} else if ( (*p) > 70 ) {
		if ( 97 <= (*p) && (*p) <= 102 )
			goto st432;
	} else
		goto st432;
	goto st0;
st432:
	if ( ++p == pe )
		goto _test_eof432;
case 432:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1619;
		case 32: goto tr1619;
		case 40: goto tr1620;
		case 41: goto tr1621;
		case 266: goto tr1623;
		case 269: goto tr1624;
		case 315: goto tr1625;
		case 522: goto tr1626;
		case 525: goto tr1627;
		case 571: goto tr1628;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1622;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1622;
	} else
		goto tr1622;
	goto st0;
tr1633:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st433;
tr1608:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
	goto st433;
tr1626:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
	goto st433;
st433:
	if ( ++p == pe )
		goto _test_eof433;
case 433:
#line 19911 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1629;
		case 32: goto tr1629;
		case 40: goto tr1630;
		case 41: goto tr1631;
		case 522: goto tr1633;
		case 525: goto tr1634;
		case 571: goto tr1635;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1632;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1632;
	} else
		goto tr1632;
	goto st0;
tr1634:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st434;
tr1609:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
	goto st434;
tr1627:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
	goto st434;
st434:
	if ( ++p == pe )
		goto _test_eof434;
case 434:
#line 19969 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st433;
	goto st0;
tr1635:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st435;
tr1610:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 545 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
	goto st435;
tr1628:
#line 546 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
	goto st435;
st435:
	if ( ++p == pe )
		goto _test_eof435;
case 435:
#line 19998 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st433;
		case 525: goto st434;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st435;
	goto st0;
tr1640:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st436;
st436:
	if ( ++p == pe )
		goto _test_eof436;
case 436:
#line 20049 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1636;
		case 32: goto tr1636;
		case 40: goto tr1637;
		case 41: goto tr1638;
		case 522: goto tr1640;
		case 525: goto tr1641;
		case 571: goto tr1642;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1639;
	goto st0;
tr1641:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st437;
st437:
	if ( ++p == pe )
		goto _test_eof437;
case 437:
#line 20091 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st436;
	goto st0;
tr1642:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st438;
st438:
	if ( ++p == pe )
		goto _test_eof438;
case 438:
#line 20110 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st436;
		case 525: goto st437;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st438;
	goto st0;
tr1647:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st439;
tr1583:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st439;
st439:
	if ( ++p == pe )
		goto _test_eof439;
case 439:
#line 20167 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1643;
		case 32: goto tr1643;
		case 40: goto tr1644;
		case 41: goto tr1645;
		case 92: goto tr1646;
		case 522: goto tr1647;
		case 525: goto tr1648;
		case 571: goto tr1649;
	}
	goto st0;
tr1648:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st440;
tr1584:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st440;
st440:
	if ( ++p == pe )
		goto _test_eof440;
case 440:
#line 20214 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st439;
	goto st0;
tr1649:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st441;
tr1585:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st441;
st441:
	if ( ++p == pe )
		goto _test_eof441;
case 441:
#line 20239 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st439;
		case 525: goto st440;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st441;
	goto st0;
tr300:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st442;
tr1661:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st442;
tr1667:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st442;
st442:
	if ( ++p == pe )
		goto _test_eof442;
case 442:
#line 20300 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1650;
		case 32: goto tr1650;
		case 40: goto tr1651;
		case 41: goto tr1652;
		case 65: goto tr1653;
		case 67: goto tr1654;
		case 68: goto tr1655;
		case 77: goto tr1656;
		case 78: goto tr1657;
		case 80: goto tr1658;
		case 83: goto tr1659;
		case 84: goto tr1660;
		case 97: goto tr1653;
		case 99: goto tr1654;
		case 100: goto tr1655;
		case 109: goto tr1656;
		case 110: goto tr1657;
		case 112: goto tr1658;
		case 115: goto tr1659;
		case 116: goto tr1660;
		case 522: goto tr1661;
		case 525: goto tr1662;
		case 571: goto tr1663;
	}
	goto st0;
tr301:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st443;
tr1662:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st443;
tr1668:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st443;
st443:
	if ( ++p == pe )
		goto _test_eof443;
case 443:
#line 20366 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st442;
	goto st0;
tr302:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st444;
tr1663:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st444;
tr1669:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st444;
st444:
	if ( ++p == pe )
		goto _test_eof444;
case 444:
#line 20395 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st442;
		case 525: goto st443;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st444;
	goto st0;
tr299:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 512 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st445;
st445:
	if ( ++p == pe )
		goto _test_eof445;
case 445:
#line 20448 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1664;
		case 32: goto tr1664;
		case 40: goto tr1665;
		case 41: goto tr1666;
		case 522: goto tr1667;
		case 525: goto tr1668;
		case 571: goto tr1669;
	}
	goto st0;
tr1674:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st446;
st446:
	if ( ++p == pe )
		goto _test_eof446;
case 446:
#line 20488 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1670;
		case 32: goto tr1670;
		case 40: goto tr1671;
		case 41: goto tr1672;
		case 65: goto tr1653;
		case 67: goto tr1654;
		case 68: goto tr1655;
		case 77: goto tr1656;
		case 78: goto tr1657;
		case 80: goto tr1658;
		case 83: goto tr1659;
		case 84: goto tr1660;
		case 97: goto tr1653;
		case 99: goto tr1654;
		case 100: goto tr1655;
		case 109: goto tr1656;
		case 110: goto tr1657;
		case 112: goto tr1658;
		case 115: goto tr1659;
		case 116: goto tr1660;
		case 522: goto tr1674;
		case 525: goto tr1675;
		case 571: goto tr1676;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1673;
	goto st0;
tr1675:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st447;
st447:
	if ( ++p == pe )
		goto _test_eof447;
case 447:
#line 20546 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st446;
	goto st0;
tr1676:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st448;
st448:
	if ( ++p == pe )
		goto _test_eof448;
case 448:
#line 20565 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st446;
		case 525: goto st447;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st448;
	goto st0;
tr1690:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st449;
st449:
	if ( ++p == pe )
		goto _test_eof449;
case 449:
#line 20616 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1677;
		case 32: goto tr1677;
		case 40: goto tr1678;
		case 41: goto tr1679;
		case 65: goto tr1681;
		case 67: goto tr1682;
		case 68: goto tr1683;
		case 73: goto tr1684;
		case 77: goto tr1685;
		case 78: goto tr1686;
		case 80: goto tr1687;
		case 83: goto tr1688;
		case 84: goto tr1689;
		case 97: goto tr1681;
		case 99: goto tr1682;
		case 100: goto tr1683;
		case 105: goto tr1684;
		case 109: goto tr1685;
		case 110: goto tr1686;
		case 112: goto tr1687;
		case 115: goto tr1688;
		case 116: goto tr1689;
		case 266: goto tr132;
		case 269: goto tr133;
		case 315: goto tr134;
		case 522: goto tr1690;
		case 525: goto tr1691;
		case 571: goto tr1692;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1680;
	goto st0;
tr1691:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st450;
st450:
	if ( ++p == pe )
		goto _test_eof450;
case 450:
#line 20679 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st449;
	goto st0;
tr1692:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st451;
st451:
	if ( ++p == pe )
		goto _test_eof451;
case 451:
#line 20698 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st449;
		case 525: goto st450;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st451;
	goto st0;
tr1984:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st452;
tr1698:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st452;
tr1989:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st452;
st452:
	if ( ++p == pe )
		goto _test_eof452;
case 452:
#line 20759 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st453;
		case 13: goto st454;
		case 34: goto st455;
		case 92: goto st459;
	}
	goto st452;
tr1699:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st453;
st453:
	if ( ++p == pe )
		goto _test_eof453;
case 453:
#line 20775 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1699;
		case 13: goto tr1700;
		case 34: goto tr1701;
		case 92: goto tr1702;
	}
	goto tr1698;
tr1700:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st454;
st454:
	if ( ++p == pe )
		goto _test_eof454;
case 454:
#line 20791 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st453;
	goto st0;
tr1701:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st455;
st455:
	if ( ++p == pe )
		goto _test_eof455;
case 455:
#line 20803 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1703;
		case 32: goto tr1703;
		case 40: goto tr1704;
		case 41: goto tr1705;
		case 522: goto tr1706;
		case 525: goto tr1707;
		case 571: goto tr1708;
	}
	goto st0;
tr6:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st456;
tr1712:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st456;
tr1706:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
	goto st456;
tr1901:
#line 489 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
	goto st456;
tr1912:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st456;
st456:
	if ( ++p == pe )
		goto _test_eof456;
case 456:
#line 20861 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1709;
		case 32: goto tr1709;
		case 40: goto tr1710;
		case 41: goto tr1711;
		case 65: goto tr1681;
		case 67: goto tr1682;
		case 68: goto tr1683;
		case 73: goto tr1684;
		case 77: goto tr1685;
		case 78: goto tr1686;
		case 80: goto tr1687;
		case 83: goto tr1688;
		case 84: goto tr1689;
		case 97: goto tr1681;
		case 99: goto tr1682;
		case 100: goto tr1683;
		case 105: goto tr1684;
		case 109: goto tr1685;
		case 110: goto tr1686;
		case 112: goto tr1687;
		case 115: goto tr1688;
		case 116: goto tr1689;
		case 522: goto tr1712;
		case 525: goto tr1713;
		case 571: goto tr1714;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1680;
	goto st0;
tr7:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st457;
tr1713:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st457;
tr1707:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
	goto st457;
tr1902:
#line 489 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
	goto st457;
tr1913:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st457;
st457:
	if ( ++p == pe )
		goto _test_eof457;
case 457:
#line 20939 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st456;
	goto st0;
tr8:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st458;
tr1714:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st458;
tr1708:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
	goto st458;
tr1903:
#line 489 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
	goto st458;
tr1914:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st458;
st458:
	if ( ++p == pe )
		goto _test_eof458;
case 458:
#line 20976 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st456;
		case 525: goto st457;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st458;
	goto st0;
tr1702:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st459;
st459:
	if ( ++p == pe )
		goto _test_eof459;
case 459:
#line 21027 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st453;
		case 13: goto st454;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st460;
	goto st452;
st460:
	if ( ++p == pe )
		goto _test_eof460;
case 460:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st461;
	goto st0;
st461:
	if ( ++p == pe )
		goto _test_eof461;
case 461:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st452;
	goto st0;
tr1990:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st462;
st462:
	if ( ++p == pe )
		goto _test_eof462;
case 462:
#line 21060 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 65: goto st463;
		case 79: goto st487;
		case 84: goto st511;
		case 97: goto st463;
		case 111: goto st487;
		case 116: goto st511;
	}
	goto st0;
st463:
	if ( ++p == pe )
		goto _test_eof463;
case 463:
	switch( (*p) ) {
		case 68: goto st464;
		case 100: goto st464;
	}
	goto st0;
st464:
	if ( ++p == pe )
		goto _test_eof464;
case 464:
	switch( (*p) ) {
		case 68: goto st465;
		case 100: goto st465;
	}
	goto st0;
st465:
	if ( ++p == pe )
		goto _test_eof465;
case 465:
	switch( (*p) ) {
		case 82: goto st466;
		case 114: goto st466;
	}
	goto st0;
st466:
	if ( ++p == pe )
		goto _test_eof466;
case 466:
	if ( (*p) == 95 )
		goto st467;
	goto st0;
st467:
	if ( ++p == pe )
		goto _test_eof467;
case 467:
	switch( (*p) ) {
		case 76: goto st468;
		case 108: goto st468;
	}
	goto st0;
st468:
	if ( ++p == pe )
		goto _test_eof468;
case 468:
	switch( (*p) ) {
		case 73: goto st469;
		case 105: goto st469;
	}
	goto st0;
st469:
	if ( ++p == pe )
		goto _test_eof469;
case 469:
	switch( (*p) ) {
		case 77: goto st470;
		case 109: goto st470;
	}
	goto st0;
st470:
	if ( ++p == pe )
		goto _test_eof470;
case 470:
	switch( (*p) ) {
		case 73: goto st471;
		case 105: goto st471;
	}
	goto st0;
st471:
	if ( ++p == pe )
		goto _test_eof471;
case 471:
	switch( (*p) ) {
		case 84: goto st472;
		case 116: goto st472;
	}
	goto st0;
st472:
	if ( ++p == pe )
		goto _test_eof472;
case 472:
	if ( (*p) == 95 )
		goto st473;
	goto st0;
st473:
	if ( ++p == pe )
		goto _test_eof473;
case 473:
	switch( (*p) ) {
		case 86: goto st474;
		case 118: goto st474;
	}
	goto st0;
st474:
	if ( ++p == pe )
		goto _test_eof474;
case 474:
	switch( (*p) ) {
		case 52: goto st475;
		case 54: goto st481;
	}
	goto st0;
st475:
	if ( ++p == pe )
		goto _test_eof475;
case 475:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st476;
		case 32: goto st476;
		case 40: goto tr1734;
		case 41: goto tr1735;
		case 522: goto st478;
		case 525: goto st479;
		case 571: goto st480;
	}
	goto st0;
tr1734:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st476;
tr1735:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st476;
tr1750:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st476;
tr1751:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st476;
tr1752:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st476;
st476:
	if ( ++p == pe )
		goto _test_eof476;
case 476:
#line 21237 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st476;
		case 32: goto st476;
		case 40: goto tr1734;
		case 41: goto tr1735;
		case 522: goto st478;
		case 525: goto st479;
		case 571: goto st480;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1739;
	goto st0;
tr1739:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st477;
tr1753:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st477;
st477:
	if ( ++p == pe )
		goto _test_eof477;
case 477:
#line 21285 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1740;
		case 32: goto tr1740;
		case 40: goto tr1741;
		case 41: goto tr1742;
		case 266: goto tr1744;
		case 269: goto tr1745;
		case 315: goto tr1746;
		case 522: goto tr1747;
		case 525: goto tr1748;
		case 571: goto tr1749;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st477;
	goto st0;
tr1754:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st478;
st478:
	if ( ++p == pe )
		goto _test_eof478;
case 478:
#line 21330 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1750;
		case 32: goto tr1750;
		case 40: goto tr1751;
		case 41: goto tr1752;
		case 522: goto tr1754;
		case 525: goto tr1755;
		case 571: goto tr1756;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1753;
	goto st0;
tr1755:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st479;
st479:
	if ( ++p == pe )
		goto _test_eof479;
case 479:
#line 21372 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st478;
	goto st0;
tr1756:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st480;
st480:
	if ( ++p == pe )
		goto _test_eof480;
case 480:
#line 21391 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st478;
		case 525: goto st479;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st480;
	goto st0;
st481:
	if ( ++p == pe )
		goto _test_eof481;
case 481:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st482;
		case 32: goto st482;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 522: goto st484;
		case 525: goto st485;
		case 571: goto st486;
	}
	goto st0;
tr1758:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st482;
tr1759:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st482;
tr1774:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st482;
tr1775:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st482;
tr1776:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st482;
st482:
	if ( ++p == pe )
		goto _test_eof482;
case 482:
#line 21497 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st482;
		case 32: goto st482;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 522: goto st484;
		case 525: goto st485;
		case 571: goto st486;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1763;
	goto st0;
tr1763:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st483;
tr1777:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st483;
st483:
	if ( ++p == pe )
		goto _test_eof483;
case 483:
#line 21545 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1764;
		case 32: goto tr1764;
		case 40: goto tr1765;
		case 41: goto tr1766;
		case 266: goto tr1768;
		case 269: goto tr1769;
		case 315: goto tr1770;
		case 522: goto tr1771;
		case 525: goto tr1772;
		case 571: goto tr1773;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st483;
	goto st0;
tr1778:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st484;
st484:
	if ( ++p == pe )
		goto _test_eof484;
case 484:
#line 21590 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1774;
		case 32: goto tr1774;
		case 40: goto tr1775;
		case 41: goto tr1776;
		case 522: goto tr1778;
		case 525: goto tr1779;
		case 571: goto tr1780;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1777;
	goto st0;
tr1779:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st485;
st485:
	if ( ++p == pe )
		goto _test_eof485;
case 485:
#line 21632 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st484;
	goto st0;
tr1780:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st486;
st486:
	if ( ++p == pe )
		goto _test_eof486;
case 486:
#line 21651 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st484;
		case 525: goto st485;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st486;
	goto st0;
st487:
	if ( ++p == pe )
		goto _test_eof487;
case 487:
	switch( (*p) ) {
		case 82: goto st488;
		case 114: goto st488;
	}
	goto st0;
st488:
	if ( ++p == pe )
		goto _test_eof488;
case 488:
	switch( (*p) ) {
		case 73: goto st489;
		case 105: goto st489;
	}
	goto st0;
st489:
	if ( ++p == pe )
		goto _test_eof489;
case 489:
	switch( (*p) ) {
		case 71: goto st490;
		case 103: goto st490;
	}
	goto st0;
st490:
	if ( ++p == pe )
		goto _test_eof490;
case 490:
	switch( (*p) ) {
		case 73: goto st491;
		case 105: goto st491;
	}
	goto st0;
st491:
	if ( ++p == pe )
		goto _test_eof491;
case 491:
	switch( (*p) ) {
		case 78: goto st492;
		case 110: goto st492;
	}
	goto st0;
st492:
	if ( ++p == pe )
		goto _test_eof492;
case 492:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st493;
		case 32: goto st493;
		case 40: goto tr1787;
		case 41: goto tr1788;
		case 522: goto st508;
		case 525: goto st509;
		case 571: goto st510;
	}
	goto st0;
tr1787:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st493;
tr1788:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st493;
tr1853:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st493;
tr1855:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st493;
tr1856:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st493;
st493:
	if ( ++p == pe )
		goto _test_eof493;
case 493:
#line 21802 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st493;
		case 32: goto st493;
		case 34: goto tr1793;
		case 40: goto tr1787;
		case 41: goto tr1788;
		case 64: goto tr1794;
		case 92: goto tr1795;
		case 522: goto st508;
		case 525: goto st509;
		case 571: goto st510;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1792;
		} else
			goto tr1792;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1792;
		} else if ( _widec >= 37 )
			goto tr1792;
	} else
		goto tr1792;
	goto st0;
tr1792:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st494;
tr1810:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st494;
tr1852:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st494;
st494:
	if ( ++p == pe )
		goto _test_eof494;
case 494:
#line 21869 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1797;
		case 32: goto tr1797;
		case 40: goto tr1798;
		case 41: goto tr1799;
		case 92: goto st495;
		case 266: goto tr1801;
		case 269: goto tr1802;
		case 315: goto tr1803;
		case 522: goto tr1804;
		case 525: goto tr1805;
		case 571: goto tr1806;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st494;
		} else
			goto st494;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st494;
		} else if ( _widec >= 35 )
			goto st494;
	} else
		goto st494;
	goto st0;
tr1795:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st495;
tr1814:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st495;
tr1858:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st495;
st495:
	if ( ++p == pe )
		goto _test_eof495;
case 495:
#line 21937 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st496;
		case 13: goto st497;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st498;
	goto st494;
st496:
	if ( ++p == pe )
		goto _test_eof496;
case 496:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1811;
		case 32: goto tr1811;
		case 40: goto tr1812;
		case 41: goto tr1813;
		case 92: goto tr1814;
		case 266: goto tr1815;
		case 269: goto tr1816;
		case 315: goto tr1817;
		case 522: goto tr1818;
		case 525: goto tr1819;
		case 571: goto tr1820;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1810;
		} else
			goto tr1810;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1810;
		} else if ( _widec >= 35 )
			goto tr1810;
	} else
		goto tr1810;
	goto st0;
st497:
	if ( ++p == pe )
		goto _test_eof497;
case 497:
	if ( (*p) == 10 )
		goto st496;
	goto st0;
st498:
	if ( ++p == pe )
		goto _test_eof498;
case 498:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st499;
	goto st0;
st499:
	if ( ++p == pe )
		goto _test_eof499;
case 499:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st494;
	goto st0;
tr1793:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st500;
tr1827:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st500;
tr1854:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st500;
st500:
	if ( ++p == pe )
		goto _test_eof500;
case 500:
#line 22040 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st501;
		case 13: goto st502;
		case 34: goto st503;
		case 92: goto st504;
	}
	goto st500;
tr1828:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st501;
st501:
	if ( ++p == pe )
		goto _test_eof501;
case 501:
#line 22056 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1828;
		case 13: goto tr1829;
		case 34: goto tr1830;
		case 92: goto tr1831;
	}
	goto tr1827;
tr1829:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st502;
st502:
	if ( ++p == pe )
		goto _test_eof502;
case 502:
#line 22072 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st501;
	goto st0;
tr1830:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st503;
st503:
	if ( ++p == pe )
		goto _test_eof503;
case 503:
#line 22084 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1832;
		case 32: goto tr1832;
		case 40: goto tr1833;
		case 41: goto tr1834;
		case 266: goto tr1835;
		case 269: goto tr1836;
		case 315: goto tr1837;
		case 522: goto tr1838;
		case 525: goto tr1839;
		case 571: goto tr1840;
	}
	goto st0;
tr1831:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st504;
st504:
	if ( ++p == pe )
		goto _test_eof504;
case 504:
#line 22127 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st501;
		case 13: goto st502;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st505;
	goto st500;
st505:
	if ( ++p == pe )
		goto _test_eof505;
case 505:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st506;
	goto st0;
st506:
	if ( ++p == pe )
		goto _test_eof506;
case 506:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st500;
	goto st0;
tr1794:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st507;
tr1857:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st507;
st507:
	if ( ++p == pe )
		goto _test_eof507;
case 507:
#line 22166 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1843;
		case 32: goto tr1843;
		case 40: goto tr1844;
		case 41: goto tr1845;
		case 266: goto tr1846;
		case 269: goto tr1847;
		case 315: goto tr1848;
		case 522: goto tr1849;
		case 525: goto tr1850;
		case 571: goto tr1851;
	}
	goto st0;
tr1859:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st508;
st508:
	if ( ++p == pe )
		goto _test_eof508;
case 508:
#line 22209 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1853;
		case 32: goto tr1853;
		case 34: goto tr1854;
		case 40: goto tr1855;
		case 41: goto tr1856;
		case 64: goto tr1857;
		case 92: goto tr1858;
		case 522: goto tr1859;
		case 525: goto tr1860;
		case 571: goto tr1861;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1852;
		} else
			goto tr1852;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1852;
		} else if ( _widec >= 37 )
			goto tr1852;
	} else
		goto tr1852;
	goto st0;
tr1860:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st509;
st509:
	if ( ++p == pe )
		goto _test_eof509;
case 509:
#line 22266 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st508;
	goto st0;
tr1861:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st510;
st510:
	if ( ++p == pe )
		goto _test_eof510;
case 510:
#line 22285 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st508;
		case 525: goto st509;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st510;
	goto st0;
st511:
	if ( ++p == pe )
		goto _test_eof511;
case 511:
	switch( (*p) ) {
		case 84: goto st512;
		case 116: goto st512;
	}
	goto st0;
st512:
	if ( ++p == pe )
		goto _test_eof512;
case 512:
	switch( (*p) ) {
		case 76: goto st513;
		case 108: goto st513;
	}
	goto st0;
st513:
	if ( ++p == pe )
		goto _test_eof513;
case 513:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st514;
		case 32: goto st514;
		case 40: goto tr1865;
		case 41: goto tr1866;
		case 522: goto st517;
		case 525: goto st518;
		case 571: goto st519;
	}
	goto st0;
tr1865:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st514;
tr1866:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st514;
tr1891:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st514;
tr1892:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st514;
tr1893:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st514;
st514:
	if ( ++p == pe )
		goto _test_eof514;
case 514:
#line 22409 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st514;
		case 32: goto st514;
		case 40: goto tr1865;
		case 41: goto tr1866;
		case 522: goto st517;
		case 525: goto st518;
		case 571: goto st519;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1870;
	goto st0;
tr1870:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st515;
tr1894:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st515;
st515:
	if ( ++p == pe )
		goto _test_eof515;
case 515:
#line 22457 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1871;
		case 32: goto tr1871;
		case 40: goto tr1872;
		case 41: goto tr1873;
		case 68: goto tr1875;
		case 72: goto tr1875;
		case 77: goto tr1875;
		case 87: goto tr1875;
		case 100: goto tr1875;
		case 104: goto tr1875;
		case 109: goto tr1875;
		case 119: goto tr1875;
		case 266: goto tr1876;
		case 269: goto tr1877;
		case 315: goto tr1878;
		case 522: goto tr1879;
		case 525: goto tr1880;
		case 571: goto tr1881;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st515;
	goto st0;
tr1875:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 512 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st516;
st516:
	if ( ++p == pe )
		goto _test_eof516;
case 516:
#line 22512 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1882;
		case 32: goto tr1882;
		case 40: goto tr1883;
		case 41: goto tr1884;
		case 266: goto tr1885;
		case 269: goto tr1886;
		case 315: goto tr1887;
		case 522: goto tr1888;
		case 525: goto tr1889;
		case 571: goto tr1890;
	}
	goto st0;
tr1895:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st517;
st517:
	if ( ++p == pe )
		goto _test_eof517;
case 517:
#line 22555 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1891;
		case 32: goto tr1891;
		case 40: goto tr1892;
		case 41: goto tr1893;
		case 522: goto tr1895;
		case 525: goto tr1896;
		case 571: goto tr1897;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1894;
	goto st0;
tr1896:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st518;
st518:
	if ( ++p == pe )
		goto _test_eof518;
case 518:
#line 22597 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st517;
	goto st0;
tr1897:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st519;
st519:
	if ( ++p == pe )
		goto _test_eof519;
case 519:
#line 22616 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st517;
		case 525: goto st518;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st519;
	goto st0;
tr1986:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st520;
tr1991:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st520;
st520:
	if ( ++p == pe )
		goto _test_eof520;
case 520:
#line 22673 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1898;
		case 32: goto tr1898;
		case 40: goto tr1899;
		case 41: goto tr1900;
		case 522: goto tr1901;
		case 525: goto tr1902;
		case 571: goto tr1903;
	}
	goto st0;
tr1987:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st521;
tr1911:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st521;
tr1992:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st521;
st521:
	if ( ++p == pe )
		goto _test_eof521;
case 521:
#line 22723 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st522;
		case 13: goto st523;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st524;
	goto st1;
st522:
	if ( ++p == pe )
		goto _test_eof522;
case 522:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1908;
		case 32: goto tr1908;
		case 40: goto tr1909;
		case 41: goto tr1910;
		case 92: goto tr1911;
		case 522: goto tr1912;
		case 525: goto tr1913;
		case 571: goto tr1914;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1907;
		} else
			goto tr1907;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1907;
		} else if ( _widec >= 35 )
			goto tr1907;
	} else
		goto tr1907;
	goto st0;
st523:
	if ( ++p == pe )
		goto _test_eof523;
case 523:
	if ( (*p) == 10 )
		goto st522;
	goto st0;
st524:
	if ( ++p == pe )
		goto _test_eof524;
case 524:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st525;
	goto st0;
st525:
	if ( ++p == pe )
		goto _test_eof525;
case 525:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st1;
	goto st0;
st526:
	if ( ++p == pe )
		goto _test_eof526;
case 526:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr65;
		case 32: goto tr65;
		case 40: goto tr66;
		case 41: goto tr67;
		case 266: goto tr69;
		case 269: goto tr70;
		case 315: goto tr71;
		case 522: goto tr72;
		case 525: goto tr73;
		case 571: goto tr74;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st527;
	goto st0;
st527:
	if ( ++p == pe )
		goto _test_eof527;
case 527:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr65;
		case 32: goto tr65;
		case 40: goto tr66;
		case 41: goto tr67;
		case 266: goto tr69;
		case 269: goto tr70;
		case 315: goto tr71;
		case 522: goto tr72;
		case 525: goto tr73;
		case 571: goto tr74;
	}
	goto st0;
st528:
	if ( ++p == pe )
		goto _test_eof528;
case 528:
	if ( (*p) == 46 )
		goto st12;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st529;
	goto st0;
st529:
	if ( ++p == pe )
		goto _test_eof529;
case 529:
	if ( (*p) == 46 )
		goto st12;
	goto st0;
st530:
	if ( ++p == pe )
		goto _test_eof530;
case 530:
	if ( (*p) == 46 )
		goto st10;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st531;
	goto st0;
st531:
	if ( ++p == pe )
		goto _test_eof531;
case 531:
	if ( (*p) == 46 )
		goto st10;
	goto st0;
st532:
	if ( ++p == pe )
		goto _test_eof532;
case 532:
	if ( (*p) == 46 )
		goto st8;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st533;
	goto st0;
st533:
	if ( ++p == pe )
		goto _test_eof533;
case 533:
	if ( (*p) == 46 )
		goto st8;
	goto st0;
tr1924:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st534;
st534:
	if ( ++p == pe )
		goto _test_eof534;
case 534:
#line 22939 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1920;
		case 32: goto tr1920;
		case 40: goto tr1921;
		case 41: goto tr1922;
		case 522: goto tr1924;
		case 525: goto tr1925;
		case 571: goto tr1926;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1923;
	goto st0;
tr1925:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st535;
st535:
	if ( ++p == pe )
		goto _test_eof535;
case 535:
#line 22981 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st534;
	goto st0;
tr1926:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st536;
st536:
	if ( ++p == pe )
		goto _test_eof536;
case 536:
#line 23000 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st534;
		case 525: goto st535;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st536;
	goto st0;
st537:
	if ( ++p == pe )
		goto _test_eof537;
case 537:
	switch( (*p) ) {
		case 65: goto st538;
		case 97: goto st538;
	}
	goto st0;
st538:
	if ( ++p == pe )
		goto _test_eof538;
case 538:
	switch( (*p) ) {
		case 65: goto st539;
		case 97: goto st539;
	}
	goto st0;
st539:
	if ( ++p == pe )
		goto _test_eof539;
case 539:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st540;
		case 32: goto st540;
		case 40: goto tr1930;
		case 41: goto tr1931;
		case 522: goto st558;
		case 525: goto st559;
		case 571: goto st560;
	}
	goto st0;
tr1930:
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st540;
tr1931:
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st540;
tr1962:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st540;
tr1963:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 547 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st540;
tr1964:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 548 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st540;
st540:
	if ( ++p == pe )
		goto _test_eof540;
case 540:
#line 23124 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st540;
		case 32: goto st540;
		case 40: goto tr1930;
		case 41: goto tr1931;
		case 522: goto st558;
		case 525: goto st559;
		case 571: goto st560;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 58 )
			goto tr1935;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1935;
	} else
		goto tr1935;
	goto st0;
tr1935:
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st541;
tr1965:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 486 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st541;
st541:
	if ( ++p == pe )
		goto _test_eof541;
case 541:
#line 23178 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1936;
		case 32: goto tr1936;
		case 40: goto tr1937;
		case 41: goto tr1938;
		case 58: goto st542;
		case 266: goto tr1941;
		case 269: goto tr1942;
		case 315: goto tr1943;
		case 522: goto tr1944;
		case 525: goto tr1945;
		case 571: goto tr1946;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st541;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st541;
	} else
		goto st541;
	goto st0;
st542:
	if ( ++p == pe )
		goto _test_eof542;
case 542:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1936;
		case 32: goto tr1936;
		case 40: goto tr1937;
		case 41: goto tr1938;
		case 58: goto st542;
		case 266: goto tr1941;
		case 269: goto tr1942;
		case 315: goto tr1943;
		case 522: goto tr1944;
		case 525: goto tr1945;
		case 571: goto tr1946;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st543;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st541;
	} else
		goto st541;
	goto st0;
st543:
	if ( ++p == pe )
		goto _test_eof543;
case 543:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1936;
		case 32: goto tr1936;
		case 40: goto tr1937;
		case 41: goto tr1938;
		case 46: goto st544;
		case 58: goto st542;
		case 266: goto tr1941;
		case 269: goto tr1942;
		case 315: goto tr1943;
		case 522: goto tr1944;
		case 525: goto tr1945;
		case 571: goto tr1946;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st556;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st541;
	} else
		goto st541;
	goto st0;
st544:
	if ( ++p == pe )
		goto _test_eof544;
case 544:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st545;
	goto st0;
st545:
	if ( ++p == pe )
		goto _test_eof545;
case 545:
	if ( (*p) == 46 )
		goto st546;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st554;
	goto st0;
st546:
	if ( ++p == pe )
		goto _test_eof546;
case 546:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st547;
	goto st0;
st547:
	if ( ++p == pe )
		goto _test_eof547;
case 547:
	if ( (*p) == 46 )
		goto st548;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st552;
	goto st0;
st548:
	if ( ++p == pe )
		goto _test_eof548;
case 548:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st549;
	goto st0;
st549:
	if ( ++p == pe )
		goto _test_eof549;
case 549:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1936;
		case 32: goto tr1936;
		case 40: goto tr1937;
		case 41: goto tr1938;
		case 266: goto tr1941;
		case 269: goto tr1942;
		case 315: goto tr1943;
		case 522: goto tr1944;
		case 525: goto tr1945;
		case 571: goto tr1946;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st550;
	goto st0;
st550:
	if ( ++p == pe )
		goto _test_eof550;
case 550:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1936;
		case 32: goto tr1936;
		case 40: goto tr1937;
		case 41: goto tr1938;
		case 266: goto tr1941;
		case 269: goto tr1942;
		case 315: goto tr1943;
		case 522: goto tr1944;
		case 525: goto tr1945;
		case 571: goto tr1946;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st551;
	goto st0;
st551:
	if ( ++p == pe )
		goto _test_eof551;
case 551:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1936;
		case 32: goto tr1936;
		case 40: goto tr1937;
		case 41: goto tr1938;
		case 266: goto tr1941;
		case 269: goto tr1942;
		case 315: goto tr1943;
		case 522: goto tr1944;
		case 525: goto tr1945;
		case 571: goto tr1946;
	}
	goto st0;
st552:
	if ( ++p == pe )
		goto _test_eof552;
case 552:
	if ( (*p) == 46 )
		goto st548;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st553;
	goto st0;
st553:
	if ( ++p == pe )
		goto _test_eof553;
case 553:
	if ( (*p) == 46 )
		goto st548;
	goto st0;
st554:
	if ( ++p == pe )
		goto _test_eof554;
case 554:
	if ( (*p) == 46 )
		goto st546;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st555;
	goto st0;
st555:
	if ( ++p == pe )
		goto _test_eof555;
case 555:
	if ( (*p) == 46 )
		goto st546;
	goto st0;
st556:
	if ( ++p == pe )
		goto _test_eof556;
case 556:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1936;
		case 32: goto tr1936;
		case 40: goto tr1937;
		case 41: goto tr1938;
		case 46: goto st544;
		case 58: goto st542;
		case 266: goto tr1941;
		case 269: goto tr1942;
		case 315: goto tr1943;
		case 522: goto tr1944;
		case 525: goto tr1945;
		case 571: goto tr1946;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st557;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st541;
	} else
		goto st541;
	goto st0;
st557:
	if ( ++p == pe )
		goto _test_eof557;
case 557:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1936;
		case 32: goto tr1936;
		case 40: goto tr1937;
		case 41: goto tr1938;
		case 46: goto st544;
		case 58: goto st542;
		case 266: goto tr1941;
		case 269: goto tr1942;
		case 315: goto tr1943;
		case 522: goto tr1944;
		case 525: goto tr1945;
		case 571: goto tr1946;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st541;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st541;
	} else
		goto st541;
	goto st0;
tr1966:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st558;
st558:
	if ( ++p == pe )
		goto _test_eof558;
case 558:
#line 23610 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1962;
		case 32: goto tr1962;
		case 40: goto tr1963;
		case 41: goto tr1964;
		case 522: goto tr1966;
		case 525: goto tr1967;
		case 571: goto tr1968;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 58 )
			goto tr1965;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1965;
	} else
		goto tr1965;
	goto st0;
tr1967:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st559;
st559:
	if ( ++p == pe )
		goto _test_eof559;
case 559:
#line 23658 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st558;
	goto st0;
tr1968:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st560;
st560:
	if ( ++p == pe )
		goto _test_eof560;
case 560:
#line 23677 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st558;
		case 525: goto st559;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st560;
	goto st0;
tr1973:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st561;
st561:
	if ( ++p == pe )
		goto _test_eof561;
case 561:
#line 23728 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 78: goto st562;
		case 110: goto st562;
	}
	goto st0;
st562:
	if ( ++p == pe )
		goto _test_eof562;
case 562:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st87;
		case 32: goto st87;
		case 40: goto tr304;
		case 41: goto tr305;
		case 522: goto st442;
		case 525: goto st443;
		case 571: goto st444;
	}
	goto st0;
tr30:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st563;
tr1974:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st563;
tr1980:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st563;
st563:
	if ( ++p == pe )
		goto _test_eof563;
case 563:
#line 23787 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1970;
		case 32: goto tr1970;
		case 40: goto tr1971;
		case 41: goto tr1972;
		case 65: goto tr1653;
		case 67: goto tr1654;
		case 68: goto tr1655;
		case 73: goto tr1973;
		case 77: goto tr1656;
		case 78: goto tr1657;
		case 80: goto tr1658;
		case 83: goto tr1659;
		case 84: goto tr1660;
		case 97: goto tr1653;
		case 99: goto tr1654;
		case 100: goto tr1655;
		case 105: goto tr1973;
		case 109: goto tr1656;
		case 110: goto tr1657;
		case 112: goto tr1658;
		case 115: goto tr1659;
		case 116: goto tr1660;
		case 522: goto tr1974;
		case 525: goto tr1975;
		case 571: goto tr1976;
	}
	goto st0;
tr31:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st564;
tr1975:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st564;
tr1981:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st564;
st564:
	if ( ++p == pe )
		goto _test_eof564;
case 564:
#line 23855 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st563;
	goto st0;
tr32:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st565;
tr1976:
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st565;
tr1982:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
	goto st565;
st565:
	if ( ++p == pe )
		goto _test_eof565;
case 565:
#line 23884 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st563;
		case 525: goto st564;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st565;
	goto st0;
tr29:
#line 511 "./zscan_rfc1035.rl"
	{ z->uval = atoi(z->tstart); z->tstart = NULL; }
#line 512 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st566;
st566:
	if ( ++p == pe )
		goto _test_eof566;
case 566:
#line 23937 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 549 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1977;
		case 32: goto tr1977;
		case 40: goto tr1978;
		case 41: goto tr1979;
		case 522: goto tr1980;
		case 525: goto tr1981;
		case 571: goto tr1982;
	}
	goto st0;
	}
	_test_eof1: cs = 1; goto _test_eof; 
	_test_eof2: cs = 2; goto _test_eof; 
	_test_eof3: cs = 3; goto _test_eof; 
	_test_eof4: cs = 4; goto _test_eof; 
	_test_eof5: cs = 5; goto _test_eof; 
	_test_eof6: cs = 6; goto _test_eof; 
	_test_eof7: cs = 7; goto _test_eof; 
	_test_eof8: cs = 8; goto _test_eof; 
	_test_eof9: cs = 9; goto _test_eof; 
	_test_eof10: cs = 10; goto _test_eof; 
	_test_eof11: cs = 11; goto _test_eof; 
	_test_eof12: cs = 12; goto _test_eof; 
	_test_eof13: cs = 13; goto _test_eof; 
	_test_eof14: cs = 14; goto _test_eof; 
	_test_eof568: cs = 568; goto _test_eof; 
	_test_eof15: cs = 15; goto _test_eof; 
	_test_eof16: cs = 16; goto _test_eof; 
	_test_eof17: cs = 17; goto _test_eof; 
	_test_eof18: cs = 18; goto _test_eof; 
	_test_eof19: cs = 19; goto _test_eof; 
	_test_eof20: cs = 20; goto _test_eof; 
	_test_eof21: cs = 21; goto _test_eof; 
	_test_eof22: cs = 22; goto _test_eof; 
	_test_eof23: cs = 23; goto _test_eof; 
	_test_eof24: cs = 24; goto _test_eof; 
	_test_eof25: cs = 25; goto _test_eof; 
	_test_eof26: cs = 26; goto _test_eof; 
	_test_eof27: cs = 27; goto _test_eof; 
	_test_eof28: cs = 28; goto _test_eof; 
	_test_eof29: cs = 29; goto _test_eof; 
	_test_eof30: cs = 30; goto _test_eof; 
	_test_eof31: cs = 31; goto _test_eof; 
	_test_eof32: cs = 32; goto _test_eof; 
	_test_eof33: cs = 33; goto _test_eof; 
	_test_eof34: cs = 34; goto _test_eof; 
	_test_eof35: cs = 35; goto _test_eof; 
	_test_eof36: cs = 36; goto _test_eof; 
	_test_eof37: cs = 37; goto _test_eof; 
	_test_eof38: cs = 38; goto _test_eof; 
	_test_eof39: cs = 39; goto _test_eof; 
	_test_eof40: cs = 40; goto _test_eof; 
	_test_eof41: cs = 41; goto _test_eof; 
	_test_eof42: cs = 42; goto _test_eof; 
	_test_eof43: cs = 43; goto _test_eof; 
	_test_eof44: cs = 44; goto _test_eof; 
	_test_eof45: cs = 45; goto _test_eof; 
	_test_eof46: cs = 46; goto _test_eof; 
	_test_eof47: cs = 47; goto _test_eof; 
	_test_eof48: cs = 48; goto _test_eof; 
	_test_eof49: cs = 49; goto _test_eof; 
	_test_eof50: cs = 50; goto _test_eof; 
	_test_eof51: cs = 51; goto _test_eof; 
	_test_eof52: cs = 52; goto _test_eof; 
	_test_eof53: cs = 53; goto _test_eof; 
	_test_eof54: cs = 54; goto _test_eof; 
	_test_eof55: cs = 55; goto _test_eof; 
	_test_eof56: cs = 56; goto _test_eof; 
	_test_eof57: cs = 57; goto _test_eof; 
	_test_eof58: cs = 58; goto _test_eof; 
	_test_eof59: cs = 59; goto _test_eof; 
	_test_eof60: cs = 60; goto _test_eof; 
	_test_eof61: cs = 61; goto _test_eof; 
	_test_eof62: cs = 62; goto _test_eof; 
	_test_eof63: cs = 63; goto _test_eof; 
	_test_eof64: cs = 64; goto _test_eof; 
	_test_eof65: cs = 65; goto _test_eof; 
	_test_eof66: cs = 66; goto _test_eof; 
	_test_eof67: cs = 67; goto _test_eof; 
	_test_eof68: cs = 68; goto _test_eof; 
	_test_eof69: cs = 69; goto _test_eof; 
	_test_eof70: cs = 70; goto _test_eof; 
	_test_eof71: cs = 71; goto _test_eof; 
	_test_eof72: cs = 72; goto _test_eof; 
	_test_eof73: cs = 73; goto _test_eof; 
	_test_eof74: cs = 74; goto _test_eof; 
	_test_eof75: cs = 75; goto _test_eof; 
	_test_eof76: cs = 76; goto _test_eof; 
	_test_eof77: cs = 77; goto _test_eof; 
	_test_eof78: cs = 78; goto _test_eof; 
	_test_eof79: cs = 79; goto _test_eof; 
	_test_eof80: cs = 80; goto _test_eof; 
	_test_eof81: cs = 81; goto _test_eof; 
	_test_eof82: cs = 82; goto _test_eof; 
	_test_eof83: cs = 83; goto _test_eof; 
	_test_eof84: cs = 84; goto _test_eof; 
	_test_eof85: cs = 85; goto _test_eof; 
	_test_eof86: cs = 86; goto _test_eof; 
	_test_eof87: cs = 87; goto _test_eof; 
	_test_eof88: cs = 88; goto _test_eof; 
	_test_eof89: cs = 89; goto _test_eof; 
	_test_eof90: cs = 90; goto _test_eof; 
	_test_eof91: cs = 91; goto _test_eof; 
	_test_eof92: cs = 92; goto _test_eof; 
	_test_eof93: cs = 93; goto _test_eof; 
	_test_eof94: cs = 94; goto _test_eof; 
	_test_eof95: cs = 95; goto _test_eof; 
	_test_eof96: cs = 96; goto _test_eof; 
	_test_eof97: cs = 97; goto _test_eof; 
	_test_eof98: cs = 98; goto _test_eof; 
	_test_eof99: cs = 99; goto _test_eof; 
	_test_eof100: cs = 100; goto _test_eof; 
	_test_eof101: cs = 101; goto _test_eof; 
	_test_eof102: cs = 102; goto _test_eof; 
	_test_eof103: cs = 103; goto _test_eof; 
	_test_eof104: cs = 104; goto _test_eof; 
	_test_eof105: cs = 105; goto _test_eof; 
	_test_eof106: cs = 106; goto _test_eof; 
	_test_eof107: cs = 107; goto _test_eof; 
	_test_eof108: cs = 108; goto _test_eof; 
	_test_eof109: cs = 109; goto _test_eof; 
	_test_eof110: cs = 110; goto _test_eof; 
	_test_eof111: cs = 111; goto _test_eof; 
	_test_eof112: cs = 112; goto _test_eof; 
	_test_eof113: cs = 113; goto _test_eof; 
	_test_eof114: cs = 114; goto _test_eof; 
	_test_eof115: cs = 115; goto _test_eof; 
	_test_eof116: cs = 116; goto _test_eof; 
	_test_eof117: cs = 117; goto _test_eof; 
	_test_eof118: cs = 118; goto _test_eof; 
	_test_eof119: cs = 119; goto _test_eof; 
	_test_eof120: cs = 120; goto _test_eof; 
	_test_eof121: cs = 121; goto _test_eof; 
	_test_eof122: cs = 122; goto _test_eof; 
	_test_eof123: cs = 123; goto _test_eof; 
	_test_eof124: cs = 124; goto _test_eof; 
	_test_eof125: cs = 125; goto _test_eof; 
	_test_eof126: cs = 126; goto _test_eof; 
	_test_eof127: cs = 127; goto _test_eof; 
	_test_eof128: cs = 128; goto _test_eof; 
	_test_eof129: cs = 129; goto _test_eof; 
	_test_eof130: cs = 130; goto _test_eof; 
	_test_eof131: cs = 131; goto _test_eof; 
	_test_eof132: cs = 132; goto _test_eof; 
	_test_eof133: cs = 133; goto _test_eof; 
	_test_eof134: cs = 134; goto _test_eof; 
	_test_eof135: cs = 135; goto _test_eof; 
	_test_eof136: cs = 136; goto _test_eof; 
	_test_eof137: cs = 137; goto _test_eof; 
	_test_eof138: cs = 138; goto _test_eof; 
	_test_eof139: cs = 139; goto _test_eof; 
	_test_eof140: cs = 140; goto _test_eof; 
	_test_eof141: cs = 141; goto _test_eof; 
	_test_eof142: cs = 142; goto _test_eof; 
	_test_eof143: cs = 143; goto _test_eof; 
	_test_eof144: cs = 144; goto _test_eof; 
	_test_eof145: cs = 145; goto _test_eof; 
	_test_eof146: cs = 146; goto _test_eof; 
	_test_eof147: cs = 147; goto _test_eof; 
	_test_eof148: cs = 148; goto _test_eof; 
	_test_eof149: cs = 149; goto _test_eof; 
	_test_eof150: cs = 150; goto _test_eof; 
	_test_eof151: cs = 151; goto _test_eof; 
	_test_eof152: cs = 152; goto _test_eof; 
	_test_eof153: cs = 153; goto _test_eof; 
	_test_eof154: cs = 154; goto _test_eof; 
	_test_eof155: cs = 155; goto _test_eof; 
	_test_eof156: cs = 156; goto _test_eof; 
	_test_eof157: cs = 157; goto _test_eof; 
	_test_eof158: cs = 158; goto _test_eof; 
	_test_eof159: cs = 159; goto _test_eof; 
	_test_eof160: cs = 160; goto _test_eof; 
	_test_eof161: cs = 161; goto _test_eof; 
	_test_eof162: cs = 162; goto _test_eof; 
	_test_eof163: cs = 163; goto _test_eof; 
	_test_eof164: cs = 164; goto _test_eof; 
	_test_eof165: cs = 165; goto _test_eof; 
	_test_eof166: cs = 166; goto _test_eof; 
	_test_eof167: cs = 167; goto _test_eof; 
	_test_eof168: cs = 168; goto _test_eof; 
	_test_eof169: cs = 169; goto _test_eof; 
	_test_eof170: cs = 170; goto _test_eof; 
	_test_eof171: cs = 171; goto _test_eof; 
	_test_eof172: cs = 172; goto _test_eof; 
	_test_eof173: cs = 173; goto _test_eof; 
	_test_eof174: cs = 174; goto _test_eof; 
	_test_eof175: cs = 175; goto _test_eof; 
	_test_eof176: cs = 176; goto _test_eof; 
	_test_eof177: cs = 177; goto _test_eof; 
	_test_eof178: cs = 178; goto _test_eof; 
	_test_eof179: cs = 179; goto _test_eof; 
	_test_eof180: cs = 180; goto _test_eof; 
	_test_eof181: cs = 181; goto _test_eof; 
	_test_eof182: cs = 182; goto _test_eof; 
	_test_eof183: cs = 183; goto _test_eof; 
	_test_eof184: cs = 184; goto _test_eof; 
	_test_eof185: cs = 185; goto _test_eof; 
	_test_eof186: cs = 186; goto _test_eof; 
	_test_eof187: cs = 187; goto _test_eof; 
	_test_eof188: cs = 188; goto _test_eof; 
	_test_eof189: cs = 189; goto _test_eof; 
	_test_eof190: cs = 190; goto _test_eof; 
	_test_eof191: cs = 191; goto _test_eof; 
	_test_eof192: cs = 192; goto _test_eof; 
	_test_eof193: cs = 193; goto _test_eof; 
	_test_eof194: cs = 194; goto _test_eof; 
	_test_eof195: cs = 195; goto _test_eof; 
	_test_eof196: cs = 196; goto _test_eof; 
	_test_eof197: cs = 197; goto _test_eof; 
	_test_eof198: cs = 198; goto _test_eof; 
	_test_eof199: cs = 199; goto _test_eof; 
	_test_eof200: cs = 200; goto _test_eof; 
	_test_eof201: cs = 201; goto _test_eof; 
	_test_eof202: cs = 202; goto _test_eof; 
	_test_eof203: cs = 203; goto _test_eof; 
	_test_eof204: cs = 204; goto _test_eof; 
	_test_eof205: cs = 205; goto _test_eof; 
	_test_eof206: cs = 206; goto _test_eof; 
	_test_eof207: cs = 207; goto _test_eof; 
	_test_eof208: cs = 208; goto _test_eof; 
	_test_eof209: cs = 209; goto _test_eof; 
	_test_eof210: cs = 210; goto _test_eof; 
	_test_eof211: cs = 211; goto _test_eof; 
	_test_eof212: cs = 212; goto _test_eof; 
	_test_eof213: cs = 213; goto _test_eof; 
	_test_eof214: cs = 214; goto _test_eof; 
	_test_eof215: cs = 215; goto _test_eof; 
	_test_eof216: cs = 216; goto _test_eof; 
	_test_eof217: cs = 217; goto _test_eof; 
	_test_eof218: cs = 218; goto _test_eof; 
	_test_eof219: cs = 219; goto _test_eof; 
	_test_eof220: cs = 220; goto _test_eof; 
	_test_eof221: cs = 221; goto _test_eof; 
	_test_eof222: cs = 222; goto _test_eof; 
	_test_eof223: cs = 223; goto _test_eof; 
	_test_eof224: cs = 224; goto _test_eof; 
	_test_eof225: cs = 225; goto _test_eof; 
	_test_eof226: cs = 226; goto _test_eof; 
	_test_eof227: cs = 227; goto _test_eof; 
	_test_eof228: cs = 228; goto _test_eof; 
	_test_eof229: cs = 229; goto _test_eof; 
	_test_eof230: cs = 230; goto _test_eof; 
	_test_eof231: cs = 231; goto _test_eof; 
	_test_eof232: cs = 232; goto _test_eof; 
	_test_eof233: cs = 233; goto _test_eof; 
	_test_eof234: cs = 234; goto _test_eof; 
	_test_eof235: cs = 235; goto _test_eof; 
	_test_eof236: cs = 236; goto _test_eof; 
	_test_eof237: cs = 237; goto _test_eof; 
	_test_eof238: cs = 238; goto _test_eof; 
	_test_eof239: cs = 239; goto _test_eof; 
	_test_eof240: cs = 240; goto _test_eof; 
	_test_eof241: cs = 241; goto _test_eof; 
	_test_eof242: cs = 242; goto _test_eof; 
	_test_eof243: cs = 243; goto _test_eof; 
	_test_eof244: cs = 244; goto _test_eof; 
	_test_eof245: cs = 245; goto _test_eof; 
	_test_eof246: cs = 246; goto _test_eof; 
	_test_eof247: cs = 247; goto _test_eof; 
	_test_eof248: cs = 248; goto _test_eof; 
	_test_eof249: cs = 249; goto _test_eof; 
	_test_eof250: cs = 250; goto _test_eof; 
	_test_eof251: cs = 251; goto _test_eof; 
	_test_eof252: cs = 252; goto _test_eof; 
	_test_eof253: cs = 253; goto _test_eof; 
	_test_eof254: cs = 254; goto _test_eof; 
	_test_eof255: cs = 255; goto _test_eof; 
	_test_eof256: cs = 256; goto _test_eof; 
	_test_eof257: cs = 257; goto _test_eof; 
	_test_eof258: cs = 258; goto _test_eof; 
	_test_eof259: cs = 259; goto _test_eof; 
	_test_eof260: cs = 260; goto _test_eof; 
	_test_eof261: cs = 261; goto _test_eof; 
	_test_eof262: cs = 262; goto _test_eof; 
	_test_eof263: cs = 263; goto _test_eof; 
	_test_eof264: cs = 264; goto _test_eof; 
	_test_eof265: cs = 265; goto _test_eof; 
	_test_eof266: cs = 266; goto _test_eof; 
	_test_eof267: cs = 267; goto _test_eof; 
	_test_eof268: cs = 268; goto _test_eof; 
	_test_eof269: cs = 269; goto _test_eof; 
	_test_eof270: cs = 270; goto _test_eof; 
	_test_eof271: cs = 271; goto _test_eof; 
	_test_eof272: cs = 272; goto _test_eof; 
	_test_eof273: cs = 273; goto _test_eof; 
	_test_eof274: cs = 274; goto _test_eof; 
	_test_eof275: cs = 275; goto _test_eof; 
	_test_eof276: cs = 276; goto _test_eof; 
	_test_eof277: cs = 277; goto _test_eof; 
	_test_eof278: cs = 278; goto _test_eof; 
	_test_eof279: cs = 279; goto _test_eof; 
	_test_eof280: cs = 280; goto _test_eof; 
	_test_eof281: cs = 281; goto _test_eof; 
	_test_eof282: cs = 282; goto _test_eof; 
	_test_eof283: cs = 283; goto _test_eof; 
	_test_eof284: cs = 284; goto _test_eof; 
	_test_eof285: cs = 285; goto _test_eof; 
	_test_eof286: cs = 286; goto _test_eof; 
	_test_eof287: cs = 287; goto _test_eof; 
	_test_eof288: cs = 288; goto _test_eof; 
	_test_eof289: cs = 289; goto _test_eof; 
	_test_eof290: cs = 290; goto _test_eof; 
	_test_eof291: cs = 291; goto _test_eof; 
	_test_eof292: cs = 292; goto _test_eof; 
	_test_eof293: cs = 293; goto _test_eof; 
	_test_eof294: cs = 294; goto _test_eof; 
	_test_eof295: cs = 295; goto _test_eof; 
	_test_eof296: cs = 296; goto _test_eof; 
	_test_eof297: cs = 297; goto _test_eof; 
	_test_eof298: cs = 298; goto _test_eof; 
	_test_eof299: cs = 299; goto _test_eof; 
	_test_eof300: cs = 300; goto _test_eof; 
	_test_eof301: cs = 301; goto _test_eof; 
	_test_eof302: cs = 302; goto _test_eof; 
	_test_eof303: cs = 303; goto _test_eof; 
	_test_eof304: cs = 304; goto _test_eof; 
	_test_eof305: cs = 305; goto _test_eof; 
	_test_eof306: cs = 306; goto _test_eof; 
	_test_eof307: cs = 307; goto _test_eof; 
	_test_eof308: cs = 308; goto _test_eof; 
	_test_eof309: cs = 309; goto _test_eof; 
	_test_eof310: cs = 310; goto _test_eof; 
	_test_eof311: cs = 311; goto _test_eof; 
	_test_eof312: cs = 312; goto _test_eof; 
	_test_eof313: cs = 313; goto _test_eof; 
	_test_eof314: cs = 314; goto _test_eof; 
	_test_eof315: cs = 315; goto _test_eof; 
	_test_eof316: cs = 316; goto _test_eof; 
	_test_eof317: cs = 317; goto _test_eof; 
	_test_eof318: cs = 318; goto _test_eof; 
	_test_eof319: cs = 319; goto _test_eof; 
	_test_eof320: cs = 320; goto _test_eof; 
	_test_eof321: cs = 321; goto _test_eof; 
	_test_eof322: cs = 322; goto _test_eof; 
	_test_eof323: cs = 323; goto _test_eof; 
	_test_eof324: cs = 324; goto _test_eof; 
	_test_eof325: cs = 325; goto _test_eof; 
	_test_eof326: cs = 326; goto _test_eof; 
	_test_eof327: cs = 327; goto _test_eof; 
	_test_eof328: cs = 328; goto _test_eof; 
	_test_eof329: cs = 329; goto _test_eof; 
	_test_eof330: cs = 330; goto _test_eof; 
	_test_eof331: cs = 331; goto _test_eof; 
	_test_eof332: cs = 332; goto _test_eof; 
	_test_eof333: cs = 333; goto _test_eof; 
	_test_eof334: cs = 334; goto _test_eof; 
	_test_eof335: cs = 335; goto _test_eof; 
	_test_eof336: cs = 336; goto _test_eof; 
	_test_eof337: cs = 337; goto _test_eof; 
	_test_eof338: cs = 338; goto _test_eof; 
	_test_eof339: cs = 339; goto _test_eof; 
	_test_eof340: cs = 340; goto _test_eof; 
	_test_eof341: cs = 341; goto _test_eof; 
	_test_eof342: cs = 342; goto _test_eof; 
	_test_eof343: cs = 343; goto _test_eof; 
	_test_eof344: cs = 344; goto _test_eof; 
	_test_eof345: cs = 345; goto _test_eof; 
	_test_eof346: cs = 346; goto _test_eof; 
	_test_eof347: cs = 347; goto _test_eof; 
	_test_eof348: cs = 348; goto _test_eof; 
	_test_eof349: cs = 349; goto _test_eof; 
	_test_eof350: cs = 350; goto _test_eof; 
	_test_eof351: cs = 351; goto _test_eof; 
	_test_eof352: cs = 352; goto _test_eof; 
	_test_eof353: cs = 353; goto _test_eof; 
	_test_eof354: cs = 354; goto _test_eof; 
	_test_eof355: cs = 355; goto _test_eof; 
	_test_eof356: cs = 356; goto _test_eof; 
	_test_eof357: cs = 357; goto _test_eof; 
	_test_eof358: cs = 358; goto _test_eof; 
	_test_eof359: cs = 359; goto _test_eof; 
	_test_eof360: cs = 360; goto _test_eof; 
	_test_eof361: cs = 361; goto _test_eof; 
	_test_eof362: cs = 362; goto _test_eof; 
	_test_eof363: cs = 363; goto _test_eof; 
	_test_eof364: cs = 364; goto _test_eof; 
	_test_eof365: cs = 365; goto _test_eof; 
	_test_eof366: cs = 366; goto _test_eof; 
	_test_eof367: cs = 367; goto _test_eof; 
	_test_eof368: cs = 368; goto _test_eof; 
	_test_eof369: cs = 369; goto _test_eof; 
	_test_eof370: cs = 370; goto _test_eof; 
	_test_eof371: cs = 371; goto _test_eof; 
	_test_eof372: cs = 372; goto _test_eof; 
	_test_eof373: cs = 373; goto _test_eof; 
	_test_eof374: cs = 374; goto _test_eof; 
	_test_eof375: cs = 375; goto _test_eof; 
	_test_eof376: cs = 376; goto _test_eof; 
	_test_eof377: cs = 377; goto _test_eof; 
	_test_eof378: cs = 378; goto _test_eof; 
	_test_eof379: cs = 379; goto _test_eof; 
	_test_eof380: cs = 380; goto _test_eof; 
	_test_eof381: cs = 381; goto _test_eof; 
	_test_eof382: cs = 382; goto _test_eof; 
	_test_eof383: cs = 383; goto _test_eof; 
	_test_eof384: cs = 384; goto _test_eof; 
	_test_eof385: cs = 385; goto _test_eof; 
	_test_eof386: cs = 386; goto _test_eof; 
	_test_eof387: cs = 387; goto _test_eof; 
	_test_eof388: cs = 388; goto _test_eof; 
	_test_eof389: cs = 389; goto _test_eof; 
	_test_eof390: cs = 390; goto _test_eof; 
	_test_eof391: cs = 391; goto _test_eof; 
	_test_eof392: cs = 392; goto _test_eof; 
	_test_eof393: cs = 393; goto _test_eof; 
	_test_eof394: cs = 394; goto _test_eof; 
	_test_eof395: cs = 395; goto _test_eof; 
	_test_eof396: cs = 396; goto _test_eof; 
	_test_eof397: cs = 397; goto _test_eof; 
	_test_eof398: cs = 398; goto _test_eof; 
	_test_eof399: cs = 399; goto _test_eof; 
	_test_eof400: cs = 400; goto _test_eof; 
	_test_eof401: cs = 401; goto _test_eof; 
	_test_eof402: cs = 402; goto _test_eof; 
	_test_eof403: cs = 403; goto _test_eof; 
	_test_eof404: cs = 404; goto _test_eof; 
	_test_eof405: cs = 405; goto _test_eof; 
	_test_eof406: cs = 406; goto _test_eof; 
	_test_eof407: cs = 407; goto _test_eof; 
	_test_eof408: cs = 408; goto _test_eof; 
	_test_eof409: cs = 409; goto _test_eof; 
	_test_eof410: cs = 410; goto _test_eof; 
	_test_eof411: cs = 411; goto _test_eof; 
	_test_eof412: cs = 412; goto _test_eof; 
	_test_eof413: cs = 413; goto _test_eof; 
	_test_eof414: cs = 414; goto _test_eof; 
	_test_eof415: cs = 415; goto _test_eof; 
	_test_eof416: cs = 416; goto _test_eof; 
	_test_eof417: cs = 417; goto _test_eof; 
	_test_eof418: cs = 418; goto _test_eof; 
	_test_eof419: cs = 419; goto _test_eof; 
	_test_eof420: cs = 420; goto _test_eof; 
	_test_eof421: cs = 421; goto _test_eof; 
	_test_eof422: cs = 422; goto _test_eof; 
	_test_eof423: cs = 423; goto _test_eof; 
	_test_eof424: cs = 424; goto _test_eof; 
	_test_eof425: cs = 425; goto _test_eof; 
	_test_eof426: cs = 426; goto _test_eof; 
	_test_eof427: cs = 427; goto _test_eof; 
	_test_eof428: cs = 428; goto _test_eof; 
	_test_eof429: cs = 429; goto _test_eof; 
	_test_eof430: cs = 430; goto _test_eof; 
	_test_eof431: cs = 431; goto _test_eof; 
	_test_eof432: cs = 432; goto _test_eof; 
	_test_eof433: cs = 433; goto _test_eof; 
	_test_eof434: cs = 434; goto _test_eof; 
	_test_eof435: cs = 435; goto _test_eof; 
	_test_eof436: cs = 436; goto _test_eof; 
	_test_eof437: cs = 437; goto _test_eof; 
	_test_eof438: cs = 438; goto _test_eof; 
	_test_eof439: cs = 439; goto _test_eof; 
	_test_eof440: cs = 440; goto _test_eof; 
	_test_eof441: cs = 441; goto _test_eof; 
	_test_eof442: cs = 442; goto _test_eof; 
	_test_eof443: cs = 443; goto _test_eof; 
	_test_eof444: cs = 444; goto _test_eof; 
	_test_eof445: cs = 445; goto _test_eof; 
	_test_eof446: cs = 446; goto _test_eof; 
	_test_eof447: cs = 447; goto _test_eof; 
	_test_eof448: cs = 448; goto _test_eof; 
	_test_eof449: cs = 449; goto _test_eof; 
	_test_eof450: cs = 450; goto _test_eof; 
	_test_eof451: cs = 451; goto _test_eof; 
	_test_eof452: cs = 452; goto _test_eof; 
	_test_eof453: cs = 453; goto _test_eof; 
	_test_eof454: cs = 454; goto _test_eof; 
	_test_eof455: cs = 455; goto _test_eof; 
	_test_eof456: cs = 456; goto _test_eof; 
	_test_eof457: cs = 457; goto _test_eof; 
	_test_eof458: cs = 458; goto _test_eof; 
	_test_eof459: cs = 459; goto _test_eof; 
	_test_eof460: cs = 460; goto _test_eof; 
	_test_eof461: cs = 461; goto _test_eof; 
	_test_eof462: cs = 462; goto _test_eof; 
	_test_eof463: cs = 463; goto _test_eof; 
	_test_eof464: cs = 464; goto _test_eof; 
	_test_eof465: cs = 465; goto _test_eof; 
	_test_eof466: cs = 466; goto _test_eof; 
	_test_eof467: cs = 467; goto _test_eof; 
	_test_eof468: cs = 468; goto _test_eof; 
	_test_eof469: cs = 469; goto _test_eof; 
	_test_eof470: cs = 470; goto _test_eof; 
	_test_eof471: cs = 471; goto _test_eof; 
	_test_eof472: cs = 472; goto _test_eof; 
	_test_eof473: cs = 473; goto _test_eof; 
	_test_eof474: cs = 474; goto _test_eof; 
	_test_eof475: cs = 475; goto _test_eof; 
	_test_eof476: cs = 476; goto _test_eof; 
	_test_eof477: cs = 477; goto _test_eof; 
	_test_eof478: cs = 478; goto _test_eof; 
	_test_eof479: cs = 479; goto _test_eof; 
	_test_eof480: cs = 480; goto _test_eof; 
	_test_eof481: cs = 481; goto _test_eof; 
	_test_eof482: cs = 482; goto _test_eof; 
	_test_eof483: cs = 483; goto _test_eof; 
	_test_eof484: cs = 484; goto _test_eof; 
	_test_eof485: cs = 485; goto _test_eof; 
	_test_eof486: cs = 486; goto _test_eof; 
	_test_eof487: cs = 487; goto _test_eof; 
	_test_eof488: cs = 488; goto _test_eof; 
	_test_eof489: cs = 489; goto _test_eof; 
	_test_eof490: cs = 490; goto _test_eof; 
	_test_eof491: cs = 491; goto _test_eof; 
	_test_eof492: cs = 492; goto _test_eof; 
	_test_eof493: cs = 493; goto _test_eof; 
	_test_eof494: cs = 494; goto _test_eof; 
	_test_eof495: cs = 495; goto _test_eof; 
	_test_eof496: cs = 496; goto _test_eof; 
	_test_eof497: cs = 497; goto _test_eof; 
	_test_eof498: cs = 498; goto _test_eof; 
	_test_eof499: cs = 499; goto _test_eof; 
	_test_eof500: cs = 500; goto _test_eof; 
	_test_eof501: cs = 501; goto _test_eof; 
	_test_eof502: cs = 502; goto _test_eof; 
	_test_eof503: cs = 503; goto _test_eof; 
	_test_eof504: cs = 504; goto _test_eof; 
	_test_eof505: cs = 505; goto _test_eof; 
	_test_eof506: cs = 506; goto _test_eof; 
	_test_eof507: cs = 507; goto _test_eof; 
	_test_eof508: cs = 508; goto _test_eof; 
	_test_eof509: cs = 509; goto _test_eof; 
	_test_eof510: cs = 510; goto _test_eof; 
	_test_eof511: cs = 511; goto _test_eof; 
	_test_eof512: cs = 512; goto _test_eof; 
	_test_eof513: cs = 513; goto _test_eof; 
	_test_eof514: cs = 514; goto _test_eof; 
	_test_eof515: cs = 515; goto _test_eof; 
	_test_eof516: cs = 516; goto _test_eof; 
	_test_eof517: cs = 517; goto _test_eof; 
	_test_eof518: cs = 518; goto _test_eof; 
	_test_eof519: cs = 519; goto _test_eof; 
	_test_eof520: cs = 520; goto _test_eof; 
	_test_eof521: cs = 521; goto _test_eof; 
	_test_eof522: cs = 522; goto _test_eof; 
	_test_eof523: cs = 523; goto _test_eof; 
	_test_eof524: cs = 524; goto _test_eof; 
	_test_eof525: cs = 525; goto _test_eof; 
	_test_eof526: cs = 526; goto _test_eof; 
	_test_eof527: cs = 527; goto _test_eof; 
	_test_eof528: cs = 528; goto _test_eof; 
	_test_eof529: cs = 529; goto _test_eof; 
	_test_eof530: cs = 530; goto _test_eof; 
	_test_eof531: cs = 531; goto _test_eof; 
	_test_eof532: cs = 532; goto _test_eof; 
	_test_eof533: cs = 533; goto _test_eof; 
	_test_eof534: cs = 534; goto _test_eof; 
	_test_eof535: cs = 535; goto _test_eof; 
	_test_eof536: cs = 536; goto _test_eof; 
	_test_eof537: cs = 537; goto _test_eof; 
	_test_eof538: cs = 538; goto _test_eof; 
	_test_eof539: cs = 539; goto _test_eof; 
	_test_eof540: cs = 540; goto _test_eof; 
	_test_eof541: cs = 541; goto _test_eof; 
	_test_eof542: cs = 542; goto _test_eof; 
	_test_eof543: cs = 543; goto _test_eof; 
	_test_eof544: cs = 544; goto _test_eof; 
	_test_eof545: cs = 545; goto _test_eof; 
	_test_eof546: cs = 546; goto _test_eof; 
	_test_eof547: cs = 547; goto _test_eof; 
	_test_eof548: cs = 548; goto _test_eof; 
	_test_eof549: cs = 549; goto _test_eof; 
	_test_eof550: cs = 550; goto _test_eof; 
	_test_eof551: cs = 551; goto _test_eof; 
	_test_eof552: cs = 552; goto _test_eof; 
	_test_eof553: cs = 553; goto _test_eof; 
	_test_eof554: cs = 554; goto _test_eof; 
	_test_eof555: cs = 555; goto _test_eof; 
	_test_eof556: cs = 556; goto _test_eof; 
	_test_eof557: cs = 557; goto _test_eof; 
	_test_eof558: cs = 558; goto _test_eof; 
	_test_eof559: cs = 559; goto _test_eof; 
	_test_eof560: cs = 560; goto _test_eof; 
	_test_eof561: cs = 561; goto _test_eof; 
	_test_eof562: cs = 562; goto _test_eof; 
	_test_eof563: cs = 563; goto _test_eof; 
	_test_eof564: cs = 564; goto _test_eof; 
	_test_eof565: cs = 565; goto _test_eof; 
	_test_eof566: cs = 566; goto _test_eof; 

	_test_eof: {}
	if ( p == eof )
	{
	switch ( cs ) {
	case 568: 
#line 552 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	break;
#line 24546 "./zscan_rfc1035.c"
	}
	}

	_out: {}
	}

#line 727 "./zscan_rfc1035.rl"


        if(cs == zone_error) {
            parse_error_noargs("General parse error");
        }
        else if(eof && cs < zone_first_final) {
            if(eof > buf && *(eof - 1) != '\n')
                parse_error_noargs("Trailing incomplete or unparseable record at end of file (missing newline at end of file?)");
            else
                parse_error_noargs("Trailing incomplete or unparseable record at end of file");
        }
    }
}

// This is broken out into a separate function (called via
//   function pointer to eliminate the possibility of
//   inlining on non-gcc compilers, I hope) to avoid issues with
//   setjmp and all of the local auto variables in zscan_rfc1035() below.
typedef bool (*sij_func_t)(zscan_t*,char*,const unsigned,const int);
F_NONNULL F_NOINLINE
static bool _scan_isolate_jmp(zscan_t* z, char* buf, const unsigned bufsize, const int fd) {
    dmn_assert(z); dmn_assert(buf); dmn_assert(fd >= 0);

    volatile bool failed = true;

    if(!sigsetjmp(z->jbuf, 0)) {
        scanner(z, buf, bufsize, fd);
        failed = false;
    }
    else {
        failed = true;
    }

    return failed;
}

bool zscan_rfc1035(zone_t* zone, const char* fn) {
    dmn_assert(zone);
    dmn_assert(zone->dname);
    dmn_assert(fn);
    log_debug("rfc1035: Scanning zone '%s'", logf_dname(zone->dname));

    const int fd = open(fn, O_RDONLY);
    if(fd < 0) {
        log_err("rfc1035: Cannot open file '%s' for reading: %s", logf_pathname(fn), logf_errno());
        return true;
    }

    unsigned bufsize = MAX_BUFSIZE;
    {
        struct stat fdstat;
        if(!fstat(fd, &fdstat)) {
#ifdef HAVE_POSIX_FADVISE
            (void)posix_fadvise(fd, 0, fdstat.st_size, POSIX_FADV_SEQUENTIAL);
#endif
            if(fdstat.st_size < (int)bufsize)
                bufsize = fdstat.st_size;
        }
        else {
            log_warn("rfc1035: fstat(%s) failed for advice, not critical...", logf_pathname(fn));
        }
    }

    zscan_t* z = calloc(1, sizeof(zscan_t));
    z->lcount = 1;
    z->def_ttl = gconfig.zones_default_ttl;
    z->zone = zone;
    dname_copy(z->origin, zone->dname);
    z->lhs_dname[0] = 1; // set lhs to relative origin initially

    char* buf = malloc(bufsize + 1);
    buf[bufsize] = 0;

    sij_func_t sij = &_scan_isolate_jmp;
    bool failed = sij(z, buf, bufsize, fd);

    if(close(fd)) {
        log_err("rfc1035: Cannot close file '%s': %s", logf_pathname(fn), logf_errno());
        failed = true;
    }

    free(buf);

    if(z->texts) {
        for(unsigned i = 0; i < z->num_texts; i++)
            if(z->texts[i])
                free(z->texts[i]);
        free(z->texts);
    }
    if(z->rfc3597_data)
        free(z->rfc3597_data);
    free(z);

    return failed;
}

