/*
 *      fhist - file history and comparison tools
 *      Copyright (C) 2000, 2002, 2008 Peter Miller
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 3 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program. If not, see
 *      <http://www.gnu.org/licenses/>.
 */

#include <ac/unistd.h>

#include <error.h>
#include <fcheck.h>
#include <input/file.h>
#include <input/private.h>
#include <input/stdin.h>
#include <mem.h>
#include <str.h>
#include <error_intl.h>

typedef struct input_file_ty input_file_ty;
struct input_file_ty
{
        input_ty        inherited;
        FILE            *fp;
        string_ty       *fn;
        int             unlink_on_close;
        long            pos;
};


static void
destruct(input_ty *p)
{
        input_file_ty   *this;

        this = (input_file_ty *)p;
        fclose_and_check(this->fp, this->fn->str_text);
        if (this->unlink_on_close)
                unlink(this->fn->str_text);
        str_free(this->fn);
        this->fp = 0;
        this->fn = 0;
}


static long
iread(input_ty *p, void *data, long len)
{
        input_file_ty   *this;
        long            result;

        if (len < 0)
                return 0;
        this = (input_file_ty *)p;
        result = fread(data, (size_t)1, (size_t)len, this->fp);
        if (result <= 0 && ferror(this->fp))
        {
                sub_context_ty  *scp;

                scp = sub_context_new();
                sub_errno_set(scp);
                sub_var_set_string(scp, "File_Name", this->fn);
                fatal_intl(scp, i18n("read \"$filename\": $errno"));
                /* NOTREACHED */
        }
        this->pos += result;
        return result;
}


static int
get(input_ty *p)
{
        input_file_ty   *this;
        int             c;

        this = (input_file_ty *)p;
        c = getc(this->fp);
        if (c == EOF && ferror(this->fp))
        {
                sub_context_ty  *scp;

                scp = sub_context_new();
                sub_errno_set(scp);
                sub_var_set_string(scp, "File_Name", this->fn);
                fatal_intl(scp, i18n("read \"$filename\": $errno"));
                /* NOTREACHED */
        }
        if (c != EOF)
                this->pos++;
        return c;
}


static long
itell(input_ty *p)
{
        input_file_ty   *this;

        this = (input_file_ty *)p;
        return this->pos;
}


static const char *
name(input_ty *p)
{
        input_file_ty   *this;

        this = (input_file_ty *)p;
        return this->fn->str_text;
}


static long
length(input_ty *p)
{
        input_file_ty   *this;
        struct stat     st;
        int             err;

        this = (input_file_ty *)p;
#ifdef S_IFLNK
        err = lstat(this->fn->str_text, &st);
#else
        err = stat(this->fn->str_text, &st);
#endif
        if (err < 0)
        {
                sub_context_ty  *scp;

                scp = sub_context_new();
                sub_errno_set(scp);
                sub_var_set_string(scp, "File_Name", this->fn);
                fatal_intl(scp, i18n("stat \"$filename\": $errno"));
                /* NOTREACHED */
        }
        return st.st_size;
}


static input_vtbl_ty vtbl =
{
        sizeof(input_file_ty),
        destruct,
        iread,
        get,
        itell,
        name,
        length,
};


input_ty *
input_file_open(const char *fn)
{
        input_ty        *result;
        input_file_ty   *this;

        if (!fn || !*fn)
                return input_stdin();
        result = input_new(&vtbl);
        this = (input_file_ty *)result;
        this->fp = fopen_and_check(fn, "rb");
        this->fn = str_from_c(fn);
        this->unlink_on_close = 0;
        this->pos = 0;
        return result;
}


void
input_file_unlink_on_close(input_ty *fp)
{
        input_file_ty   *this;

        if (fp->vptr != &vtbl)
                return;
        this = (input_file_ty *)fp;
        this->unlink_on_close = 1;
}
