# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Unit tests for the signing no-operation task."""

from tempfile import TemporaryDirectory
from unittest import mock

from django.test import TestCase

from debusine.artifacts.models import WorkRequestResults
from debusine.signing.tasks import SigningNoop
from debusine.tasks import TaskConfigError


class SigningNoopTests(TestCase):
    """Test the SigningNoop task."""

    def test_execute_result_success(self) -> None:
        """If result=SUCCESS, the task returns SUCCESS."""
        noop = SigningNoop({"result": WorkRequestResults.SUCCESS})
        noop._debug_log_files_directory = TemporaryDirectory(
            prefix="debusine-tests-"
        )
        self.addCleanup(noop._debug_log_files_directory.cleanup)
        with mock.patch.object(
            noop, "_upload_work_request_debug_logs", autospec=True
        ):
            self.assertEqual(noop.execute(), WorkRequestResults.SUCCESS)

    def test_execute_result_failure(self) -> None:
        """If result=FAILURE, the task returns FAILURE."""
        noop = SigningNoop({"result": WorkRequestResults.FAILURE})
        noop._debug_log_files_directory = TemporaryDirectory(
            prefix="debusine-tests-"
        )
        self.addCleanup(noop._debug_log_files_directory.cleanup)
        with mock.patch.object(
            noop, "_upload_work_request_debug_logs", autospec=True
        ):
            self.assertEqual(noop.execute(), WorkRequestResults.FAILURE)

    def test_no_additional_properties(self) -> None:
        """Assert no additional properties."""
        error_msg = "Extra inputs are not permitted"
        with self.assertRaisesRegex(TaskConfigError, error_msg):
            SigningNoop({"input": {}})

    def test_label(self) -> None:
        """Test get_label."""
        task = SigningNoop({})
        self.assertEqual(task.get_label(), "noop")
