
#include <iostream>
#include <string>
#include <signal.h>

#include <boost/program_options.hpp>
#include <boost/bind.hpp>

#include <rpcdriver/rpcdriver.h>

#include <dcpp/stdinc.h>
#include <dcpp/DCPlusPlus.h>
#include <dcpp/SettingsManager.h>
#include <dcpp/ShareManager.h>
#include <dcpp/QueueManager.h>

#include "SessionManager.h"
#include "TransferManager.h"
#include "ClientNotifier.h"

#include "commandhandlers.h"
#include "ClientNotifier.h"
#include "filelog.h"

namespace prog_opts = boost::program_options;

Log* logger = new FileLog();

void callBack(void* x, const std::string& a);

namespace {
	
/**
 * Start the rpc server and register the neccesary commands.
 * @param port The port that we should listen to.
 * @return Pointer to the created instance of the rpc server.
 */
	rpc::RpcDriverPtr initRpcServer( int port, std::string password )
{
	rpc::RpcServerDriver *fulpekare = new rpc::RpcServerDriver(port); 
	rpc::RpcDriverPtr driver( fulpekare );

	dcqt_backend::ClientNotifier::instance()->setRpcDriver( driver );
	
	rpc::RpcCommandHandlerPtr authenticateHandler( new dcqt_backend::AuthenticateCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
			driver.get(),
			_1, _2) ,password,driver) );
	driver->registerCommand(authenticateHandler);
	
	rpc::RpcCommandHandlerPtr createSessionHandler( new dcqt_backend::CreateSessionCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																											driver.get(),
																											_1, _2) ) );
	driver->registerCommand( createSessionHandler );
	
	rpc::RpcCommandHandlerPtr closeSessionHandler( new dcqt_backend::CloseSessionCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																										 driver.get(),
																										 _1, _2) ) );
	driver->registerCommand( closeSessionHandler );
	
	rpc::RpcCommandHandlerPtr sendChatHandler( new dcqt_backend::SendChatCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																								  driver.get(),
																								  _1, _2) ) );
	driver->registerCommand( sendChatHandler );
	
	rpc::RpcCommandHandlerPtr searchHandler( new dcqt_backend::SearchCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																							  driver.get(),
																							  _1, _2) ) );
	driver->registerCommand( searchHandler );
	
	rpc::RpcCommandHandlerPtr getSharedDirs( new dcqt_backend::GetSharedDirectoriesCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																											driver.get(),
																											_1, _2) ) );
	driver->registerCommand( getSharedDirs );
	
	rpc::RpcCommandHandlerPtr addShareDir( new dcqt_backend::AddShareCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																							  driver.get(),
																							  _1, _2) ) );
	driver->registerCommand( addShareDir );
	
	rpc::RpcCommandHandlerPtr removeSharedDir( new dcqt_backend::RemoveShareCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																									 driver.get(),
																									 _1, _2) ) );
	driver->registerCommand( removeSharedDir );
	
	rpc::RpcCommandHandlerPtr downloadFile( new dcqt_backend::DownloadFileCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																								   driver.get(),
																								   _1, _2) ) );
	driver->registerCommand( downloadFile );
	
	rpc::RpcCommandHandlerPtr getFileList( new dcqt_backend::GetUserFileListCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																									 driver.get(),
																									 _1, _2) ) );
	driver->registerCommand( getFileList );
	
	
	rpc::RpcCommandHandlerPtr getHubList( new dcqt_backend::GetHubListCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																							   driver.get(),
																							   _1, _2) ) );
	driver->registerCommand( getHubList );
	
	rpc::RpcCommandHandlerPtr testbool( new dcqt_backend::TestBoolHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																						driver.get(),
																						_1, _2) ) );
	driver->registerCommand( testbool );
	
	rpc::RpcCommandHandlerPtr getRunning( new dcqt_backend::RequestRunningSessionsCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																										   driver.get(),
																										   _1, _2) ) );
	driver->registerCommand( getRunning );
	
	rpc::RpcCommandHandlerPtr connectToUser( new dcqt_backend::ConnectToUserCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																									 driver.get(),
																									 _1, _2) ) );
	driver->registerCommand( connectToUser );
	
	rpc::RpcCommandHandlerPtr removeQueueItem( new dcqt_backend::RemoveQueueItemCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																										   driver.get(),
																										   _1, _2) ) );
	driver->registerCommand( removeQueueItem );
	
	rpc::RpcCommandHandlerPtr addFav( new dcqt_backend::AddFavoriteHubCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																										   driver.get(),
																										   _1, _2) ) );
	driver->registerCommand( addFav );
	
	rpc::RpcCommandHandlerPtr removeFav( new dcqt_backend::RemoveFavoriteHubCmdHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																									 driver.get(),
																									 _1, _2) ) );
	driver->registerCommand( removeFav );
	
	rpc::RpcCommandHandlerPtr favList( new dcqt_backend::GetFavHubsHandler( boost::bind( &rpc::RpcDriver::queueCommand,
																						 driver.get(),
																						 _1, _2) ) );
	driver->registerCommand( favList );
	
	rpc::RpcCommandHandlerPtr getSettings( new dcqt_backend::GetSettingsHandler( boost::bind(&rpc::RpcDriver::queueCommand,
																						driver.get(),_1,_2) ));
	driver->registerCommand( getSettings );

	rpc::RpcCommandHandlerPtr setSettings( new dcqt_backend::SetSettingsHandler( boost::bind(&rpc::RpcDriver::queueCommand,
																						driver.get(),_1,_2) ));
	driver->registerCommand( setSettings );

	rpc::RpcCommandHandlerPtr die( new dcqt_backend::DieCmdHandler( boost::bind(&rpc::RpcDriver::queueCommand,
																						driver.get(),_1,_2), driver ));
	driver->registerCommand( die );
	
	rpc::RpcCommandHandlerPtr removeSource( new dcqt_backend::RemoveSourceHandler( boost::bind(&rpc::RpcDriver::queueCommand,
																						driver.get(),_1,_2) ));
	driver->registerCommand( removeSource );
	
	fulpekare->startListening();

	return driver;
}


bool dcppInited = false;

/**
 * Initialze the dcpp thingies.
 */
void initDcpp()
{
	logger->info( boost::format("Init Dcpp") );
	startup(callBack, NULL);
	dcppInited = true;
    SettingsManager::getInstance()->load();
    TimerManager::getInstance()->start();
	// Set the port that the SearchManager should use
	SearchManager::getInstance()->setPort( SettingsManager::getInstance()->get( SettingsManager::UDP_PORT, true ) );
}


void initDcqtBackend()
{
	dcqt_backend::SessionManager::instance();
    dcqt_backend::TransferManager::newInstance();
}

}

// For dcpp to send messages
void callBack(void* x, const string& a) {
    cout << "dcpp: " << a << endl;
	logger->debug(boost::format("dcpp: %1%") % a,1);
}

rpc::RpcDriverPtr driver;


void quithandler(int sig) {
    driver->stopSender();
}


int main (int argc, char *argv[]) {
	
	prog_opts::options_description desc("Allowed options");
	desc.add_options()
		("help", 
		 "Display help message")
		("addr", 
		 prog_opts::value<std::string>()->default_value("localhost"),
		 "Address to connect to")
		("port", 
		 prog_opts::value<int>()->default_value(6161), 
		 "Specify port")
		 ("password",
		  prog_opts::value<std::string>()->default_value(""),
		  "Specify a password that the backend will require from connecting clients");
	
	prog_opts::variables_map vm;
	prog_opts::store(prog_opts::parse_command_line(argc, argv, desc), vm);
	
	if(vm.count("help")) {
		std::cout << desc << std::endl;
		return 1;
	}

  	//----------------------------------------------------------
    // Install signal handler
    struct sigaction act;
    act.sa_handler = quithandler;
    act.sa_flags = 0;
    sigaction(SIGINT,&act,0);
	
	try {
		driver = initRpcServer( vm["port"].as<int>(),vm["password"].as<std::string>() );
		
		initDcpp();
		//wtf?
		signal(SIGPIPE, SIG_IGN);
		initDcqtBackend();
		
		driver->waitForCompletion();
	}
	catch(...) {
	}

	if(dcppInited) {
		dcqt_backend::TransferManager::deleteInstance();
		shutdown();
	}

	logger->info(boost::format("Exiting normally"));

	
    return 0;
}
