/* 
$Id: courierpassd.c,v 1.1 2005/10/15 21:52:08 astjean Exp $

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

Copyright (c) Andrew St. Jean <andrew@arda.homeunix.net> 2002-2005

courierpassd.c

A change password server for Courier.

Based on poppassd by Pawel Krawczyk which is itself based
on an earlier version written by
John Norstad <j-norstad@nwu.edu>, Roy Smith <roy@nyu.edu>, 
and Daniel L. Leavitt <dll.mitre.org>.

See README for more information.
*/
 
 
/* These need to be quoted because they are only used as
 * parts of format strings for sscanf
 */
#define MAX_LEN_USERNAME	"64"	/* maximum length of username */
#define MAX_LEN_PASS		"128"	/* maximum length of password */

#define MAX_LEN_MODULE          129  /* max length for module name */
#define MAX_LEN_SERVICE		129  /* max length for service name */

#define BUFSIZE 1024

#include "courierpassd.h"

#define LOCK_TRIES 30

void
#ifdef HAVE_STDARG_H
WriteToClient(char *fmt, ...)
#else
WriteToClient(fmt, va_alist)
char *fmt;
va_dcl
#endif
{
	va_list ap;

	VA_START(ap, fmt);
	vfprintf(stdout, fmt, ap);
	fputs("\r\n", stdout);
	fflush(stdout);
	va_end(ap);
}

void
ReadFromClient (char *line)
{
	char *sp;

	memset(line, 0x00, BUFSIZE);
	fgets(line, BUFSIZE-1, stdin);
	if ((sp = strchr(line, '\n')) != NULL) *sp = '\0'; 
	if ((sp = strchr(line, '\r')) != NULL) *sp = '\0'; 
	
	/* convert initial keyword on line to lower case. */
	
	for (sp = line; isalpha(*sp); sp++) *sp = tolower(*sp);
	line[BUFSIZE-1] = '\0';
}

static int
callback_func(struct authinfo *a, void *dummy) {
	return(0);
}

static int
callback_pre(struct authinfo *a, void *min_uid) {

	struct passwd *pw;

	if (a->sysuserid != NULL){
		if (a->sysuserid < (uid_t *)min_uid) return(-1);
	} else if (a->sysusername) {
		pw = getpwnam(a->sysusername);
		if (!pw){
			logging(LOG_ERR, "getpwnam failed: %s", strerror(errno));	
			/* perror("getpwnam"); */
			return(-2);
		}

		if (pw->pw_uid < (uid_t)min_uid) return(-1);
	}

	return(0);
}

size_t
validate_str(const char *data){
	return(strspn(data, SAFE_CHARS));
}

void
end(int exit_code) {
	end_logging();
	exit(exit_code);
}

static const char *short_options = "s:hV";

static struct option long_options[] = {
	{ "service", required_argument, NULL, 's' },
	{ "help", no_argument, NULL, 'h' },
	{ "version", no_argument, NULL, 'V' },
	{ "stderr", no_argument, NULL, OPT_STDERR },
	{ NULL, 0, NULL, 0 }
};

const char *usage =
"Usage: " PACKAGE " [ OPTION [ OPTION ]]\n"
"\n"
"Changes passwords using the Courier authentication\n"
"library. Authentication tokens are delivered using the\n"
"poppassd protocol. This program is intended to be run\n"
"from a super-server such as tcpserver or xinetd.\n"
"\n"
"Options:\n"
"  -s, --service=SERVICE\tspecify the service name to use\n"
"\t\t\tdefault value is 'login'\n"
"  --stderr\t\tsend logging information to standard error\n"
"\t\t\tinstead of syslog\n"
"  -h, --help\t\tdisplay this help and exit\n"
"  -V, --version\t\tdisplay version information and exit\n";

int opt_stderr = 0;

int main (int argc, char **argv)
{
	char	line[BUFSIZE];
	char	user[BUFSIZE];
	char	oldpass[BUFSIZE];
	char	newpass[BUFSIZE];
	int	i, rc;
	size_t	n;
	int	success=0;
	char	service[MAX_LEN_SERVICE];
     
	*user = *oldpass = *newpass = *service = 0;

	/* set the default value for service */
	strlcpy(service, "login", 6);

	init_logging();

	/* Process the command line options. */
	opterr = 0;
	while (1) {
		int option_index = 0;
		int c = getopt_long(argc, argv, short_options, long_options, &option_index);

		if (c == -1) break;

		switch (c) {
		case 's':
			n = validate_str(optarg);
			if (optarg[n]){
				logging(LOG_ERR, "illegal character in service name: %c", optarg[n]);
				end(2);
			}
			n = strlcpy(service, optarg, sizeof(service));
			if (n >= sizeof(service)){
				logging(LOG_ERR, "service name too long (max %d characters)", MAX_LEN_SERVICE - 1);
				end(2);
			}
			break;
		case OPT_STDERR:
			opt_stderr = 1;
			break;
		case 'h':
			puts(usage);
			end(0);
		case 'V':
			puts(PACKAGE " " VERSION);
			end(0);
		case '?':
			puts("Invalid command line, see --help");
			end(2);
		}
	}

	WriteToClient("200 courierpassd v%s hello, who are you?", VERSION);
	ReadFromClient(line);
	if( strlen(line) > atoi(MAX_LEN_USERNAME) ) {
		WriteToClient ("500 Username too long (max %d).", atoi(MAX_LEN_USERNAME));
		end(1);
	}
	sscanf(line, "user %" MAX_LEN_USERNAME "s", user) ;
	if (strlen(user) == 0 )
	{
		WriteToClient("500 Username required.");
		end(1);
	}

	WriteToClient("200 Your password please.");
	ReadFromClient(line);
	if( strlen(line) > atoi(MAX_LEN_PASS) )
	{
		WriteToClient("500 Password length exceeded (max %d).",
			      atoi(MAX_LEN_PASS) );
		end(1);
	}
	sscanf (line, "pass %" MAX_LEN_PASS "s", oldpass);
	if(strlen(oldpass) == 0) 
	{
		WriteToClient("500 Password required.");
		end(1);
	}

	/* First we authenticate the user with her old password. */
	rc = auth_login(service, user, oldpass, callback_func, 0);

	if (rc < 0){
		WriteToClient("500 Old password is incorrect or user not found.");
		logging(LOG_ERR, "old password is incorrect or user not found for %s", user);
		/* pause to make brute force attacks harder */
		sleep(BAD_PASS_DELAY);
		end(1);
	} else if (rc > 0) {
		WriteToClient("500 Server error, password not changed.");
	        logging(LOG_ERR, "failed to change password for user %s", user);
		end(1);
	}

	WriteToClient("200 Your new password please.");
	ReadFromClient(line);
	if( strlen(line) > atoi(MAX_LEN_PASS) )
	{
		WriteToClient("500 Password length exceeded (max %d).",
				atoi(MAX_LEN_PASS) );
		end(1);
	}

	sscanf (line, "newpass %" MAX_LEN_PASS "s", newpass);
	if (strlen(newpass) == 0) {
		WriteToClient("500 New password required.");
		end(1);
	}

	/* Does this user have a restricted uid? */
	rc = auth_getuserinfo(service, user, callback_pre, (uid_t *)MIN_UID);

	if (rc != 0){
		WriteToClient ("500 Restricted or non-existant uid.");
		logging(LOG_ERR, "attempt to change password for a restricted or non-existant uid %s", user);
		end(1);
	}

	/* Change the user's password */
	rc = auth_passwd(service, user, oldpass, newpass);

	if (rc != 0){
		WriteToClient("500 Server error, password not changed.");
		logging(LOG_ERR, "failed to change password for user %s", user);
		/* pause to make brute force attacks harder */
		sleep(BAD_PASS_DELAY);
		end(1);
	}
		
	WriteToClient("200 Password changed, thank-you.");
	logging(LOG_ERR, "changed password for %s", user);

	ReadFromClient(line);
		
	if (strncmp(line, "quit", 4) != 0) {
		WriteToClient("500 Quit required.");
		end(1);
	}

	WriteToClient("200 Bye.");
	end(0);
	return(0);
}

