// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/** @const */ var TOTAL_RESULT_COUNT = 25;

/**
 * Test C++ fixture for downloads WebUI testing.
 * @constructor
 * @extends {testing.Test}
 */
function DownloadsUIBrowserTest() {}

/**
 * Base fixture for Downloads WebUI testing.
 * @extends {testing.Test}
 * @constructor
 */
function BaseDownloadsWebUITest() {}

BaseDownloadsWebUITest.prototype = {
  __proto__: testing.Test.prototype,

  /**
   * Browse to the downloads page & call our preLoad().
   */
  browsePreload: 'chrome://downloads/',

  /** @override */
  typedefCppFixture: 'DownloadsUIBrowserTest',

  /** @override */
  testGenPreamble: function() {
    GEN('  SetDeleteAllowed(true);');
  },

  /** @override */
  runAccessibilityChecks: true,

  /** @override */
  accessibilityIssuesAreErrors: true,

  /**
   * Sends TOTAL_RESULT_COUNT fake downloads to the page. This can't be called
   * in the preLoad, because it requires the global Download object to have
   * been created by the page.
   * @override
   */
  setUp: function() {
    testing.Test.prototype.setUp.call(this);

    this.createdDownloads = [];

    // The entries will begin at 1:00 AM on Sept 2, 2008, and will be spaced
    // two minutes apart.
    var timestamp = new Date(2008, 9, 2, 1, 0).getTime();
    for (var i = 0; i < TOTAL_RESULT_COUNT; ++i) {
      this.createDownload(i, timestamp);
      timestamp += 2 * 60 * 1000;  // Next visit is two minutes later.
    }
    downloads.Manager.updateAll(this.createdDownloads);
    expectEquals(downloads.Manager.size(), TOTAL_RESULT_COUNT);

    this.updateAccessibilityAuditConfig();
  },

  /**
   * Disables failing accessibility audits. This should be removed when all
   * audit issues have been resolved.
   */
  updateAccessibilityAuditConfig: function() {
    // Enable when failure is resolved.
    // AX_TEXT_01: http://crbug.com/559217
    this.accessibilityAuditConfig.ignoreSelectors(
        'controlsWithoutLabel',
        '#term');

    // Enable when failure is resolved.
    // AX_FOCUS_03: http://crbug.com/559219
    this.accessibilityAuditConfig.ignoreSelectors(
        'tabIndexGreaterThanZero',
        '#term');
  },

  /**
   * Creates a download object to be passed to the page, following the expected
   * backend format (see downloads_dom_handler.cc).
   * @param {number} id A unique ID for the download.
   * @param {number} timestamp The time the download purportedly started.
   * @return {!Object} A fake download object.
   */
  createDownload: function(id, timestamp) {
    this.createdDownloads.unshift({
      id: id,
      started: timestamp,
      otr: false,
      state: downloads.States.COMPLETE,
      retry: false,
      file_path: '/path/to/file',
      file_url: 'http://google.com/' + timestamp,
      file_name: 'download_' + timestamp,
      url: 'http://google.com/' + timestamp,
      file_externally_removed: false,
      danger_type: downloads.DangerType.NOT_DANGEROUS,
      last_reason_text: '',
      since_string: 'today',
      date_string: 'today',
      percent: 100,
      progress_status_text: 'done',
      received: 128,
    });
    return this.createdDownloads[0];
  },

  /**
   * Creates a dangerous download object. See downloads_dom_handler.cc.
   * @param {number} id The ID of the download.
   * @param {number} timestamp The time this download started.
   * @return {!Object} A fake, dangerous download object.
   */
  createDangerousDownload: function(id, timestamp) {
    this.createdDownloads.unshift({
      id: id,
      started: timestamp,
      otr: false,
      state: downloads.States.DANGEROUS,
      retry: false,
      file_path: '/oh/noes.jpg.exe',
      file_url: 'http://evil.com/cute/kittens' + timestamp,
      file_name: 'evil.' + timestamp + '.jar',
      file_url: 'http://evil.com/cute/kittens' + timestamp,
      file_externally_removed: false,
      danger_type: downloads.DangerType.DANGEROUS_FILE,
      last_reason_text: '',
      since_string: 'today',
      date_string: 'today',
      percent: 0,
      progress_status_text: '',
      received: 128,
    });
    return this.createdDownloads[0];
  },

  /**
   * Simulates getting no results from C++.
   */
  sendEmptyList: function() {
    downloads.Manager.updateAll([]);
    assertEquals(0, downloads.Manager.size());
  },

  /**
   * Check that |element| is showing and contains |text|.
   * @param {Element} element
   * @param {string} text
   */
  checkShowing: function(element, text) {
    expectFalse(element.hidden);
    expectNotEquals(-1, element.textContent.indexOf(text));
  },

  /**
   * Asserts the correctness of the state of the UI elements that delete the
   * download history.
   * @param {boolean} visible True if download deletion UI should be visible.
   */
  expectDeleteControlsVisible: function(visible) {
    // "Clear all" should only be showing when deletions are allowed.
    expectEquals(!visible, $('clear-all').hidden);

    // "Remove from list" links should only exist when deletions are allowed.
    var query = '#downloads-display .safe .remove';
    if (!visible)
      query += '[hidden]';
    expectEquals(TOTAL_RESULT_COUNT, document.querySelectorAll(query).length);
  },
};
