/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


// pml
#include <pml/Atom.h>
#include <pml/StructuralComponent.h>
#include <pml/Cell.h>



#include "PMManagerDC.h"
#include "AtomDCWidget.h"
#include "AtomDC.h"
#include "CellDC.h"
#include "StructuralComponentDC.h"



// qt stuff
#include <QVBoxLayout>
#include <QLabel>
#include <QTableWidget>
#include <QHeaderView>

#include <QMessageBox>

// ---------------------- Constructor ----------------------------
AtomDCWidget::AtomDCWidget(AtomDC *adc, QWidget* parent) : QWidget(parent) {
    // create the layout
    QVBoxLayout * widgetLayout = new QVBoxLayout(this);

    // the text label
    infoLabel = new QLabel(this);
    widgetLayout->addWidget(infoLabel);

    // and the list widget
    usedInTable = new QTableWidget(this);
    usedInTable->setSelectionBehavior(QAbstractItemView::SelectRows);

    // headers
    usedInTable->setColumnCount(4); // very very important!
    QStringList headerTitles;
    headerTitles << " " << "Name/Id" << "E/I" << "Children";
    usedInTable->setHorizontalHeaderLabels(headerTitles);
    usedInTable->horizontalHeader()->setStretchLastSection(false);
    usedInTable->horizontalHeader()->setResizeMode(0, QHeaderView::Stretch); // First column stretch to what is needed
    usedInTable->horizontalHeader()->setResizeMode(1, QHeaderView::Fixed); // Last column never resize
    usedInTable->horizontalHeader()->setVisible(true);

    widgetLayout->addWidget(usedInTable);

    // add the tool tip-top
    usedInTable->setToolTip("List of all cells and StructuralComponents<br>using this atom.<p>E/I = <b>E</b>xclusive or <b>I</b>nformative");
    usedInTable->setToolTip("List of all cells and StructuralComponents<br>using this atom.<p>E/I = <b>E</b>xclusive or <b>I</b>nformative");

    // click allow selection
    connect(usedInTable, SIGNAL(cellClicked(int, int)),
            this, SLOT(usedInTableClicked(int , int )));
}

#include "cell_20x20.xpm"
#include "structuralcomponent_20x20.xpm"
QPixmap * cellPixmap = NULL;
QPixmap * structuralComponentPixmap = NULL;

// ---------------------- updateProperties ----------------------------
void AtomDCWidget::updateProperties(AtomDC *adc) {
    myDC = adc;
    Atom *a = myDC->getAtom();

    // create the pixmap if non existant
    if (!cellPixmap)
        cellPixmap = new QPixmap(cell_20x20);

    if (!structuralComponentPixmap)
        structuralComponentPixmap = new QPixmap(structuralcomponent_20x20);

    // set the text
    infoLabel->setText("Atom #" + QString::number(a->getIndex()) + " used in:");

    // fill in the list
    usedInTable->clearContents();
    usedInTable->setRowCount(a->getNumberOfStructuralComponents());
    usedInTable->verticalHeader()->setVisible(false);
    usedInTable->setSortingEnabled(false); // temporarily for insertion!

    StructuralComponent *sc;
    for (unsigned int i = 0; i < a->getNumberOfStructuralComponents(); i++) {
        sc = a->getStructuralComponent(i);

        // add the icon
        if (sc->isInstanceOf("Cell"))
            usedInTable->setItem(i, 0, new QTableWidgetItem((*cellPixmap), ""));
        else
            usedInTable->setItem(i, 0, new QTableWidgetItem((*structuralComponentPixmap), ""));

        // name and E/I
        bool isExclusive;

        if (sc->isInstanceOf("Cell")) {
            usedInTable->setItem(i, 1, new QTableWidgetItem(QString::number(dynamic_cast<Cell *>(sc)->getIndex())));
            isExclusive = dynamic_cast<Cell *>(sc)->getStructuralComponent(0)->isExclusive();
        } else {
            usedInTable->setItem(i, 1, new QTableWidgetItem(sc->getName().c_str()));
            isExclusive = sc->isExclusive();
        }

        // add exclusive/inclusive info
        if (isExclusive)
            usedInTable->setItem(i, 2, new QTableWidgetItem("E"));
        else
            usedInTable->setItem(i, 2, new QTableWidgetItem("I"));

        // nr of children
        usedInTable->setItem(i, 3, new QTableWidgetItem(QString::number(sc->getNumberOfStructures())));
    }

    // update view
    usedInTable->resizeColumnsToContents();
    usedInTable->setSortingEnabled(true);
    usedInTable->sortByColumn(1, Qt::AscendingOrder);
}

// ---------------------- usedInTableClicked ----------------------------
void AtomDCWidget::usedInTableClicked(int row, int column) {
    // get the currently selected SC of a
    Atom *a = myDC->getAtom();
    StructuralComponent *sc = a->getStructuralComponent(row);

    // get the corresponding DC
    camitk::Component * toSelect;
    if (sc->isInstanceOf("Cell"))
        toSelect = myDC->getPMManagerDC()->getDC(dynamic_cast<Cell *>(sc));
    else
        toSelect = myDC->getPMManagerDC()->getDC(sc);

    // select it!
    toSelect->setSelected(true);
    toSelect->refresh();
}


