//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/SampleForm.h
//! @brief     Defines class SampleForm
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SAMPLEFORM_H
#define BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SAMPLEFORM_H

#include <QPushButton>
#include <QVBoxLayout>

class LayerForm;
class LayerItem;
class SampleEditorController;
class SampleItem;

//! Form to present/edit a sample
class SampleForm : public QWidget {
    Q_OBJECT
public:
    SampleForm(QWidget* parent, SampleItem* sampleItem, SampleEditorController* ec);

    //! Show or hide all buttons related to structure editing (like "add layer", "remove particle")
    void showInlineEditButtons(bool b);

    //! Create widgets for the new layer.
    void onLayerAdded(LayerItem* layerItem);

    //! Call this when a layerItem has been moved to a different position.
    //!
    //! This updates the item's position in the layout.
    void onLayerMoved(LayerItem* layerItem);

    //! Call this before removing (deleting) a LayerItem.
    //!
    //! Any widgets related to the item will be deleted or scheduled for later deletion.
    void onAboutToRemoveLayer(LayerItem* layerItem);

    void updateRowVisibilities();

    //! Update the presented units in all contained widgets according to current settings.
    void updateUnits();

    void ensureVisible(QWidget* w);

    //! Show values in Angstrom or nanometers
    void setUseAngstrom(bool angstrom);
    bool useAngstrom() const;

    //! Show values in radiants or degrees
    void setUseRadiant(bool radiant);
    bool useRadiant() const;

    //! Shows or hides the "Add Layer" buttons.
    void showAddLayerButtons(bool show);

    //! Search for the next LayerForm, starting from the given widget.
    //!
    //! The search starts with the given widget itself If it is a LayerForm, it is returned.
    //! If no following LayerForm is found, nullptr is returned.
    LayerForm* findNextLayerForm(QWidget* w);

private:
    QVBoxLayout* m_layout;
    SampleItem* m_sampleItem;     //!< Ptr is borrowed, don't delete
    SampleEditorController* m_ec; //!< Ptr is borrowed, don't delete
    bool m_showInlineEditButtons = false;
    bool m_useAngstrom;
    bool m_useRadiant;
    QList<QPushButton*> m_addLayerButtons;
};

#endif // BORNAGAIN_GUI_VIEW_SAMPLEDESIGNER_SAMPLEFORM_H
