// ---------------------------------------------------------------------------
// - IpPoint.cpp                                                             -
// - afnix:mth module - integer plane point class implementation             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2012 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "IpPoint.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
 
namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default point

  IpPoint::IpPoint (void) {
    d_x = 0L;
    d_y = 0L;
  }
  
  // create a point by coordinates

  IpPoint::IpPoint (const long x, const long y) {
    d_x = x;
    d_y = y;
  }

  // copy construct this object

  IpPoint::IpPoint (const IpPoint& that) {
    that.rdlock ();
    try {
      d_x = that.d_x;
      d_y = that.d_y;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // return the class name
  
  String IpPoint::repr (void) const {
    return "IpPoint";
  }
  
  // return a clone of this object
  
  Object* IpPoint::clone (void) const {
    return new IpPoint (*this);
  }

  // assign an object to this one

  IpPoint& IpPoint::operator = (const IpPoint& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      d_x = that.d_x;
      d_y = that.d_y;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // set the point x coordinate

  void IpPoint::setx (const long x) {
    wrlock ();
    try {
      d_x = x;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the point x coordinate

  long IpPoint::getx (void) const {
    rdlock ();
    try {
      long result = d_x;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the point y coordinate

  void IpPoint::sety (const long y) {
    wrlock ();
    try {
      d_y = y;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the point y coordinate

  long IpPoint::gety (void) const {
    rdlock ();
    try {
      long result = d_y;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      
      throw;
    }
  }

  // set the point coordinates

  void IpPoint::set (const long x, const long y) {
    wrlock ();
    try {
      d_x = x;
      d_y = y;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 5;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the rvi supported quarks
  static const long QUARK_SET  = zone.intern ("set");
  static const long QUARK_SETX = zone.intern ("set-x");
  static const long QUARK_GETX = zone.intern ("get-x");
  static const long QUARK_SETY = zone.intern ("set-y");
  static const long QUARK_GETY = zone.intern ("get-y");

  // create a new object in a generic way
  
  Object* IpPoint::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) return new IpPoint;
    // check for 2 arguments
    if (argc == 2) {
      long x = argv->getlong (0);
      long y = argv->getlong (1);
      return new IpPoint (x, y);
    }
    // invalid arguments
    throw Exception ("argument-error", 
		     "invalid arguments with plane point object");
  }

  // return true if the given quark is defined

  bool IpPoint::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true){
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* IpPoint::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETX) return new Integer (getx ());
      if (quark == QUARK_GETY) return new Integer (gety ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETX) {
	long x = argv->getlong (0);
	setx (x);
	return nilp;
      }
      if (quark == QUARK_SETY) {
	long y = argv->getlong (0);
	sety (y);
	return nilp;
      }
    }
    // dispatch 2 arguments
    if (argc == 2) {
      if (quark == QUARK_SET) {
	long x = argv->getlong (0);
	long y = argv->getlong (1);
	set (x, y);
	return nilp;
      }
    }
    // call the object
    return Object::apply (robj, nset, quark, argv);
  }
}
