!{\src2tex{textfont=tt}}
!!****f* ABINIT/scalapack
!! NAME
!! scalapack
!!
!! FUNCTION
!! This module contains functions and subroutine using ScaLAPACK library.
!! The code have to be compiled with the HAVE_SCALAPACK CPP flags.
!!
!! COPYRIGHT
!! Copyright (C) 2001-2005 ABINIT group (CS,GZ,FB)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!!
!! TODO
!! To be translated
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif
!-------------------------------------------------------
! set up of a processor grid for ScaLAPACK
! as a function of the total number of processrs attributed to the grid
!-------------------------------------------------------

SUBROUTINE construire_grille_scalapack(grille,nbprocs, communicator)

#if defined HAVE_SCALAPACK

  use defs_scalapack
  use defs_basis

  TYPE(grille_scalapack),INTENT(out)     :: grille
  INTEGER,INTENT(in)                     :: nbprocs
  INTEGER, INTENT(in)                    :: communicator

  INTEGER  :: i

  grille%nbprocs=nbprocs

  ! recherche d'une grille rectangulaire de processeurs
  i=INT(SQRT(float(nbprocs)))
  DO WHILE (MOD(nbprocs,i) /= 0)
     i = i-1
  END DO

  grille%dims(1) = i
  grille%dims(2) = INT(nbprocs/i)

  grille%ictxt = communicator

  CALL BLACS_GRIDINIT(grille%ictxt,'R',grille%dims(1),grille%dims(2))           

#endif

END SUBROUTINE construire_grille_scalapack

!-------------------------------------------------------
! construction des donnees relatives  un processeur dans un grille
!-------------------------------------------------------

#if defined HAVE_SCALAPACK

SUBROUTINE construire_processeur_scalapack(processeur,grille,myproc, comm)

  use defs_scalapack
  use defs_basis
  TYPE(processeur_scalapack),INTENT(out)  :: processeur
  TYPE(grille_scalapack),INTENT(in)       :: grille
  INTEGER,INTENT(in)                      :: myproc
  INTEGER,INTENT(in)                      :: comm

  processeur%grille = grille

  processeur%myproc = myproc

  processeur%comm = comm

  CALL BLACS_GRIDINFO(grille%ictxt,processeur%grille%dims(1), &
       &              processeur%grille%dims(2),processeur%coords(1), &
       &              processeur%coords(2))

  ! ces valeurs sont les memes que celles calculees par BLACS_GRIDINFO
  ! sauf dans le cas ou l'argument myproc n'est pas le processeur reel
  ! sur lequel on est.
  processeur%coords(1) = INT((myproc) / grille%dims(2))
  processeur%coords(2) = MOD((myproc), grille%dims(2))


END SUBROUTINE construire_processeur_scalapack

!-------------------------------------------------------
! initialisation generale de ScaLAPACK
!-------------------------------------------------------

SUBROUTINE init_scalapack(processeur,communicator)

  use defs_scalapack
  use defs_basis
#  if defined MPI || defined MPI_FFT

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => init_scalapack
#endif
!End of the abilint section

         include 'mpif.h'
#  endif

  TYPE(processeur_scalapack),INTENT(out)    :: processeur
  INTEGER, INTENT(in)                       :: communicator

  TYPE(grille_scalapack)                    :: grille
  INTEGER                                   :: nbproc,myproc
  INTEGER                                   :: ierr


  CALL MPI_COMM_SIZE(communicator, nbproc, ierr)
  CALL MPI_COMM_RANK(communicator, myproc, ierr)

  CALL construire_grille_scalapack(grille, nbproc, communicator)
  CALL construire_processeur_scalapack(processeur, grille, myproc, communicator)

END SUBROUTINE init_scalapack


!-------------------------------------------------------
! finalisation generale de ScaLAPACK
!-------------------------------------------------------

SUBROUTINE end_scalapack(processeur)

  use defs_scalapack
  use defs_basis
  TYPE(processeur_scalapack),INTENT(inout)    :: processeur

  CALL BLACS_GRIDEXIT(processeur%grille%ictxt)

  !CALL BLACS_EXIT(0)

END SUBROUTINE end_scalapack

!-------------------------------------------------------
! initialisation d'une matrice ScaLAPACK (chaque processeur initialise
! sa propre partie de la matrice)
!-------------------------------------------------------

SUBROUTINE init_matrice_scalapack(matrice,nbli_global, &
     &                                      nbco_global,processeur,tbloc)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => init_matrice_scalapack
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(out)    :: matrice
  TYPE(processeur_scalapack),INTENT(in),TARGET  :: processeur
  INTEGER,INTENT(in)                     :: nbli_global,nbco_global
  INTEGER,INTENT(in),OPTIONAL            :: tbloc

  INTEGER, PARAMETER                :: TAILLE_BLOCS = 40
  INTEGER             :: info,taille

  INTEGER :: NUMROC
  EXTERNAL NUMROC

  IF (PRESENT(tbloc)) THEN
     taille = tbloc
  ELSE
     taille = TAILLE_BLOCS
  END IF

  ! champs du type matrice:
  matrice%processeur => processeur
  matrice%taille_blocs(1) = MIN(taille,nbli_global)
  matrice%taille_blocs(2) = MIN(taille,nbco_global)
  matrice%taille_globale(1) = nbli_global
  matrice%taille_globale(2) = nbco_global


  ! taille du buffer local
  matrice%taille_locale(1) = NUMROC(nbli_global,matrice%taille_blocs(1), &
       &                            processeur%coords(1),0, &
       &                            processeur%grille%dims(1))

  matrice%taille_locale(2) = NUMROC(nbco_global,matrice%taille_blocs(2), &
       &                            processeur%coords(2),0, &
       &                            processeur%grille%dims(2))

  CALL idx_loc(matrice,matrice%taille_globale(1),matrice%taille_globale(2), &
       &       matrice%taille_locale(1),matrice%taille_locale(2))

  !initialisation de la description ScaLAPACK de la matrice
  CALL DESCINIT(matrice%descriptif%tab, nbli_global, nbco_global, &
       &        matrice%taille_blocs(1), matrice%taille_blocs(2), 0,0 , &
       &        processeur%grille%ictxt, MAX(1,matrice%taille_locale(1)), &
       &        info)

  IF (info /= 0) THEN
     PRINT *,processeur%myproc,'erreur initialisation matrice scalapack',info
  END IF

  ALLOCATE(matrice%buffer(matrice%taille_locale(1),matrice%taille_locale(2)))

  matrice%buffer(:,:) = (0._DP,0._DP)

END SUBROUTINE init_matrice_scalapack

!-------------------------------------------------------
! copie d'une matrice ScaLAPACK
!-------------------------------------------------------
SUBROUTINE matrice_copie(source,destination)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(in)  :: source
  TYPE(matrice_scalapack),INTENT(out) :: destination

  destination%processeur => source%processeur
  destination%taille_globale = source%taille_globale
  destination%taille_locale = source%taille_locale
  destination%taille_blocs = source%taille_blocs
  destination%descriptif%tab = source%descriptif%tab

  ALLOCATE(destination%buffer(destination%taille_locale(1),destination%taille_locale(2)))

  destination%buffer = source%buffer

  IF(ASSOCIATED(source%ipiv)) THEN
     ALLOCATE(destination%ipiv(SIZE(source%ipiv)))
     destination%ipiv = source%ipiv
  END IF
!    destination% = source%

END SUBROUTINE matrice_copie

!-------------------------------------------------------
! destruction des champs d'un matrice ScaLAPACK
!-------------------------------------------------------
SUBROUTINE destruction_matrice_scalapack(matrice)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(inout)    :: matrice

  NULLIFY(matrice%processeur)
  matrice%taille_globale = 0
  IF (ASSOCIATED(matrice%buffer)) THEN
     DEALLOCATE(matrice%buffer)
  ENDIF
  IF (ASSOCIATED(matrice%ipiv)) THEN
     DEALLOCATE(matrice%ipiv)
  ENDIF
  matrice%taille_blocs = 0
  matrice%taille_locale = 0
  matrice%descriptif%tab = 0

END SUBROUTINE destruction_matrice_scalapack


!-------------------------------------------------------
!-------------------------------------------------------
! routines:
! acces aux composantes d'un matrice
!-------------------------------------------------------
!-------------------------------------------------------



!-------------------------------------------------------
! interrogation d'une composante par ses indices locaux
!-------------------------------------------------------

FUNCTION matrice_get_local(matrice,i,j)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(in)    :: matrice
  INTEGER, INTENT(in)                   :: i,j
  COMPLEX(dp)                           :: matrice_get_local

  matrice_get_local = matrice%buffer(i,j)

END FUNCTION matrice_get_local

!-------------------------------------------------------
! positionnement d'une composante de la matrice par ses indices locaux
!-------------------------------------------------------

SUBROUTINE matrice_set_local(matrice,i,j,valeur)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(out)   :: matrice
  INTEGER, INTENT(in)                   :: i,j
  COMPLEX(dp)                           :: valeur

  matrice%buffer(i,j) = valeur

END SUBROUTINE matrice_set_local

!-------------------------------------------------------
! ajout d'une valeur a une composante de la matrice par ses indices locaux
!-------------------------------------------------------

SUBROUTINE matrice_add_local(matrice,i,j,valeur)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(out)   :: matrice
  INTEGER, INTENT(in)                   :: i,j
  COMPLEX(dp)                           :: valeur

  matrice%buffer(i,j) = matrice%buffer(i,j) + valeur

END SUBROUTINE matrice_add_local

!-------------------------------------------------------
! determine si un terme d'indices locaux donne est stocke en local
!-------------------------------------------------------

FUNCTION idx_processeur_est_local(matrice,i,j)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => idx_processeur_est_local
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(in)    :: matrice
  INTEGER, INTENT(in)                   :: i,j

  LOGICAL ::idx_processeur_est_local
!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifndef HAVE_FORTRAN_INTERFACES
 INTEGER :: idx_processeur_concerne
#endif
!End of the abilint section

  idx_processeur_est_local = (matrice%processeur%coords(1) == &
       &                      idx_processeur_concerne(matrice,i,1)) &
       &                    .AND. (matrice%processeur%coords(2) == &
       &                           idx_processeur_concerne(matrice,j,2))

END FUNCTION idx_processeur_est_local

!-------------------------------------------------------
! determine le numero de ligne/colonne de processeurs concernes par un
! numero de ligne/colonne globale de la matrice
!-------------------------------------------------------

FUNCTION idx_processeur_concerne(matrice,idx,lico)

  use defs_scalapack
  use defs_basis

  TYPE(matrice_scalapack),INTENT(in)    :: matrice
  INTEGER, INTENT(in)                   :: idx,lico !lico= ligne ou colonne

  INTEGER :: idx_processeur_concerne
  INTEGER :: INDXG2P
  EXTERNAL INDXG2P


  idx_processeur_concerne = INDXG2P(idx,matrice%taille_blocs(lico),0,0, &
       &                               matrice%processeur%grille%dims(lico))
END FUNCTION idx_processeur_concerne

!-------------------------------------------------------
! determine les indices locaux d'un terme de la matrice par rapport a ses
! indices globaux independemment du processeur qui est concerne
!-------------------------------------------------------

SUBROUTINE idx_loc(matrice,i,j,iloc,jloc)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => idx_loc
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(in)    :: matrice
  INTEGER, INTENT(in)                   :: i,j
  INTEGER, INTENT(out)                  :: iloc,jloc

  INTEGER :: NUMROC
  EXTERNAL NUMROC
!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifndef HAVE_FORTRAN_INTERFACES
 INTEGER :: glob_loc
#endif
!End of the abilint section

  iloc = glob_loc(matrice,i,1)

  jloc = glob_loc(matrice,j,2)

END SUBROUTINE idx_loc

!-------------------------------------------------------
! 
!-------------------------------------------------------
FUNCTION glob_loc(matrice,idx,lico)

  use defs_scalapack
  use defs_basis

  TYPE(matrice_scalapack),INTENT(in)    :: matrice
  INTEGER, INTENT(in)                   :: idx, lico

  INTEGER :: glob_loc

  INTEGER :: NUMROC
  EXTERNAL NUMROC

  glob_loc = NUMROC(idx,matrice%taille_blocs(lico), &
       &        matrice%processeur%coords(lico),0, &
       &        matrice%processeur%grille%dims(lico))


END FUNCTION glob_loc

!-------------------------------------------------------
! determine les indices globaux d'un terme de la matrice par rapport a ses
! indices locaux 
!-------------------------------------------------------

SUBROUTINE idx_glob(matrice,iloc,jloc,i,j)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => idx_glob
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(in)    :: matrice
  INTEGER, INTENT(out)                  :: i,j
  INTEGER, INTENT(in)                   :: iloc,jloc

  INTEGER :: nbcycli,nbcycco,resteli,resteco,nblocsli,nblocsco
!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifndef HAVE_FORTRAN_INTERFACES
 INTEGER :: loc_glob
#endif
!End of the abilint section

  i = loc_glob(matrice,matrice%processeur,iloc,1)
  j = loc_glob(matrice,matrice%processeur,jloc,2)

END SUBROUTINE idx_glob

!-------------------------------------------------------
! determine l'indice global d'un indice local (ligne ou colonne)
! en fonction d'un processeur donn
!-------------------------------------------------------
FUNCTION loc_glob(matrice,proc,idx,lico)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(in)    :: matrice
  TYPE(processeur_scalapack),INTENT(in) :: proc
  INTEGER, INTENT(in)                   :: idx,lico

  INTEGER :: loc_glob

  INTEGER :: nbcyc,reste,nblocs

  nbcyc = INT((idx-1)/matrice%taille_blocs(lico))
  reste = MOD(idx-1,matrice%taille_blocs(lico))
  nblocs = nbcyc*proc%grille%dims(lico)+ &
       & proc%coords(lico)

  loc_glob = nblocs * matrice%taille_blocs(lico) + reste + 1

END FUNCTION loc_glob

!-------------------------------------------------------
! interrogation d'un terme de la matrice a partir de ses indices globaux
! ATTENTION: ceci n'est valable que si le processeur procedant a l'appel
! possede effectivement ce terme en local
!-------------------------------------------------------

FUNCTION matrice_get_global(matrice,i,j)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_get_global
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(in)    :: matrice
  INTEGER, INTENT(in)                   :: i,j
  COMPLEX(dp)                           :: matrice_get_global

  INTEGER :: iloc,jloc
!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifndef HAVE_FORTRAN_INTERFACES
 COMPLEX(dp) :: matrice_get_local
#endif
!End of the abilint section

  CALL idx_loc(matrice,i,j,iloc,jloc)

  matrice_get_global = matrice_get_local(matrice,iloc,jloc)

END FUNCTION matrice_get_global

!-------------------------------------------------------
! positionnement de la valeur d'un terme de la matrice par ses indices globaux
! ATTENTION: ceci n'est valable que si le processeur procedant a l'appel
! possede effectivement ce terme en local
!-------------------------------------------------------

SUBROUTINE matrice_set_global(matrice,i,j,valeur)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_set_global
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(inout) :: matrice
  INTEGER, INTENT(in)                   :: i,j
  COMPLEX(dp)                           :: valeur

  INTEGER :: iloc,jloc

  CALL idx_loc(matrice,i,j,iloc,jloc)

  CALL matrice_set_local(matrice,iloc,jloc,valeur)

END SUBROUTINE matrice_set_global

!-------------------------------------------------------
! ajout d'une valeur a un terme de la matrice par ses indices globaux
! ATTENTION: ceci n'est valable que si le processeur procedant a l'appel
! possede effectivement ce terme en local
!-------------------------------------------------------

SUBROUTINE matrice_add_global(matrice,i,j,valeur)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_add_global
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(inout) :: matrice
  INTEGER, INTENT(in)                   :: i,j
  COMPLEX(dp)                           :: valeur

  INTEGER :: iloc,jloc

  CALL idx_loc(matrice,i,j,iloc,jloc)

  CALL matrice_add_local(matrice,iloc,jloc,valeur)

END SUBROUTINE matrice_add_global


!-------------------------------------------------------
! routine de verification d'une matrice ScaLAPACK par rapport a une
! matrice entiere
!-------------------------------------------------------

SUBROUTINE matrice_verification_globale(matrice,reference)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_verification_globale
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(in)   :: matrice
  COMPLEX(dp),DIMENSION(:,:)           :: reference

  INTEGER :: i,j,iglob,jglob
  COMPLEX(dp) :: diff,z
  REAL(dp) :: err
  INTEGER :: cptr
!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifndef HAVE_FORTRAN_INTERFACES
 COMPLEX(dp) :: matrice_get_local
#endif
!End of the abilint section

  err = 0._DP
  cptr = 0

  DO i=1,matrice%taille_locale(1)
     DO j=1,matrice%taille_locale(2)
        CALL idx_glob(matrice,i,j,iglob,jglob)
        z = matrice_get_local(matrice,i,j)
        diff = z-reference(iglob,jglob)
        err = err + ABS(diff)/MAX(ABS(z),MAX(1.E-35,ABS(reference(iglob,jglob))))
        cptr = cptr + 1
     END DO
  END DO

  IF (cptr /= 0) THEN
     PRINT *,matrice%processeur%myproc,"erreur Linf matrice scalapack", &
          &  err,"sur",cptr,"termes"
  END IF
END SUBROUTINE matrice_verification_globale

!-------------------------------------------------------
! routine de verification d'une portion de matrice ScaLAPACK par rapport a un
! vecteur 
!-------------------------------------------------------

SUBROUTINE matrice_verif_globale_vecteur(matrice,reference,decli,nom)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_verif_globale_vecteur
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(in)   :: matrice
  COMPLEX(dp),DIMENSION(:)             :: reference
  INTEGER, OPTIONAL                    :: decli
  CHARACTER(len=*),OPTIONAL            :: nom

  INTEGER :: i,j,deb
  COMPLEX(dp) :: diff,z
  REAL(dp) :: err
  INTEGER :: cptr
  CHARACTER(len=150)            :: nomvec
!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifndef HAVE_FORTRAN_INTERFACES
 INTEGER :: idx_processeur_concerne
 COMPLEX(dp) :: matrice_get_global
#endif
!End of the abilint section

  IF (PRESENT(decli)) THEN
     deb = decli
  ELSE
     deb = 0
  END IF
  IF (PRESENT(nom)) THEN
     nomvec=nom
  ELSE
     nomvec='(vecteur)'
  END IF

  err = 0._DP
  cptr = 0

  j=1
  IF (matrice%processeur%coords(2) == &
       & idx_processeur_concerne(matrice,j,2)) THEN
     DO i=1,MIN(UBOUND(reference,1),matrice%taille_globale(1)-deb)
        IF (matrice%processeur%coords(1) == &
             & idx_processeur_concerne(matrice,deb+i,1)) THEN
           z = matrice_get_global(matrice,deb+i,j)
           diff = z-reference(i)
           err = err + ABS(diff)/MAX(ABS(z),MAX(1.E-35,ABS(reference(i))))
           !print *,'erreur',i,j
           cptr = cptr + 1
        END IF
     END DO
  END IF
  IF (cptr /= 0) THEN
     PRINT *,matrice%processeur%myproc,"erreur L1 ",TRIM(nomvec), &
          &  err/cptr,'sur',cptr,'termes stockes'
  END IF
END SUBROUTINE matrice_verif_globale_vecteur


!-----------------------------------------------------------------
! routine de remplissage d'une matrice ScaLAPACK par rapport a une
! matrice entiere
!-----------------------------------------------------------------

SUBROUTINE matrice_from_globale(matrice,reference)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_from_globale
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(inout)  :: matrice
  REAL(dp),DIMENSION(:)                  :: reference
  COMPLEX(dp)::val

  INTEGER :: i,j,iglob,jglob,ind
  REAL(dp) :: err
  INTEGER :: cptr

!    err = 0._DP
!    cptr = 0

  DO i=1,matrice%taille_locale(1)
     DO j=1,matrice%taille_locale(2)
        CALL idx_glob(matrice,i,j,iglob,jglob)

        ind = jglob*(jglob-1)+2*iglob-1
        val = dcmplx(reference(ind),reference(ind+1))
        CALL matrice_set_local(matrice,i,j,val)

!          cptr = cptr + 1
     END DO
  END DO

!    IF (cptr /= 0) THEN
!       PRINT *,matrice%processeur%myproc,"erreur Linf matrice scalapack", &
!            &  err,"sur",cptr,"termes"
!    END IF

END SUBROUTINE matrice_from_globale

!-------------------------------------------------------------------
! routine de remplissage d'une matrice entiere ScaLAPACK par rapport
! a une matrice ScaLAPACK
!-------------------------------------------------------------------
SUBROUTINE matrice_to_reference(matrice,reference)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_to_reference
#endif
!End of the abilint section

  TYPE(matrice_scalapack),INTENT(in)        :: matrice
  REAL(dp),DIMENSION(:,:),INTENT(inout)     :: reference

  INTEGER  :: i,j,iglob,jglob,ind
  REAL(dp) :: err
  INTEGER  :: cptr

!    err = 0._DP
!    cptr = 0

  DO i=1,matrice%taille_locale(1)
     DO j=1,matrice%taille_locale(2)
        CALL idx_glob(matrice,i,j,iglob,jglob)

        ind=(iglob-1)*2+1
        reference(ind,jglob)   = REAL(matrice%buffer(i,j))
        reference(ind+1,jglob) = IMAG(matrice%buffer(i,j))

!          cptr = cptr + 1
     END DO
  END DO

!    IF (cptr /= 0) THEN
!       PRINT *,matrice%processeur%myproc,"erreur Linf matrice scalapack", &
!            &  err,"sur",cptr,"termes"
!    END IF

END SUBROUTINE matrice_to_reference

!-------------------------------------------------------
! stockage d'un vecteur dans une colonne d'une matrice ScaLAPACK
!-------------------------------------------------------
SUBROUTINE matrice_rangement_local_vecteur(matrice,tableau,decli,decco)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_rangement_local_vecteur
#endif
!End of the abilint section

  COMPLEX(dp),DIMENSION(:),INTENT(in)       :: tableau
  TYPE(matrice_scalapack),INTENT(inout)     :: matrice
  INTEGER, INTENT(in)                       :: decli ! decalage de lignes
                                                     ! dans la matrice
  INTEGER, INTENT(in)                       :: decco ! decalage de colonnes
                                                     ! dans la matrice

  INTEGER :: numli


  ! si le terme est stocke localement
!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifndef HAVE_FORTRAN_INTERFACES
 INTEGER :: idx_processeur_concerne
#endif
!End of the abilint section

  IF (matrice%processeur%coords(2) == &
       & idx_processeur_concerne(matrice,decco+1,2)) THEN

     DO numli = 1, UBOUND(tableau,1)
        IF (matrice%processeur%coords(1) == &
             & idx_processeur_concerne(matrice,decli+numli,1)) THEN

           CALL matrice_set_global(matrice,decli+numli,decco+1, &
                &                  tableau(numli))

        END IF
     END DO
  END IF

END SUBROUTINE matrice_rangement_local_vecteur

!-------------------------------------------------------
! extrait les composantes d'une colonne de la matrice vers un vecteur.
! les termes du vecteur non stockes localement par la matrice sont laisses
! inchanges.
!-------------------------------------------------------
SUBROUTINE matrice_extraction_vecteur(matrice,tableau,decli,decco,finli)

  use defs_scalapack
  use defs_basis

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi, except_this_one => matrice_extraction_vecteur
#endif
!End of the abilint section

  COMPLEX(dp),DIMENSION(:),INTENT(out)      :: tableau
  TYPE(matrice_scalapack),INTENT(in)        :: matrice
  INTEGER, INTENT(in)                       :: decli ! decalage de lignes
                                                     ! dans la matrice
  INTEGER, INTENT(in)                       :: decco ! decalage de colonnes
                                                     ! dans la matrice
  INTEGER, INTENT(in), OPTIONAL             :: finli

  INTEGER :: i,iglob,jglob,maxli,debli,debco
  COMPLEX(dp) :: x
!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifndef HAVE_FORTRAN_INTERFACES
 INTEGER :: idx_processeur_concerne
 COMPLEX(dp) :: matrice_get_local
#endif
!End of the abilint section

  IF (PRESENT(finli)) THEN
     maxli = finli
  ELSE
     maxli = matrice%taille_globale(1)
  END IF

  i = decli+1

  DO WHILE (.NOT. matrice%processeur%coords(1) == &
       & idx_processeur_concerne(matrice,i,1))
     i=i+1
  END DO

  IF (i .GT. maxli) RETURN

  CALL idx_loc(matrice,i,decco+1,debli,debco)

  ! si on est concerne par la colonne jglob
  IF (matrice%processeur%coords(2) == &
       & idx_processeur_concerne(matrice,decco+1,2)) THEN

     DO i = debli,matrice%taille_locale(1)
        x = matrice_get_local(matrice,i,debco)
        CALL idx_glob(matrice,i,debco,iglob,jglob)

        IF (iglob .GT. maxli) THEN
           RETURN
        END IF
        tableau(iglob-decli) = x
     END DO
  END IF

END SUBROUTINE matrice_extraction_vecteur

!-------------------------------------------------------
! factorisation A=LU d'une matrice (carree)
!-------------------------------------------------------
SUBROUTINE matrice_pzgetrf(matrice)

  use defs_scalapack
  use defs_basis

  TYPE(matrice_scalapack),INTENT(inout)        :: matrice

  INTEGER                                        :: info, nipiv

  EXTERNAL PZGETRF

  IF (.NOT. ASSOCIATED(matrice%ipiv)) THEN
     nipiv = matrice%taille_globale(1) + matrice%taille_blocs(1)
     ALLOCATE(matrice%ipiv(nipiv))
     matrice%ipiv(:) = 0
  END IF

  CALL PZGETRF(matrice%taille_globale(1), matrice%taille_globale(2), &
       &       matrice%buffer(1,1),1,1,matrice%descriptif%tab, &
       &       matrice%ipiv, info)

  IF (info /= 0) THEN
     PRINT *,matrice%processeur%myproc,'erreur pzgetrf',info
  END IF
END SUBROUTINE matrice_pzgetrf

!-------------------------------------------------------
! resolution d'un systeme lineaire dont la matrice est factorisee
!-------------------------------------------------------
SUBROUTINE matrice_pzgetrs(matrice,vecteur)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(in)        :: matrice
  TYPE(matrice_scalapack),INTENT(inout)        :: vecteur

  INTEGER :: info

  CALL PZGETRS('N',matrice%taille_globale(1),vecteur%taille_globale(2), &
       &       matrice%buffer,1,1,matrice%descriptif%tab,matrice%ipiv, &
       &       vecteur%buffer,1,1,vecteur%descriptif%tab,info)

  IF (info /= 0) THEN
     PRINT *,matrice%processeur%myproc,'erreur pzgetrs',info
  END IF
END SUBROUTINE matrice_pzgetrs

!-------------------------------------------------------
! factorisation A=LLT d'une matrice (hermitienne)
!-------------------------------------------------------
SUBROUTINE matrice_pzpotrf(matrice)

  use defs_scalapack
  use defs_basis

  TYPE(matrice_scalapack),INTENT(inout)        :: matrice

  INTEGER                                        :: info, nipiv

  EXTERNAL PZPOTRF

  CALL PZPOTRF('U',matrice%taille_globale(1), &
       &       matrice%buffer(1,1),1,1,matrice%descriptif%tab,info)

  IF (info /= 0) THEN
     PRINT *,matrice%processeur%myproc,'erreur pzpotrf',info
  END IF
END SUBROUTINE matrice_pzpotrf

!-------------------------------------------------------
! resolution d'un systeme lineaire dont la matrice (hermitienne) est factorisee
!-------------------------------------------------------
SUBROUTINE matrice_pzpotrs(matrice,vecteur)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(in)        :: matrice
  TYPE(matrice_scalapack),INTENT(inout)        :: vecteur

  INTEGER :: info

  CALL PZPOTRS('U',matrice%taille_globale(1),vecteur%taille_globale(2), &
       &       matrice%buffer,1,1,matrice%descriptif%tab, &
       &       vecteur%buffer,1,1,vecteur%descriptif%tab,info)

  IF (info /= 0) THEN
     PRINT *,matrice%processeur%myproc,'erreur pzpotrs',info
  END IF
END SUBROUTINE matrice_pzpotrs

!-------------------------------------------------------
! produit matrice*matrice etendu.
! cette routine effectue le calcul suivant:
! C := alpha*A*B - beta*C
!
! pour effectuer un simple produit matrice x vecteur, il suffit
! de passer alpha = (1.,0.) et beta (0.,0.)
!-------------------------------------------------------
SUBROUTINE matrice_pzgemm(matrice1,alpha,matrice2,beta,resultat)

  use defs_scalapack
  use defs_basis
  TYPE(matrice_scalapack),INTENT(in)        :: matrice1,matrice2
  TYPE(matrice_scalapack),INTENT(inout)     :: resultat
  COMPLEX(dp), intent(in)                   :: alpha, beta

  CALL PZGEMM('N','N',matrice1%taille_globale(1),matrice2%taille_globale(2),&
       &      matrice1%taille_globale(2),alpha,matrice1%buffer,1,1, &
       &      matrice1%descriptif%tab,matrice2%buffer,1,1, &
       &      matrice2%descriptif%tab,beta,resultat%buffer,1,1, &
       &      resultat%descriptif%tab)

END SUBROUTINE matrice_pzgemm


!-------------------------------------------------------
!  Calcul of eigenvalues and eigenvectors
!-------------------------------------------------------
SUBROUTINE matrice_pzheevx(processeur,matrice,resultat,eigen,communicateur)

  use defs_scalapack
  use defs_basis
#  if defined MPI || defined MPI_FFT
         include 'mpif.h'
#  endif

  TYPE(processeur_scalapack),INTENT(in)       :: processeur
  TYPE(matrice_scalapack),INTENT(in)          :: matrice
  TYPE(matrice_scalapack),INTENT(inout)       :: resultat
  DOUBLE PRECISION,DIMENSION(:),INTENT(inout) :: eigen
  INTEGER,INTENT(in)  :: communicateur

  INTEGER            :: LRWORK,LIWORK,LWORK,INFO
 
  INTEGER         , dimension(1) :: IWORK_tmp
  DOUBLE PRECISION, dimension(1) :: RWORK_tmp
  COMPLEX(dp)     , dimension(1) :: WORK_tmp

  INTEGER         , allocatable  :: IWORK(:)
  DOUBLE PRECISION, allocatable  :: RWORK(:)
  COMPLEX(dp)     , allocatable  :: WORK(:)

  INTEGER,          allocatable :: ICLUSTR(:)
  INTEGER,          allocatable :: IFAIL(:)
  DOUBLE PRECISION, allocatable :: GAP(:)

  DOUBLE PRECISION, PARAMETER :: ABSTOL=-1.D+0,ORFAC=-1.D+0
  INTEGER,          PARAMETER :: IZERO=0

  INTEGER ::  M,NZ,IA,JA,IZ,JZ,ierr,TWORK_tmp(3),TWORK(3)


  INFO   = 0   

  ! Allocation des variables de resultats de calcul
  allocate(IFAIL(matrice%taille_globale(2)))
  allocate(ICLUSTR(2*processeur%grille%dims(1)*processeur%grille%dims(2)))
  allocate(GAP(processeur%grille%dims(1)*processeur%grille%dims(2)))
  
  ! Rcupration des tailles des tableaux de travail
  CALL PZHEEVX('V','A','U',&
       &      matrice%taille_globale(2),&
       &      matrice%buffer,1,1,matrice%descriptif%tab, &
       &      ZERO,ZERO,IZERO,IZERO,ABSTOL,&
       &      m,nz,eigen,ORFAC, &
       &      resultat%buffer,1,1,resultat%descriptif%tab, &
       &      WORK_tmp,-1,RWORK_tmp,-1,IWORK_tmp,-1,&
       &      IFAIL,ICLUSTR,GAP,INFO)

  TWORK_tmp(1) = IWORK_tmp(1)
  TWORK_tmp(2) = INT(RWORK_tmp(1))
  TWORK_tmp(3) = INT(WORK_tmp(1))

 ! Rcupration des maximum des tailles des tableaux de travailprocesseur%comm
  CALL MPI_ALLREDUCE(TWORK_tmp,TWORK,3,MPI_INTEGER,MPI_MAX,communicateur,ierr)

  LIWORK = TWORK(1)
  LRWORK = TWORK(2)
  LWORK  = TWORK(3)

 ! Allocation des tableaux de travail
  allocate(IWORK(LIWORK))
  allocate(WORK(LWORK))
  allocate(RWORK(LRWORK))

  ! Appel des la fonction de calcul
  CALL PZHEEVX('V','A','U',&
       &      matrice%taille_globale(2),&
       &      matrice%buffer,1,1,matrice%descriptif%tab, &
       &      ZERO,ZERO,IZERO,IZERO,ABSTOL,&
       &      m,nz,eigen,ORFAC, &
       &      resultat%buffer,1,1,resultat%descriptif%tab, &
       &      WORK,LWORK,RWORK,LRWORK,IWORK,LIWORK,&
       &      IFAIL,ICLUSTR,GAP,INFO)


  deallocate(IFAIl,ICLUSTR,GAP,WORK,RWORK,IWORK)

END SUBROUTINE matrice_pzheevx

!-------------------------------------------------------
!  Calcul of eigenvalues and eigenvectors
!  A * X = lambda * B * X 
!-------------------------------------------------------
SUBROUTINE matrice_pzhegvx(processeur,matrice1,matrice2,resultat,eigen,communicateur)

  use defs_scalapack
  use defs_basis
#  if defined MPI || defined MPI_FFT
         include 'mpif.h'
#  endif
  
  TYPE(processeur_scalapack),INTENT(in)       :: processeur
  TYPE(matrice_scalapack),INTENT(in)          :: matrice1,matrice2
  TYPE(matrice_scalapack),INTENT(inout)       :: resultat
  DOUBLE PRECISION,DIMENSION(:),INTENT(inout) :: eigen

  INTEGER,INTENT(in)  :: communicateur

  INTEGER            :: LRWORK,LIWORK,LWORK,INFO
 
  INTEGER         , dimension(1) :: IWORK_tmp
  DOUBLE PRECISION, dimension(1) :: RWORK_tmp
  COMPLEX(dp)     , dimension(1) :: WORK_tmp

  INTEGER         , allocatable  :: IWORK(:)
  DOUBLE PRECISION, allocatable  :: RWORK(:)
  COMPLEX(dp)     , allocatable  :: WORK(:)


  INTEGER,          allocatable :: ICLUSTR(:)
  INTEGER,          allocatable :: IFAIL(:)
  DOUBLE PRECISION, allocatable :: GAP(:)

  DOUBLE PRECISION, PARAMETER :: ABSTOL=-1.D+0,ORFAC=-1.D+0
  INTEGER         , PARAMETER :: IZERO=0

  INTEGER ::  M,NZ,IA,JA,IZ,JZ,ierr,TWORK_tmp(3),TWORK(3)


  INFO   = 0   

  ! Allocation des variables de rsultats de calcul
  allocate(IFAIL(matrice1%taille_globale(2)))
  allocate(ICLUSTR(2*processeur%grille%dims(1)*processeur%grille%dims(2)))
  allocate(GAP(processeur%grille%dims(1)*processeur%grille%dims(2)))

  ! Rcupration des tailles des tableaux de travail
  CALL PZHEGVX(1,'V','A','U',&
       &      matrice1%taille_globale(2),&
       &      matrice1%buffer,1,1,matrice1%descriptif%tab, &
       &      matrice2%buffer,1,1,matrice2%descriptif%tab, &
       &      ZERO,ZERO,IZERO,IZERO,ABSTOL,&
       &      m,nz,eigen,ORFAC, &
       &      resultat%buffer,1,1,resultat%descriptif%tab, &
       &      WORK_tmp,-1,RWORK_tmp,-1,IWORK_tmp,-1,&
       &      IFAIL,ICLUSTR,GAP,INFO)


  TWORK_tmp(1) = IWORK_tmp(1)
  TWORK_tmp(2) = INT(RWORK_tmp(1))
  TWORK_tmp(3) = INT(WORK_tmp(1))
 
 ! Rcupration des maximum des tailles des tableaux de travailprocesseur%comm
  CALL MPI_ALLREDUCE(TWORK_tmp,TWORK,3,MPI_INTEGER,MPI_MAX,communicateur,ierr)

  LIWORK = TWORK(1)
  LRWORK = TWORK(2)
  LWORK  = TWORK(3)

 ! Allocation des tableaux de travail
  allocate(IWORK(LIWORK))
  allocate(WORK(LWORK))
  allocate(RWORK(LRWORK))

  ! Appel des la fonction de calcul
  CALL PZHEGVX(1,'V','A','U',&
       &      matrice1%taille_globale(2),&
       &      matrice1%buffer,1,1,matrice1%descriptif%tab, &
       &      matrice2%buffer,1,1,matrice2%descriptif%tab, &
       &      ZERO,ZERO,IZERO,IZERO,ABSTOL,&
       &      m,nz,eigen,ORFAC, &
       &      resultat%buffer,1,1,resultat%descriptif%tab, &
       &      WORK,LWORK,RWORK,LRWORK,IWORK,LIWORK,&
       &      IFAIL,ICLUSTR,GAP,INFO)

  deallocate(IFAIl,ICLUSTR,GAP,IWORK,WORK,RWORK)

END SUBROUTINE matrice_pzhegvx

#endif
