"""
Unit tests for the S390xGeneration module.
"""

from unittest.mock import mock_open, patch
from DistUpgrade.S390xGeneration import (
    S390xGeneration,
    get_system_s390x_generation,
)


class TestS390xGeneration:
    """
    Unit tests for the S390xGeneration module.
    """

    def test_generation_ordering(self):
        """
        Test that S390xGeneration instances are ordered correctly.
        """
        generations = list(S390xGeneration)
        current = generations[0]
        for next_gen in generations[1:]:
            assert current < next_gen
            current = next_gen

    @patch(
        "builtins.open", new_callable=mock_open, read_data="machine : x86_64\n"
    )
    def test_get_system_s390x_generation_non_s390x(self):
        """
        Test that get_system_s390x_generation returns None for
        non-s390x architectures.

        NOTE: Non-s390x architectures do not usually have the machine field in
        /proc/cpuinfo, but we simulate it here for testing purposes.
        """
        generation = get_system_s390x_generation()
        assert generation is None

    def test_known_machine_types(self):
        """
        Test that a subset of known s390x generations are correctly identified.
        """
        known_generations = {
            2827: S390xGeneration.ZEC12,
            2964: S390xGeneration.Z13,
            3906: S390xGeneration.Z14,
            8561: S390xGeneration.Z15,
            3931: S390xGeneration.Z16,
            9175: S390xGeneration.Z17,
        }

        for machine_str, expected_generation in known_generations.items():
            with patch(
                "builtins.open",
                new_callable=mock_open,
                read_data=f"machine : {machine_str}\n",
            ):
                generation = get_system_s390x_generation()
                assert generation == expected_generation
