/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <inttypes.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <errno.h>

#include "_xitk.h"

#include "tips.h"
#include "font.h"
#include "_backend.h"

#define TIPS_MS_DELAY 1000
#define TIPS_MS_SHORT 100

typedef enum {
  TIPS_IDLE = 0, /** << nothing to do in normal mode. */
  TIPS_DELAY,    /** << waiting for normal display. */
  TIPS_SHORT,    /** << waiting for quick display. */
  TIPS_SHOW,     /** << display now. */
  TIPS_WAIT,     /** << waiting for hide. */
  TIPS_HIDE,     /** << hide now. */
  TIPS_QUICK,    /** << waiting for switch back to normal mode. */
  TIPS_REP_DELAY,/** << waiting to start click repeat. */
  TIPS_REP_FIRE, /** << send click message to widget. */
  TIPS_REP_NEXT, /** << waiting to fire next repeat. */
  TIPS_QUIT,     /** << exit now. */
  TIPS_LAST
} _tips_state_t;

struct xitk_tips_s {
  /* hold xitk lock. */
  xitk_t              *xitk;
  xitk_window_t       *xwin;
  xitk_image_t        *image;
  xitk_widget_t       *widget;
  int                  wait_time[TIPS_LAST];

  /* may hold xitk lock if you dont wait. */
  int                  num_wakes;
  unsigned int         note;
  _tips_state_t        state;
  struct timespec      never;     /** << {MAX_VALUE,0} */
  struct timespec      want_wake; /** << {0,0} immediately, {0,1} no change. */
  struct timespec      will_wake; /** << {1,0} thread is not currently waiting. */
  struct timespec      last_wake; /** << show, rep_fire. */
  pthread_t            thread;
  pthread_mutex_t      mutex;
  pthread_cond_t       wake;
};

static _tips_state_t _xitk_tips_show (xitk_tips_t *tips) {
  XITK_HV_INIT;

  if (tips->widget && tips->widget->tips_timeout && tips->widget->tips_string && tips->widget->tips_string[0]) {
    xitk_rect_t r = {0, 0, 0, 0};
    unsigned int cfore, cback;
    int disp_w, disp_h;
    int x_margin = 12, y_margin = 6;
    int bottom_gap = 16; /* To avoid mouse cursor overlaying tips on bottom of widget */

    xitk_get_display_size (tips->xitk, &disp_w, &disp_h);
    /* Get parent window position */
    xitk_window_get_window_position (tips->widget->wl->xwin, &r);
    r.x += XITK_HV_H (tips->widget->pos);
    r.y += XITK_HV_V (tips->widget->pos);
    cfore = xitk_get_cfg_num (tips->xitk, XITK_BLACK_COLOR);
    cback = xitk_get_cfg_num (tips->xitk, XITK_FOCUS_COLOR);
    /* Note: disp_w/3 is max. width, returned image with ALIGN_LEFT will be as small as possible */
    tips->image = xitk_image_from_color_string (tips->xitk,
      DEFAULT_FONT_12, disp_w / 3, x_margin >> 1, y_margin >> 1, ALIGN_LEFT, tips->widget->tips_string, cfore, cback);
    if (!tips->image) {
      tips->widget = NULL;
      tips->wait_time[TIPS_WAIT] = 0;
      return TIPS_IDLE;
    }
    r.width = tips->image->width;
    r.height = tips->image->height;
    /* Tips may be extensive help texts that user wants more time to read.
     * We used to implement this by per widget timeout values, but this
     * was never really used like that (only as an enable flag).
     * Instead, take the xitk value as basis for small texts, and hold
     * larger area tips up to 8 times longer. This also saves us the need
     * to broadcast config changes to all widgets. This should be fine since
     * tips are killed when the widget goes away or loses focus. */
    if (tips->widget->tips_timeout == XITK_TIPS_TIMEOUT_AUTO) {
      int tips_timeout = xitk_get_cfg_num (tips->xitk, XITK_TIPS_TIMEOUT);
      tips->wait_time[TIPS_WAIT] = tips_timeout * (r.height - 8) / 12 + tips_timeout * (r.width - 280) / 420;
      if (tips->wait_time[TIPS_WAIT] < tips_timeout)
        tips->wait_time[TIPS_WAIT] = tips_timeout;
      else if (tips->wait_time[TIPS_WAIT] > tips_timeout * 8)
        tips->wait_time[TIPS_WAIT] = tips_timeout * 8;
    } else {
      tips->wait_time[TIPS_WAIT] = tips->widget->tips_timeout;
    }
    /* Create the tips window, horizontally centered from parent widget.
     * If necessary, adjust position to display it fully on screen. */
    xitk_image_draw_rectangle (tips->image, 0, 0, r.width, r.height, cfore);
    r.x -= (r.width - XITK_HV_H (tips->widget->size)) >> 1;
    r.y += XITK_HV_V (tips->widget->size) + bottom_gap;
    if (r.x > disp_w - r.width)
      r.x = disp_w - r.width;
    else if (r.x < 0)
      r.x = 0;
    if (r.y > disp_h - r.height)
      /* 1 px dist to widget prevents odd behavior of mouse pointer when
       * pointer is moved slowly from widget to tips, at least under FVWM
       *                                           v                      */
      r.y -= XITK_HV_V (tips->widget->size) + r.height + bottom_gap + 1;
    /* No further alternative to y-position the tips (just either below or above widget). */
    tips->xwin = xitk_window_create_window_ext (tips->xitk,
      r.x, r.y, r.width, r.height, "tips", NULL, NULL, 1, 0, NULL, tips->image);
    if (!tips->xwin) {
      xitk_image_free_image (&tips->image);
      xitk_lock (tips->xitk, 0);
      tips->widget = NULL;
      tips->wait_time[TIPS_WAIT] = 0;
      return TIPS_QUICK;
    }
    xitk_window_set_role (tips->xwin, XITK_WR_SUBMENU);
    xitk_window_flags (tips->xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
    return TIPS_WAIT;
  } else {
    tips->widget = NULL;
    tips->wait_time[TIPS_WAIT] = 0;
    return TIPS_QUICK;
  }
}

static void _xitk_tips_hide (xitk_tips_t *tips) {
  if (tips->xwin) {
    xitk_window_flags (tips->xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, 0);
    xitk_window_destroy_window (tips->xwin);
    tips->xwin = NULL;
    xitk_image_free_image (&tips->image);
  }
}

static int _before (struct timespec *ts1, struct timespec *ts2) {
  if (ts1->tv_sec < ts2->tv_sec)
    return 1;
  if ((ts1->tv_sec == ts2->tv_sec) && (ts1->tv_nsec < ts2->tv_nsec))
    return 1;
  return 0;
}
  
static void _compute_interval (struct timespec *ts, int millisecs) {
  if (millisecs >= 0) {
    xitk_gettime_ts (ts);
  } else {
    millisecs = -millisecs;
  }
  ts->tv_sec += millisecs / 1000;
  ts->tv_nsec += (millisecs % 1000) * 1000000;
  if (ts->tv_nsec >= 1000000000) {
    ts->tv_nsec -= 1000000000;
    ts->tv_sec += 1;
  }
}

static void *_tips_loop_thread (void *data) {
  xitk_tips_t *tips = data;
  _tips_state_t state = TIPS_IDLE;

  pthread_mutex_lock (&tips->mutex);

  while (tips->state != TIPS_QUIT) {
    unsigned int wait;

    state = tips->state;
    wait = tips->wait_time[state];
    if (tips->note) {
      if (tips->want_wake.tv_sec || !tips->want_wake.tv_sec)
        tips->will_wake = tips->want_wake; /* new wake time from caller. */
    } else if (wait == ~0u) {
      tips->will_wake = tips->never; /* idle sleep until explicit wake. */
    } else if (wait) {
      tips->will_wake = tips->last_wake;
      _compute_interval (&tips->will_wake, wait); /* self timing. */
    } else {
      _compute_interval (&tips->last_wake, 0);
      tips->will_wake.tv_sec = 0; /* go on immediately. */
    }
    if (tips->will_wake.tv_sec == tips->never.tv_sec) {
      pthread_cond_wait (&tips->wake, &tips->mutex);
      tips->num_wakes++;
    } else if (tips->will_wake.tv_sec) {
      pthread_cond_timedwait (&tips->wake, &tips->mutex, &tips->will_wake);
      tips->num_wakes++;
    }
    state = tips->state;
    if (tips->note) {
      tips->note = 0;
      continue;
    }

    tips->will_wake.tv_sec = 1;
    pthread_mutex_unlock (&tips->mutex);

    switch (state) {

      case TIPS_DELAY:
      case TIPS_SHORT:
        state = TIPS_SHOW;
        break;

      case TIPS_SHOW:
        if (!xitk_lock (tips->xitk, 302)) {
          state = _xitk_tips_show (tips);
          xitk_lock (tips->xitk, 0);
        }
        break;

      case TIPS_WAIT:
        state = TIPS_HIDE;
        break;

      case TIPS_HIDE:
        if (!xitk_lock (tips->xitk, 302)) {
          _xitk_tips_hide (tips);
          tips->widget = NULL;
          tips->wait_time[TIPS_WAIT] = 0;
          xitk_lock (tips->xitk, 0);
          state = TIPS_QUICK;
        }
        break;

      case TIPS_QUICK:
        if (!xitk_lock (tips->xitk, 302)) {
          tips->widget = NULL;
          tips->wait_time[TIPS_WAIT] = 0;
          xitk_lock (tips->xitk, 0);
          state = TIPS_IDLE;
        }
        break;

      case TIPS_REP_FIRE:
        state = TIPS_REP_NEXT;
        if (!xitk_lock (tips->xitk, 302)) {
          if (tips->widget && (tips->widget->state & XITK_WIDGET_STATE_ENABLE)) {
            widget_event_t event = {
              .type = WIDGET_EVENT_CLICK,
              .x = tips->widget->wl->mouse.x,
              .y = tips->widget->wl->mouse.y,
              .pressed = 1,
              .button = 1,
              .modifier = tips->widget->wl->qual
            };
            tips->widget->event (tips->widget, &event);
            xitk_lock (tips->xitk, 0);
            break;
          }
          tips->widget = NULL;
          xitk_lock (tips->xitk, 0);
          state = TIPS_IDLE;
        }
        break;

      case TIPS_REP_NEXT:
      case TIPS_REP_DELAY:
        state = TIPS_REP_FIRE;
        break;

      default: ;
    }

    pthread_mutex_lock (&tips->mutex);
    if (!tips->note)
      tips->state = state;
  }

  pthread_mutex_unlock (&tips->mutex);
  return NULL;
}

/*
 *
 */
xitk_tips_t *xitk_tips_new (xitk_t *xitk) {
  xitk_tips_t *tips;

  tips = xitk_xmalloc (sizeof (*tips));
  if (!tips)
    return NULL;

  tips->xitk = xitk;
  if (xitk_init_NULL ()) {
    tips->xwin   = NULL;
    tips->image  = NULL;
    tips->widget = NULL;
  }
#if 0
  tips->wait_time[TIPS_SHOW] = 0;
  tips->wait_time[TIPS_WAIT] = 0;
  tips->wait_time[TIPS_HIDE] = 0;
  tips->wait_time[TIPS_REP_FIRE] = 0;
  tips->wait_time[TIPS_QUIT] = 0;
  tips->num_wakes  = 0;
  tips->note       = 0;
  tips->last_wake.tv_sec = 0;
  tips->last_wake.tv_nsec = 0;
  tips->will_wake.tv_nsec = 0;
  tips->want_wake.tv_sec = 0;
#endif
  tips->wait_time[TIPS_IDLE] = ~0u;
  tips->wait_time[TIPS_DELAY] = TIPS_MS_DELAY;
  tips->wait_time[TIPS_SHORT] = TIPS_MS_SHORT;
  tips->wait_time[TIPS_QUICK] = TIPS_MS_DELAY;
  tips->wait_time[TIPS_REP_DELAY] = 500;
  tips->wait_time[TIPS_REP_NEXT] = -100; /** << minus means smooth timing. */

  tips->state      = TIPS_IDLE;
  tips->never.tv_sec = ((uint64_t)1 << (8 * sizeof (tips->never.tv_sec) - 1)) - 1;
  tips->will_wake.tv_sec = 1;
  tips->want_wake.tv_nsec = 1;
  pthread_mutex_init (&tips->mutex, NULL);
  pthread_cond_init (&tips->wake, NULL);

  {
    pthread_attr_t       pth_attrs;
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && (_POSIX_THREAD_PRIORITY_SCHEDULING > 0)
    struct sched_param   pth_params;
#endif
    int r;
    pthread_attr_init (&pth_attrs);
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && (_POSIX_THREAD_PRIORITY_SCHEDULING > 0)
    pthread_attr_getschedparam (&pth_attrs, &pth_params);
    pth_params.sched_priority = sched_get_priority_min (SCHED_OTHER);
    pthread_attr_setschedparam (&pth_attrs, &pth_params);
#endif
    r = pthread_create (&tips->thread, &pth_attrs, _tips_loop_thread, tips);
    pthread_attr_destroy (&pth_attrs);
    if (!r)
      return tips;
  }

  pthread_cond_destroy (&tips->wake);
  pthread_mutex_destroy (&tips->mutex);
  free (tips);
  return NULL;
}

void xitk_tips_delete (xitk_tips_t **ptips) {
  xitk_tips_t *tips = *ptips;

  if (!tips)
    return;

  *ptips = NULL;

  pthread_mutex_lock (&tips->mutex);
  tips->note++;
  tips->state = TIPS_QUIT;
  pthread_cond_signal (&tips->wake);
  pthread_mutex_unlock (&tips->mutex);
  pthread_join (tips->thread, NULL);

  _xitk_tips_hide (tips);
  pthread_cond_destroy (&tips->wake);
  pthread_mutex_destroy (&tips->mutex);

  if (tips->xitk->verbosity >= 2)
    printf ("xitk.tips.wakeups (%d).\n", tips->num_wakes);
  free (tips);
}

int xitk_tips_main (xitk_tips_t *tips, xitk_widget_t *w, int mode) {
  xitk_widget_t *v = NULL;

  /* NOTE: this is called with xitk lock held. */
  if (!tips)
    return 1;
  /* Don't show when window invisible. This call may occur directly after iconifying window. */
  if (w && !(xitk_window_flags (w->wl->xwin, 0, 0) & XITK_WINF_VISIBLE))
    return 0;

  do {
    if (!w)
      break;
    if (mode != 1) {
      mode = 0;
      if (!w->tips_timeout || !w->tips_string)
        break;
      if (!w->tips_string[0])
        break;
    }
    v = w;
  } while (0);

  if ((tips->state != TIPS_QUIT) && (v != tips->widget)) {
    _xitk_tips_hide (tips);
    tips->widget = NULL;
    tips->wait_time[TIPS_WAIT] = 0;

    if (v) {
      /* set new widget. */
      static const uint8_t next[TIPS_LAST][2] = {
        [TIPS_IDLE] = {TIPS_DELAY, TIPS_REP_DELAY},
        [TIPS_DELAY] = {TIPS_DELAY, TIPS_REP_DELAY},
        [TIPS_SHORT] = {TIPS_SHORT, TIPS_REP_DELAY},
        [TIPS_SHOW] = {TIPS_SHORT, TIPS_REP_DELAY},
        [TIPS_WAIT] = {TIPS_SHORT, TIPS_REP_DELAY},
        [TIPS_HIDE] = {TIPS_SHORT, TIPS_REP_DELAY},
        [TIPS_QUICK] = {TIPS_SHORT, TIPS_REP_DELAY},
        [TIPS_REP_DELAY] = {TIPS_DELAY, TIPS_REP_DELAY},
        [TIPS_REP_FIRE] = {TIPS_DELAY, TIPS_REP_DELAY},
        [TIPS_REP_NEXT] = {TIPS_DELAY, TIPS_REP_DELAY},
        [TIPS_QUIT] = {TIPS_QUIT, TIPS_QUIT}
      };
      tips->widget = v;
      pthread_mutex_lock (&tips->mutex);
      tips->state = next[tips->state][mode];
      _compute_interval (&tips->want_wake, tips->wait_time[tips->state]);
      tips->note++;
      if (_before (&tips->want_wake, &tips->will_wake))
        pthread_cond_signal (&tips->wake);
      pthread_mutex_unlock (&tips->mutex);
    } else {
      /* hide tips. */
      static const uint8_t next[TIPS_LAST] = {
        [TIPS_IDLE] = TIPS_IDLE,
        [TIPS_DELAY] = TIPS_IDLE,
        [TIPS_SHORT] = TIPS_QUICK,
        [TIPS_SHOW] = TIPS_QUICK,
        [TIPS_WAIT] = TIPS_QUICK,
        [TIPS_HIDE] = TIPS_QUICK,
        [TIPS_QUICK] = TIPS_QUICK,
        [TIPS_REP_DELAY] = TIPS_IDLE,
        [TIPS_REP_FIRE] = TIPS_IDLE,
        [TIPS_REP_NEXT] = TIPS_IDLE,
        [TIPS_QUIT] = TIPS_QUIT
      };
      pthread_mutex_lock (&tips->mutex);
      tips->state = next[tips->state];
      tips->note++;
      if (tips->state == TIPS_QUICK) {
        _compute_interval (&tips->want_wake, TIPS_MS_DELAY);
        if (_before (&tips->want_wake, &tips->will_wake))
          pthread_cond_signal (&tips->wake);
      } else {
        tips->want_wake.tv_sec = 0, tips->want_wake.tv_nsec = 1;
      }
      pthread_mutex_unlock (&tips->mutex);
    }
  }
  return 1;
}
