"""Utilities related to LDAP."""

import ldap3

def ldap_entry_dn(entry):
    try:
        return entry.entry_dn
    except ldap3.core.exceptions.LDAPExceptionError:
        return entry.entry_get_dn()

class LDAPValidationError(Exception):
    pass

# Cf http://bugzilla.nordugrid.org/show_bug.cgi?id=2693
_ignore_must = set([
    'Mds-validfrom',
    'Mds-validto',
    'Mds-Vo-Op-name',
    'Mds-Service-protocol',
])

def _transitive_attributes(server, entry_schema, must_contain, may_contain):
    must_contain.update(entry_schema.must_contain)
    may_contain.update(entry_schema.may_contain)
    for superior in entry_schema.superior or []:
        _transitive_attributes(server,
                server.schema.object_classes[superior],
                must_contain, may_contain)

class LDAPEntry(object):
    """This is a read-only representation of an LDAP entry with attribute
    values decoded according to their schema."""

    structural_objectclass = None
    strip_attribute_prefixes = []
    lowercase_all_attributes = False

    def __init__(self, server, entry):
        if not self.structural_objectclass in entry.objectClass:
            raise ValueError(
                    'The LDAP entry at %s is not of structural object class %s.'
                    % (ldap_entry_dn(entry), self.structural_objectclass))

        entry_schema = server.schema.object_classes[self.structural_objectclass]
        must_contain, may_contain = set(), set()
        _transitive_attributes(server, entry_schema, must_contain, may_contain)

        missing = []
        for attr in must_contain.union(may_contain):
            name = attr
            for prefix in self.strip_attribute_prefixes:
                if name.startswith(prefix):
                    name = name[len(prefix):]
            if self.lowercase_all_attributes:
                name = name.lower()
            name = name.replace('-', '_')
            try:
                if server.schema.attribute_types[attr].single_value:
                    value = entry[attr].value
                else:
                    value = entry[attr].values
            except ldap3.core.exceptions.LDAPExceptionError:
                 # ldap3.core.exceptions.LDAPKeyError for older versions
                 # ldap3.core.exceptions.LDAPCursorError for newer versions
                if server.schema.attribute_types[attr].single_value:
                    value = None
                else:
                    value = []
                if attr in must_contain and not attr in _ignore_must:
                    missing.append(attr)
            setattr(self, name, value)

        if missing:
            raise LDAPValidationError(
                    '%s lacks required attribute(s) %s.'
                    % (ldap_entry_dn(entry), ', '.join(missing)))

def canonical_dn(dn):
    return ldap3.utils.dn.safe_dn(dn)

def is_proper_subdn(subdn, dn):
    subdn = canonical_dn(subdn)
    dn = canonical_dn(dn)
    return subdn != dn and subdn.endswith(dn)

def is_immediate_subdn(subdn, dn):
    subdn_comps = ldap3.utils.dn.parse_dn(subdn)
    dn_comps = ldap3.utils.dn.parse_dn(dn)
    return len(subdn_comps) > 0 and subdn_comps[1:] == dn_comps
