/*
 * Copyright (c) 2012, 2014, JANET(UK)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of JANET(UK) nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <openssl/dh.h>
#include <trust_router/tr_dh.h>
#include <openssl/bn.h>
#include <openssl/sha.h>
#include <talloc.h>
#include <assert.h>
#include <tid_internal.h>
#include <tr_debug.h>
#include <ssl-compat.h>

unsigned char tr_2048_dhprime[2048/8] = {
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  0xC9, 0x0F, 0xDA, 0xA2, 0x21, 0x68, 0xC2, 0x34,
  0xC4, 0xC6, 0x62, 0x8B, 0x80, 0xDC, 0x1C, 0xD1,
  0x29, 0x02, 0x4E, 0x08, 0x8A, 0x67, 0xCC, 0x74,
  0x02, 0x0B, 0xBE, 0xA6, 0x3B, 0x13, 0x9B, 0x22,
  0x51, 0x4A, 0x08, 0x79, 0x8E, 0x34, 0x04, 0xDD,
  0xEF, 0x95, 0x19, 0xB3, 0xCD, 0x3A, 0x43, 0x1B,
  0x30, 0x2B, 0x0A, 0x6D, 0xF2, 0x5F, 0x14, 0x37,
  0x4F, 0xE1, 0x35, 0x6D, 0x6D, 0x51, 0xC2, 0x45,
  0xE4, 0x85, 0xB5, 0x76, 0x62, 0x5E, 0x7E, 0xC6,
  0xF4, 0x4C, 0x42, 0xE9, 0xA6, 0x37, 0xED, 0x6B,
  0x0B, 0xFF, 0x5C, 0xB6, 0xF4, 0x06, 0xB7, 0xED,
  0xEE, 0x38, 0x6B, 0xFB, 0x5A, 0x89, 0x9F, 0xA5,
  0xAE, 0x9F, 0x24, 0x11, 0x7C, 0x4B, 0x1F, 0xE6,
  0x49, 0x28, 0x66, 0x51, 0xEC, 0xE4, 0x5B, 0x3D,
  0xC2, 0x00, 0x7C, 0xB8, 0xA1, 0x63, 0xBF, 0x05,
  0x98, 0xDA, 0x48, 0x36, 0x1C, 0x55, 0xD3, 0x9A,
  0x69, 0x16, 0x3F, 0xA8, 0xFD, 0x24, 0xCF, 0x5F,
  0x83, 0x65, 0x5D, 0x23, 0xDC, 0xA3, 0xAD, 0x96,
  0x1C, 0x62, 0xF3, 0x56, 0x20, 0x85, 0x52, 0xBB,
  0x9E, 0xD5, 0x29, 0x07, 0x70, 0x96, 0x96, 0x6D,
  0x67, 0x0C, 0x35, 0x4E, 0x4A, 0xBC, 0x98, 0x04,
  0xF1, 0x74, 0x6C, 0x08, 0xCA, 0x18, 0x21, 0x7C,
  0x32, 0x90, 0x5E, 0x46, 0x2E, 0x36, 0xCE, 0x3B,
  0xE3, 0x9E, 0x77, 0x2C, 0x18, 0x0E, 0x86, 0x03,
  0x9B, 0x27, 0x83, 0xA2, 0xEC, 0x07, 0xA2, 0x8F,
  0xB5, 0xC5, 0x5D, 0xF0, 0x6F, 0x4C, 0x52, 0xC9,
  0xDE, 0x2B, 0xCB, 0xF6, 0x95, 0x58, 0x17, 0x18,
  0x39, 0x95, 0x49, 0x7C, 0xEA, 0x95, 0x6A, 0xE5,
  0x15, 0xD2, 0x26, 0x18, 0x98, 0xFA, 0x05, 0x10,
  0x15, 0x72, 0x8E, 0x5A, 0x8A, 0xAC, 0xAA, 0x68,
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

DH *tr_dh_new(void)
{
  return DH_new();
}

DH *tr_create_dh_params(unsigned char *priv_key,
			size_t keylen) {

  DH *dh = NULL;
  int dh_err = 0;
  BIGNUM *p = NULL, *q = NULL, *g = NULL;

  if (NULL == (dh = DH_new()))
    return NULL;

  if ((NULL == (g = BN_new())) ||
      (NULL == (p = BN_new())) ||
      (NULL == (q = BN_new()))) {
    DH_free(dh);
    BN_free(g);
    BN_free(p);
    BN_free(q);
    return NULL;
  }

  BN_set_word(g, 2);
  BN_bin2bn(tr_2048_dhprime, sizeof(tr_2048_dhprime), p);
  BN_rshift1(q, p);

  DH_set0_pqg(dh, p, q, g);
  DH_generate_key(dh);    /* generates the public key */

  if ((priv_key) && (keylen > 0))
    DH_set0_key(dh, NULL, BN_bin2bn(priv_key, keylen, NULL));

  DH_check(dh, &dh_err);
  if (0 != dh_err) {
    tr_warning("Warning: dh_check failed with %d", dh_err);
    if (dh_err & DH_CHECK_P_NOT_PRIME)
      tr_warning(": p value is not prime");
    else if (dh_err & DH_CHECK_P_NOT_SAFE_PRIME)
      tr_warning(": p value is not a safe prime");
    else if (dh_err & DH_UNABLE_TO_CHECK_GENERATOR)
      tr_warning(": unable to check the generator value");
    else if (dh_err & DH_NOT_SUITABLE_GENERATOR)
      tr_warning(": the g value is not a generator");
    else
      tr_warning("unhandled error %i", dh_err);
  }

  return(dh);
}

DH *tr_create_matching_dh (unsigned char *priv_key,
			   size_t keylen,
			   DH *in_dh) {
  DH *dh = NULL;
  int dh_err = 0;
  const BIGNUM *in_p = NULL, *in_g = NULL;

  if (!in_dh)
    return NULL;

  if (NULL == (dh = DH_new())) {
    tr_crit("tr_create_matching_dh: unable to allocate new DH structure.");
    return NULL;
  }

  DH_get0_pqg(in_dh, &in_p, NULL, &in_g);

  if (0 == DH_set0_pqg(dh, BN_dup(in_p), NULL, BN_dup(in_g))) {
    DH_free(dh);
    tr_debug("tr_create_matching_dh: Invalid dh parameter values, can't be duped.");
    return NULL;
  }

  /* TBD -- share code with previous function */
  DH_generate_key(dh);    /* generates the public key */

  if ((priv_key) && (keylen > 0))
    DH_set0_key(dh, NULL, BN_bin2bn(priv_key, keylen, NULL));

  DH_check(dh, &dh_err);
  if (0 != dh_err) {
    tr_warning("Warning: dh_check failed with %d", dh_err);
    if (dh_err & DH_CHECK_P_NOT_PRIME)
      tr_warning(": p value is not prime");
    else if (dh_err & DH_CHECK_P_NOT_SAFE_PRIME)
      tr_warning(": p value is not a safe prime");
    else if (dh_err & DH_UNABLE_TO_CHECK_GENERATOR)
      tr_warning(": unable to check the generator value");
    else if (dh_err & DH_NOT_SUITABLE_GENERATOR)
      tr_warning(": the g value is not a generator");
    else
      tr_warning("unhandled error %i", dh_err);
  }

  return(dh);
}

void tr_destroy_dh_params(DH *dh) {

  if (dh) {
    DH_free(dh);
  }
}

DH *tr_dh_dup(DH *in)
{
  DH *out = DH_new();
  const BIGNUM *in_g = NULL, *in_p = NULL, *in_q = NULL, *in_pubkey = NULL, *in_privkey = NULL;
  BIGNUM *out_g = NULL, *out_p = NULL, *out_q = NULL, *out_pubkey = NULL, *out_privkey = NULL;

  DH_get0_pqg(in, &in_p, &in_q, &in_g);
  DH_get0_key(in, &in_pubkey, &in_privkey);

  if (out == NULL)
    return NULL;

  if (in_g==NULL)
    out_g=NULL;
  else {
    out_g=BN_dup(in_g);
    if (out_g==NULL) {
      DH_free(out);
      return NULL;
    }
  }

  if (in_p==NULL)
    out_p=NULL;
  else {
    out_p=BN_dup(in_p);
    if (out_p==NULL) {
      DH_free(out);
      return NULL;
    }
  }

  if (in_q==NULL)
    out_q=NULL;
  else {
    out_q=BN_dup(in_q);
    if (out_q==NULL) {
      DH_free(out);
      return NULL;
    }
  }

  if (in_privkey==NULL)
    out_privkey=NULL;
  else {
    out_privkey=BN_dup(in_privkey);
    if (out_privkey==NULL) {
      DH_free(out);
      return NULL;
    }
  }

  if (in_pubkey==NULL)
    out_pubkey=NULL;
  else {
    out_pubkey=BN_dup(in_pubkey);
    if (out_pubkey==NULL) {
      DH_free(out);
      return NULL;
    }
  }

  if (0 == DH_set0_pqg(out, out_p, out_q, out_g)) {
    DH_free(out);
    return NULL;
  }

  if (0 == DH_set0_key(out, out_pubkey, out_privkey)) {
    DH_free(out);
    return NULL;
  }

  return out;
}

int tr_compute_dh_key(unsigned char **pbuf,
		      const BIGNUM *pub_key,
		      DH *priv_dh) {
  size_t buflen;
  unsigned char *buf = NULL;;
  int rc = 0;

  if ((!pbuf) ||
      (!pub_key) ||
      (!priv_dh)) {
    tr_debug("tr_compute_dh_key: Invalid parameters.");
    return(-1);
  }
  *pbuf = NULL;
  buflen = DH_size(priv_dh);
  buf = malloc(buflen);
  if (buf == NULL) {
    tr_crit("tr_compute_dh_key: out of memory");
    return -1;
  }


  rc = DH_compute_key(buf, pub_key, priv_dh);
  if (0 <= rc) {
    *pbuf = buf;
  }else {
    free(buf);
  }
  return rc;
}




int tr_dh_pub_hash(TID_REQ *request,
		   unsigned char **out_digest,
		   size_t *out_len)
{
  const BIGNUM *pub = NULL;
  unsigned char *bn_bytes, *digest;

  DH_get0_key(request->tidc_dh, &pub, NULL);
  bn_bytes = talloc_zero_size(request, BN_num_bytes(pub));
  digest = talloc_zero_size(request, SHA_DIGEST_LENGTH + 1);

  assert(bn_bytes && digest);
  BN_bn2bin(pub, bn_bytes);
  SHA1(bn_bytes, BN_num_bytes(pub), digest);
  *out_digest = digest;
  *out_len = SHA_DIGEST_LENGTH;

  talloc_free(bn_bytes);
  return 0;
}

void tr_dh_free(unsigned char *dh_buf)
{
  free(dh_buf);
}

void tr_dh_destroy(DH *dh)
{
  DH_free(dh);
}
