/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.15
import QtTest 1.0
import Lomiri.SelfTest 0.1 as UT
import Lomiri.Settings.Menus 0.1 as Menus
import QMenuModel 1.0
import Utils 0.1 as Utils
import "../../../../qml/Panel/Indicators"


Item {
    id: testView
    width: units.gu(40)
    height: units.gu(70)

    IndicatorMenuItemFactory {
        id: factory
        menuModel: AyatanaMenuModel {}
    }

    Loader {
        id: loader
        property int modelIndex: 0
        property var data

        anchors {
            left: parent.left
            right: parent.right
        }

        onLoaded: {
            if (item.hasOwnProperty("menuData")) {
                item.menuData = data;
            }
        }
    }

    UT.LomiriTestCase {
        name: "IndicatorMenuItemFactory"
        when: windowShown

        property QtObject menuData: QtObject {
            property string label: "root"
            property bool sensitive: true
            property bool isSeparator: false
            property string icon: ""
            property string type: ""
            property var ext: undefined
            property string action: ""
            property var actionState: undefined
            property bool isCheck: false
            property bool isRadio: false
            property bool isToggled: false
        }

        function init() {
            menuData.label = "";
            menuData.sensitive = true;
            menuData.isSeparator = false;
            menuData.icon = "";
            menuData.type = "";
            menuData.ext = undefined;
            menuData.action = "";
            menuData.actionState = undefined;
            menuData.isCheck = false;
            menuData.isRadio = false;
            menuData.isToggled = false;
        }

        function cleanup() {
            loader.data = undefined;
            loader.sourceComponent = undefined;

            verify(loader.item == null);
        }

        function loadData(menuData) {
            loader.data = menuData;
            loader.sourceComponent = factory.load(menuData);
            tryVerify(function() { return loader.item; });
        }

        function test_createTypes_data() {
            return [
                { tag: 'volume', type: "lomiri.widgets.systemsettings.tablet.volumecontrol", objectName: "sliderMenu" },
                { tag: 'switch1', type: "lomiri.widgets.systemsettings.tablet.switch", objectName: "switchMenu" },

                { tag: 'button', type: "com.canonical.indicator.button", objectName: "buttonMenu" },
                { tag: 'button-ayatana', type: "org.ayatana.indicator.button", objectName: "buttonMenu" },
                { tag: 'separator', type: "com.canonical.indicator.div", objectName: "separatorMenu" },
                { tag: 'separator-ayatana', type: "org.ayatana.indicator.div", objectName: "separatorMenu" },
                { tag: 'section', type: "com.canonical.indicator.section", objectName: "sectionMenu" },
                { tag: 'section-ayatana', type: "org.ayatana.indicator.section", objectName: "sectionMenu" },
                { tag: 'progress', type: "com.canonical.indicator.progress", objectName: "progressMenu" },
                { tag: 'progress-ayatana', type: "org.ayatana.indicator.progress", objectName: "progressMenu" },
                { tag: 'slider1', type: "com.canonical.indicator.slider", objectName: "sliderMenu" },
                { tag: 'slider1-ayatana', type: "org.ayatana.indicator.slider", objectName: "sliderMenu" },
                { tag: 'switch2', type: "com.canonical.indicator.switch", objectName: "switchMenu" },
                { tag: 'switch2-ayatana', type: "org.ayatana.indicator.switch", objectName: "switchMenu" },
                { tag: 'alarm', type: "com.canonical.indicator.alarm", objectName: "alarmMenu" },
                { tag: 'alarm-ayatana', type: "org.ayatana.indicator.alarm", objectName: "alarmMenu" },
                { tag: 'appointment', type: "com.canonical.indicator.appointment", objectName: "appointmentMenu" },
                { tag: 'appointment-ayatana', type: "org.ayatana.indicator.appointment", objectName: "appointmentMenu" },
                { tag: 'transfer', type: "com.canonical.indicator.transfer", objectName: "transferMenu" },
                { tag: 'transfer-ayatana', type: "org.ayatana.indicator.transfer", objectName: "transferMenu" },
                { tag: 'buttonSection', type: "com.canonical.indicator.button-section", objectName: "buttonSectionMenu" },
                { tag: 'buttonSection-ayatana', type: "org.ayatana.indicator.button-section", objectName: "buttonSectionMenu" },

                { tag: 'messageItem', type: "com.canonical.indicator.messages.messageitem", objectName: "messageItem" },
                { tag: 'messageItem-ayatana', type: "org.ayatana.indicator.messages.messageitem", objectName: "messageItem" },
                { tag: 'sourceItem', type: "com.canonical.indicator.messages.sourceitem", objectName: "groupedMessage" },
                { tag: 'sourceItem-ayatana', type: "org.ayatana.indicator.messages.sourceitem", objectName: "groupedMessage" },

                { tag: 'slider2', type: "com.canonical.lomiri.slider", objectName: "sliderMenu" },
                { tag: 'slider2-ayatana', type: "org.ayatana.indicator.slider", objectName: "sliderMenu" },
                { tag: 'switch3', type: "com.canonical.lomiri.switch", objectName: "switchMenu" },
                { tag: 'switch3-ayatana', type: "org.ayatana.indicator.switch", objectName: "switchMenu" },

                { tag: 'mediaplayer', type: "com.canonical.lomiri.media-player", objectName: "mediaPayerMenu" },
                { tag: 'mediaplayer-ayatana', type: "org.ayatana.indicator.media-player", objectName: "mediaPayerMenu" },
                { tag: 'playbackitem', type: "com.canonical.lomiri.playback-item", objectName: "playbackItemMenu" },
                { tag: 'playbackitem-ayatana', type: "org.ayatana.indicator.playback-item", objectName: "playbackItemMenu" },

                { tag: 'wifisection', type: "lomiri.widgets.systemsettings.tablet.wifisection", objectName: "wifiSection" },
                { tag: 'accesspoint', type: "lomiri.widgets.systemsettings.tablet.accesspoint", objectName: "accessPoint" },
                { tag: 'modeminfoitem', type: "com.lomiri.indicator.network.modeminfoitem", objectName: "modemInfoItem" },
                { tag: 'modeminfoitem-ayatana', type: "org.ayatana.indicator.network.modeminfoitem", objectName: "modemInfoItem" },

                { tag: 'calendar', type: "com.canonical.indicator.calendar", objectName: "calendarMenu" },
                { tag: 'calendar-ayatana', type: "org.ayatana.indicator.calendar", objectName: "calendarMenu" },

                { tag: 'unknown', type: "", objectName: "standardMenu"}
            ];
        }

        function test_createTypes(data) {
            menuData.type = data.type;

            loadData(menuData);
            compare(loader.item.objectName, data.objectName, "Created object name does not match intended object (" + loader.item.objectName + " != " + data.objectName + ")");
        }

        function test_create_checkable() {
            var tmpData = menuData;
            tmpData.isCheck = true;

            loadData(tmpData);
            compare(loader.item.objectName, "checkableMenu", "Should have created a checkable menu");
        }

        function test_create_radio() {
            menuData.isRadio = true;

            loadData(menuData);
            compare(loader.item.objectName, "radioMenu", "Should have created a radio menu");
        }

        function test_create_separator() {
            menuData.isSeparator = true;

            loadData(menuData);
            compare(loader.item.objectName, "separatorMenu", "Should have created a separator menu");
        }

        function test_create_sliderMenu_data() {
            return [
                {tag: "disabled", enabled: false, minValue: 0, maxValue: 100, value1: 10.5, value2: 22 },
                {tag: "enabled", enabled: true, minValue: 0, maxValue: 100, value1: 100, value2: 50 },
            ];
        }

        function test_create_sliderMenu(data) {
            menuData.type = "com.canonical.indicator.slider"
            menuData.label = data.tag;
            menuData.sensitive = data.enabled;
            menuData.ext = {
                'minIcon': "file:///testMinIcon",
                'maxIcon': "file:///testMaxIcon",
                'minValue': data.minValue,
                'maxValue': data.maxValue
            };
            menuData.actionState = data.value1;

            loadData(menuData);
            compare(loader.item.objectName, "sliderMenu", "Should have created a slider menu");

            compare(loader.item.text, data.tag, "Label does not match data");
            compare(loader.item.minIcon, "file:///testMinIcon", "MinIcon does not match data");
            compare(loader.item.maxIcon, "file:///testMaxIcon", "MaxIcon does not match data");
            compare(loader.item.minimumValue, data.minValue, "MinValue does not match data");
            compare(loader.item.maximumValue, data.maxValue, "MaxValue does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
            compare(loader.item.value, data.value1, "Value does not match data");

            menuData.actionState = data.value2;
            compare(loader.item.value, data.value2, "Value does not match new data");
        }

        function test_create_sliderMenu_lp1283191_data() {
            return [
                {tag: "disabled", enabled: false, minValue: 0, maxValue: 100, value1: 10.5, value2: 22, manualValue: 0 },
                {tag: "enabled", enabled: true, minValue: 0, maxValue: 100, value1: 100, value2: 50, manualValue: 30 },
                {tag: "test-zero", enabled: true, minValue: 0, maxValue: 100, value1: 10, value2: 0, manualValue: 20 },
            ];
        }

        function test_create_sliderMenu_lp1283191(data) {
            menuData.type = "com.canonical.indicator.slider"
            menuData.label = data.tag;
            menuData.sensitive = data.enabled;
            menuData.ext = {
                'minIcon': "file:///testMinIcon",
                'maxIcon': "file:///testMaxIcon",
                'minValue': data.minValue,
                'maxValue': data.maxValue
            };
            menuData.actionState = data.value1;

            loadData(menuData);
            compare(loader.item.value, data.value1, "Value does not match data");

            var slider = findChild(loader.item, "slider");
            verify(slider !== null);

            slider.value = data.manualValue;
            compare(loader.item.value, data.manualValue, "Slider value does not match manual set value");

            menuData.actionState = data.value2;
            tryCompare(loader.item, "value", data.value2, 10000, "Value does not match new data");

            menuData.actionState = undefined;
            compare(loader.item.value, data.value2, "Undefined state should not update slider value");
        }

        function test_create_buttonMenu_data() {
            return [
                {label: "testLabel1", enabled: false },
                {label: "testLabel2", enabled: true },
            ];
        }

        function test_create_buttonMenu(data) {
            menuData.type = "com.canonical.indicator.button"
            menuData.label = data.label;
            menuData.sensitive = data.enabled;

            loadData(menuData);
            compare(loader.item.objectName, "buttonMenu", "Should have created a button menu");

            compare(loader.item.buttonText, data.label, "Label does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_sectionMenu_data() {
            return [
                {label: "testLabel1" },
                {label: "testLabel2" },
            ];
        }

        function test_create_sectionMenu(data) {
            menuData.type = "com.canonical.indicator.section"
            menuData.label = data.label;

            loadData(menuData);
            compare(loader.item.objectName, "sectionMenu", "Should have created a section menu");

            compare(loader.item.text, data.label, "Label does not match data");
        }

        function test_create_progressMenu_data() {
            return [
                {label: "testLabel1", value: 10 },
                {label: "testLabel2", value: 55 },
            ];
        }

        function test_create_progressMenu(data) {
            menuData.type = "com.canonical.indicator.progress"
            menuData.label = data.label;
            menuData.actionState = data.value;

            loadData(menuData);
            compare(loader.item.objectName, "progressMenu", "Should have created a progress menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.value, data.value, "Value does not match data");
        }

        function test_create_levelMenu_data() {
            return [
                {label: "testLabel1", value: 10 },
                {label: "testLabel2", value: 55 },
            ];
        }

        function test_create_levelMenu(data) {
            menuData.type = "org.ayatana.indicator.level"
            menuData.label = data.label;
            menuData.ext = {
                xAyatanaLevel: data.value
            };

            loadData(menuData);
            compare(loader.item.objectName, "levelMenu", "Should have created a level menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.value, data.value, "Value does not match data");
        }

        function test_create_standardMenu_data() {
            return [
                {label: "testLabel1", enabled: true, icon: "file:///testIcon1" },
                {label: "testLabel2", enabled: false, icon: "file:///testIcon2"},
            ];
        }

        function test_create_standardMenu(data) {
            menuData.type = ""
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.icon = data.icon;

            loadData(menuData);
            compare(loader.item.objectName, "standardMenu", "Should have created a standard menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.iconSource, data.icon, "MaxIcon does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_checkableMenu_data() {
            return [
                {label: "testLabel1", enabled: true, checked: false },
                {label: "testLabel2", enabled: false, checked: true },
            ];
        }

        function test_create_checkableMenu(data) {
            menuData.type = "";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.isCheck = true;
            menuData.isToggled = data.checked;

            loadData(menuData);
            compare(loader.item.objectName, "checkableMenu", "Should have created a checkable menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.checked, data.checked, "Checked does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_radioMenu_data() {
            return [
                {label: "testLabel1", enabled: true, checked: false },
                {label: "testLabel2", enabled: false, checked: true },
            ];
        }

        function test_create_radioMenu(data) {
            menuData.type = "";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.isRadio = true;
            menuData.isToggled = data.checked;

            loadData(menuData);
            compare(loader.item.objectName, "radioMenu", "Should have created a radio menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.checked, data.checked, "Checked does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_switchMenu_data() {
            return [
                {label: "testLabel1", enabled: true, checked: false, icon: "file:///testIcon1" },
                {label: "testLabel2", enabled: false, checked: true, icon: "file:///testIcon2", subtitleAction: "action::subtitle1" },
                {label: "testLabel2", enabled: false, checked: true, icon: "file:///testIcon3", subtitleAction: "action::subtitle2" },
            ];
        }

        function test_create_switchMenu(data) {
            menuData.type = "com.canonical.indicator.switch";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.icon = data.icon;
            menuData.isToggled = data.checked;
            menuData.ext = {
                'xAyatanaSubtitleAction': data.subtitleAction,
            };

            loadData(menuData);
            compare(loader.item.objectName, "switchMenu", "Should have created a switch menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.iconSource, data.icon, "Icon does not match data");
            compare(loader.item.checked, data.checked, "Checked does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
            compare(loader.item.subtitleAction.name, data.subtitleAction === undefined ? "" : data.subtitleAction, "Subtitle action does not match data");
        }

        function test_create_alarmMenu_data() {
            return [
                {label: "testLabel1", enabled: true, icon: "file:///testIcon1", color: Qt.rgba(0, 0, 0, 0),
                            time: new Date(2014, 04, 14)},
                {label: "testLabel2", enabled: false, icon: "file:///testIcon2", color: Qt.rgba(1, 0, 0, 0),
                            time: new Date(2015, 11, 31) },
            ];
        }

        function test_create_alarmMenu(data) {
            menuData.type = "com.canonical.indicator.alarm";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.icon = data.icon;
            menuData.ext = {
                'xAyatanaTime': data.time.getTime() / 1000 // expected in seconds
            };

            loadData(menuData);
            compare(loader.item.objectName, "alarmMenu", "Should have created a alarm menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.iconSource, data.icon, "Icon does not match data");
            compare(loader.item.time, i18n.relativeDateTime(data.time), "Time does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_appointmentMenu_data() {
            return [
                {label: "testLabel1", enabled: true, icon: "file:///testIcon1", color: Qt.rgba(1, 2, 3, 0),
                            time: new Date(2014, 04, 14), timeFormat: "%a %d %b %l:%M %p"},
                {label: "testLabel2", enabled: false, icon: "file:///testIcon2", color: Qt.rgba(1, 0, 0, 0),
                            time: new Date(2015, 11, 31), timeFormat: "%A" },
            ];
        }

        function test_create_appointmentMenu(data) {
            menuData.type = "com.canonical.indicator.appointment";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.icon = data.icon;
            menuData.ext = {
                'xAyatanaColor': data.color,
                'xAyatanaTime': data.time.getTime() / 1000 // expected in seconds
            };

            loadData(menuData);
            compare(loader.item.objectName, "appointmentMenu", "Should have created a appointment menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.iconSource, data.icon, "Icon does not match data");
            compare(loader.item.time, i18n.relativeDateTime(data.time), "Time does not match data");
            compare(loader.item.eventColor, data.color, "Colour does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_transferMenu_data() {
            return [
                {tag: "queued", label: "testLabel1", enabled: true, active: true, icon: "file:///testIcon1", progress: 0, stateText: "In queue…" },
                {tag: "running", label: "testLabel2", enabled: true, active: true, icon: "file:///testIcon2", progress: 0.1, stateText: "1 minute, 40 seconds remaining" },
                {tag: "paused", label: "testLabel3", enabled: true, active: true, icon: "file:///testIcon3", progress: 0.5, stateText: "Paused, tap to resume" },
                {tag: "cancelled", label: "testLabel4", enabled: true, active: true, icon: "file:///testIcon4", progress: 0.4, stateText: "Canceled" },
                {tag: "finished", label: "testLabel5", enabled: false, active: false, icon: "file:///testIcon5", progress: 1.0, stateText: "Finished" },
                {tag: "error", label: "testLabel6", enabled: false, active: true, icon: "file:///testIcon6", progress: 0, stateText: "Failed, tap to retry" },
            ];
        }

        function test_create_transferMenu(data) {
            ActionData.data = {
                "transfer-state.queued": {
                    'valid': true,
                    'state': {
                        'state': Menus.TransferState.Queued,
                        'percent': data.progress
                    }
                },
                "transfer-state.running": {
                    'valid': true,
                    'state': {
                        'state': Menus.TransferState.Running,
                        'seconds-left': 100,
                        'percent': data.progress
                    }
                },
                "transfer-state.paused": {
                    'valid': true,
                    'state': {
                        'state': Menus.TransferState.Paused,
                        'seconds-left': 100,
                        'percent': data.progress
                    }
                },
                "transfer-state.cancelled": {
                    'valid': true,
                    'state': {
                        'state': Menus.TransferState.Canceled,
                        'percent': data.progress
                    }
                },
                "transfer-state.finished": {
                    'valid': true,
                    'state': {
                        'state': Menus.TransferState.Finished,
                        'seconds-left': 0,
                        'percent': data.progress
                    }
                },
                "transfer-state.error": {
                    'valid': true,
                    'state': {
                        'state': Menus.TransferState.Error,
                        'seconds-left': 100,
                        'percent': data.progress
                    }
                }
            };

            menuData.type = "com.canonical.indicator.transfer";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.isToggled = data.active;
            menuData.icon = data.icon;
            menuData.ext = {
                'xAyatanaUid': data.tag
            };

            loadData(menuData);
            compare(loader.item.objectName, "transferMenu", "Should have created a transfer menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.iconSource, data.icon, "Icon does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
            compare(loader.item.progress, data.progress, "Progress does not match data");
            compare(loader.item.active, data.active, "Active does not match data");
            compare(loader.item.stateText, data.stateText, "State text does not match data");
        }

        function test_create_buttonSectionMenu_data() {
            return [
                {label: "testLabel1", enabled: true, buttonText: "buttonLabel1", icon: "file:///testIcon1" },
                {label: "testLabel2", enabled: false, buttonText: "buttonLabel1", icon: "file:///testIcon2" },
            ];
        }

        function test_create_buttonSectionMenu(data) {
            menuData.type = "com.canonical.indicator.button-section";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.icon = data.icon;
            menuData.ext = {
                'xAyatanaExtraLabel': data.buttonText
            };

            loadData(menuData);
            compare(loader.item.objectName, "buttonSectionMenu", "Should have created a transfer menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.iconSource, data.icon, "Icon does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");

            compare(loader.item.buttonText, data.buttonText, "Button text does not match data");
        }

        function test_create_wifiSection_data() {
            return [
                {label: "testLabel1", busy: false },
                {label: "testLabel2", busy: true },
            ];
        }

        function test_create_wifiSection(data) {
            menuData.type = "lomiri.widgets.systemsettings.tablet.wifisection";
            menuData.label = data.label;
            menuData.ext = { 'xCanonicalBusyAction': data.busy }

            loadData(menuData);
            compare(loader.item.objectName, "wifiSection", "Should have created a wifi section menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.busy, data.busy, "Busy does not match data");
        }

        function test_create_accessPoint_data() {
            return [
                {label: "testLabel1", enabled: true, active: false, secure: true, adHoc: false },
                {label: "testLabel2", enabled: false, active: true, secure: false, adHoc: true },
            ];
        }

        function test_create_accessPoint(data) {
            menuData.type = "lomiri.widgets.systemsettings.tablet.accesspoint";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.isToggled = data.active;
            menuData.ext = {
                'xAyatanaWifiApStrengthAction': "action::strength",
                'xAyatanaWifiApIsSecure': data.secure,
                'xAyatanaWifiApIsAdhoc': data.adHoc,
            };

            loadData(menuData);
            compare(loader.item.objectName, "accessPoint", "Should have created a access point menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.strengthAction.name, "action::strength", "Strength action incorrect");
            compare(loader.item.secure, data.secure, "Secure does not match data");
            compare(loader.item.adHoc, data.adHoc, "AdHoc does not match data");
            compare(loader.item.active, data.active, "Checked does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_modemInfoItem_data() {
            // ModemInfoItem gets all it's data through the actions.
            return [
                        {
                            enabled: true,
                            statusLabelAction: "action::statusLabel",
                            statusIconAction: "action::statusIcon",
                            connectivityIconAction: "action::connectivityIcon",
                            simIdentifierLabelAction: "action::simIdentifierLabel",
                            roamingAction: "action::roaming",
                            unlockAction: "action::unlock"
                        }
                    ];
        }

        function test_create_modemInfoItem(data) {
            menuData.type = "com.lomiri.indicator.network.modeminfoitem";
            menuData.sensitive = data.enabled;
            menuData.ext = {
                'xLomiriModemStatusLabelAction': data.statusLabelAction,
                'xLomiriModemStatusIconAction': data.statusIconAction,
                'xLomiriModemConnectivityIconAction': data.connectivityIconAction,
                'xLomiriModemSimIdentifierLabelAction': data.simIdentifierLabelAction,
                'xLomiriModemRoamingAction': data.roamingAction,
                'xLomiriModemLockedAction': data.unlockAction,
            };

            loadData(menuData);
            compare(loader.item.objectName, "modemInfoItem", "Should have created a modem info item.");

            compare(loader.item.statusLabelAction.name, data.statusLabelAction, "StatusLabel action incorrect");
            compare(loader.item.statusIconAction.name, data.statusIconAction, "StatusIcon action incorrect");
            compare(loader.item.connectivityIconAction.name, data.connectivityIconAction, "ConnectivityIcon action incorrect");
            compare(loader.item.simIdentifierLabelAction.name, data.simIdentifierLabelAction, " action incorrect");
            compare(loader.item.roamingAction.name,  data.roamingAction, "Roaming action incorrect");
            compare(loader.item.unlockAction.name, data.unlockAction, "Unlock action incorrect");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_messageItem() {
            menuData.type = "com.canonical.indicator.messages.messageitem";

            loadData(menuData);
            compare(loader.item.objectName, "messageItem", "Should have created a message menu");
        }

        function test_create_groupedMessage_data() {
            return [
                {label: "testLabel1", enabled: true, count: "5", icon: "file:///testIcon" },
                {label: "testLabel2", enabled: false, count: "10", icon: "file:///testIcon2" },
            ];
        }

        function test_create_groupedMessage(data) {
            menuData.type = "com.canonical.indicator.messages.sourceitem";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.icon = data.icon;
            menuData.ext = { 'icon': data.icon };
            menuData.actionState = [data.count];
            menuData.isToggled = true;

            loadData(menuData);
            compare(loader.item.objectName, "groupedMessage", "Should have created a group message menu");

            compare(loader.item.text, data.label, "Label does not match data");
            compare(loader.item.count, data.count, "Count does not match data");
            compare(loader.item.iconSource, data.icon, "Icon does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_mediaPayerMenu_data() {
            return [{
                    label: "player1",
                    icon: "file:://icon1",
                    albumArt: "file:://art1",
                    song: "song1",
                    artist: "artist1",
                    album: "album1",
                    running: true,
                    state: "Playing",
                    enabled: true,
                    showTrack: true
                },{
                    label: "player2",
                    icon: "file:://icon2",
                    albumArt: "file:://art2",
                    song: "song2",
                    artist: "artist2",
                    album: "album2",
                    running: false,
                    state: "Paused",
                    enabled: false,
                    showTrack: false
                },{
                    label: "player3",
                    icon: "file:://icon3",
                    albumArt: "file:://art3",
                    song: "song3",
                    artist: "artist3",
                    album: "album3",
                    running: true,
                    state: "Stopped",
                    enabled: true,
                    showTrack: false
                }
            ];
        }

        function test_create_mediaPayerMenu(data) {
            menuData.type = "com.canonical.lomiri.media-player";
            menuData.label = data.label;
            menuData.sensitive = data.enabled;
            menuData.icon = data.icon;
            menuData.actionState = {
                'art-url': data.albumArt,
                'title': data.song,
                'artist': data.artist,
                'album': data.album,
                'running': data.running,
                'state': data.state,
            };

            loadData(menuData);
            compare(loader.item.objectName, "mediaPayerMenu", "Should have created a media player menu");

            compare(loader.item.playerIcon, data.icon, "Album art does not match data");
            compare(loader.item.playerName, data.label, "Album art does not match data");
            compare(loader.item.albumArt, data.albumArt, "Album art does not match data");
            compare(loader.item.song, data.song, "Song does not match data");
            compare(loader.item.artist, data.artist, "Artist does not match data");
            compare(loader.item.album, data.album, "Album does not match data");
            compare(loader.item.showTrack, data.showTrack, "Show track does not match data");
            compare(loader.item.state, data.state, "State does not match data");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_create_playbackItemMenu_data() {
            return [{
                    playAction: "action::play",
                    nextAction: "action::next",
                    previousAction: "action::previous",
                    enabled: true
                },{
                    playAction: "action::play2",
                    nextAction: "action::next2",
                    previousAction: "action::previous2",
                    enabled: false
                }
            ];
        }

        function test_create_playbackItemMenu(data) {
            menuData.type = "com.canonical.lomiri.playback-item";
            menuData.sensitive = data.enabled;
            menuData.ext = {
                'xAyatanaPlayAction': data.playAction,
                'xAyatanaNextAction': data.nextAction,
                'xAyatanaPreviousAction': data.previousAction
            };

            loadData(menuData);
            compare(loader.item.objectName, "playbackItemMenu", "Should have created a playback menu");

            compare(loader.item.playing, false, "Playing does not match data");
            compare(loader.item.playAction.name, data.playAction, "Play action incorrect");
            compare(loader.item.nextAction.name, data.nextAction, "Next action incorrect");
            compare(loader.item.previousAction.name, data.previousAction, "Previous action incorrect");
            compare(loader.item.canPlay, false, "CanPlay should be false");
            compare(loader.item.canGoNext, false, "CanGoNext should be false");
            compare(loader.item.canGoPrevious, false, "CanGoPrevious should be false");
            compare(loader.item.enabled, data.enabled, "Enabled does not match data");
        }

        function test_lp1336715_broken_switch_bindings() {
            menuData.type = "com.canonical.indicator.switch";
            menuData.sensitive = true;
            menuData.isToggled = false;

            loadData(menuData);

            compare(loader.item.checked, false, "Loader did not load check state");
            mouseClick(loader.item,
                       loader.item.width / 2, loader.item.height / 2);
            compare(loader.item.checked, true, "Clicking switch menu should toggle check");

            menuData.isToggled = true; // toggled will not update in mock
            menuData.isToggled = false;

            compare(loader.item.checked, false, "Server updates no longer working");
        }

        // test that the server value is re-aserted if it is not confirmed.
        function test_lp1336715_switch_server_value_reassertion() {
            menuData.type = "com.canonical.indicator.switch";
            menuData.sensitive = true;
            menuData.isToggled = false;

            loadData(menuData);

            var sync = findInvisibleChild(loader.item, "sync");
            verify(sync);
            sync.syncTimeout = 500;

            compare(loader.item.checked, false, "Loader did not load check state");
            mouseClick(loader.item,
                       loader.item.width / 2, loader.item.height / 2);
            compare(loader.item.checked, true, "Clicking switch menu should toggle check");
            tryCompare(loader.item, "checked", false);
        }

        function test_create_calendar_data() {
            return [
                { calendarDay: new Date(), showWeekNumbers: false, eventDays: new Array() },
                { calendarDay: new Date(2016, 8), showWeekNumbers: true, eventDays: [31] },
                { calendarDay: new Date(), showWeekNumbers: false, eventDays: [1, 5, 10] },
            ];
        }

        function test_create_calendar(data) {
            menuData.type = "com.canonical.indicator.calendar";
            menuData.actionState = {
                "calendar-day": data.calendarDay.getTime() / 1000 | 0,
                "show-week-numbers": data.showWeekNumbers,
                "appointment-days": data.eventDays
            }

            loadData(menuData);
            compare(loader.item.objectName, "calendarMenu", "Should have created a calendar menu")

            compare(loader.item.currentDate.getFullYear(), data.calendarDay.getFullYear(), "Calendar year does not match data")
            compare(loader.item.currentDate.getMonth(), data.calendarDay.getMonth(), "Calendar month does not match data")
            compare(loader.item.currentDate.getDate(), data.calendarDay.getDate(), "Calendar day does not match data")
            compare(loader.item.showWeekNumbers, data.showWeekNumbers, "Week numbers visibility does not match data")
            compare(loader.item.eventDays, data.eventDays, "Event days do not match data")
        }
    }
}
